/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.util;

import com.google.errorprone.annotations.RestrictedApi;
import edu.umd.cs.findbugs.annotations.Nullable;
import java.io.EOFException;
import java.io.IOException;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.TreeMap;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.commons.lang3.NotImplementedException;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileAlreadyExistsException;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.TableDescriptors;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.ColumnFamilyDescriptor;
import org.apache.hadoop.hbase.client.ColumnFamilyDescriptorBuilder;
import org.apache.hadoop.hbase.client.CoprocessorDescriptorBuilder;
import org.apache.hadoop.hbase.client.TableDescriptor;
import org.apache.hadoop.hbase.client.TableDescriptorBuilder;
import org.apache.hadoop.hbase.coprocessor.MultiRowMutationEndpoint;
import org.apache.hadoop.hbase.exceptions.DeserializationException;
import org.apache.hadoop.hbase.io.encoding.DataBlockEncoding;
import org.apache.hadoop.hbase.regionserver.BloomType;
import org.apache.hadoop.hbase.regionserver.storefiletracker.StoreFileTrackerFactory;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.CommonFSUtils;
import org.apache.hadoop.hbase.util.FSUtils;
import org.apache.hadoop.hbase.util.Pair;
import org.apache.hbase.thirdparty.com.google.common.primitives.Ints;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class FSTableDescriptors
implements TableDescriptors {
    private static final Logger LOG = LoggerFactory.getLogger(FSTableDescriptors.class);
    private final FileSystem fs;
    private final Path rootdir;
    private final boolean fsreadonly;
    private final boolean usecache;
    private volatile boolean fsvisited;
    long cachehits = 0L;
    long invocations = 0L;
    public static final String TABLEINFO_FILE_PREFIX = ".tableinfo";
    public static final String TABLEINFO_DIR = ".tabledesc";
    private final Map<TableName, TableDescriptor> cache = new ConcurrentHashMap<TableName, TableDescriptor>();
    static final Comparator<FileStatus> TABLEINFO_FILESTATUS_COMPARATOR = new Comparator<FileStatus>(){

        @Override
        public int compare(FileStatus left, FileStatus right) {
            return right.getPath().getName().compareTo(left.getPath().getName());
        }
    };
    private static final PathFilter TABLEINFO_PATHFILTER = new PathFilter(){

        public boolean accept(Path p) {
            return p.getName().startsWith(FSTableDescriptors.TABLEINFO_FILE_PREFIX);
        }
    };
    static final int WIDTH_OF_SEQUENCE_ID = 10;

    public FSTableDescriptors(Configuration conf) throws IOException {
        this(CommonFSUtils.getCurrentFileSystem((Configuration)conf), CommonFSUtils.getRootDir((Configuration)conf));
    }

    public FSTableDescriptors(FileSystem fs, Path rootdir) {
        this(fs, rootdir, false, true);
    }

    public FSTableDescriptors(FileSystem fs, Path rootdir, boolean fsreadonly, boolean usecache) {
        this.fs = fs;
        this.rootdir = rootdir;
        this.fsreadonly = fsreadonly;
        this.usecache = usecache;
    }

    public static void tryUpdateMetaTableDescriptor(Configuration conf) throws IOException {
        FSTableDescriptors.tryUpdateAndGetMetaTableDescriptor(conf, CommonFSUtils.getCurrentFileSystem((Configuration)conf), CommonFSUtils.getRootDir((Configuration)conf));
    }

    public static TableDescriptor tryUpdateAndGetMetaTableDescriptor(Configuration conf, FileSystem fs, Path rootdir) throws IOException {
        Optional<Pair<FileStatus, TableDescriptor>> opt = FSTableDescriptors.getTableDescriptorFromFs(fs, CommonFSUtils.getTableDir((Path)rootdir, (TableName)TableName.META_TABLE_NAME), false);
        if (opt.isPresent()) {
            return (TableDescriptor)opt.get().getSecond();
        }
        TableDescriptorBuilder builder = FSTableDescriptors.createMetaTableDescriptorBuilder(conf);
        TableDescriptor td = StoreFileTrackerFactory.updateWithTrackerConfigs(conf, builder.build());
        LOG.info("Creating new hbase:meta table descriptor {}", (Object)td);
        TableName tableName = td.getTableName();
        Path tableDir = CommonFSUtils.getTableDir((Path)rootdir, (TableName)tableName);
        Path p = FSTableDescriptors.writeTableDescriptor(fs, td, tableDir, null);
        if (p == null) {
            throw new IOException("Failed update hbase:meta table descriptor");
        }
        LOG.info("Updated hbase:meta table descriptor to {}", (Object)p);
        return td;
    }

    public static ColumnFamilyDescriptor getTableFamilyDescForMeta(Configuration conf) {
        return ColumnFamilyDescriptorBuilder.newBuilder((byte[])HConstants.TABLE_FAMILY).setMaxVersions(conf.getInt("hbase.meta.versions", 3)).setInMemory(true).setBlocksize(8192).setScope(0).setDataBlockEncoding(DataBlockEncoding.ROW_INDEX_V1).setBloomFilterType(BloomType.ROWCOL).build();
    }

    public static ColumnFamilyDescriptor getReplBarrierFamilyDescForMeta() {
        return ColumnFamilyDescriptorBuilder.newBuilder((byte[])HConstants.REPLICATION_BARRIER_FAMILY).setMaxVersions(Integer.MAX_VALUE).setInMemory(true).setScope(0).setDataBlockEncoding(DataBlockEncoding.ROW_INDEX_V1).setBloomFilterType(BloomType.ROWCOL).build();
    }

    public static TableDescriptorBuilder createMetaTableDescriptorBuilder(Configuration conf) throws IOException {
        return TableDescriptorBuilder.newBuilder((TableName)TableName.META_TABLE_NAME).setColumnFamily(ColumnFamilyDescriptorBuilder.newBuilder((byte[])HConstants.CATALOG_FAMILY).setMaxVersions(conf.getInt("hbase.meta.versions", 3)).setInMemory(true).setBlocksize(conf.getInt("hbase.meta.blocksize", 8192)).setScope(0).setDataBlockEncoding(DataBlockEncoding.ROW_INDEX_V1).setBloomFilterType(BloomType.ROWCOL).build()).setColumnFamily(FSTableDescriptors.getTableFamilyDescForMeta(conf)).setColumnFamily(FSTableDescriptors.getReplBarrierFamilyDescForMeta()).setCoprocessor(CoprocessorDescriptorBuilder.newBuilder((String)MultiRowMutationEndpoint.class.getName()).setPriority(0x1FFFFFFF).build());
    }

    protected boolean isUsecache() {
        return this.usecache;
    }

    @Override
    @Nullable
    public TableDescriptor get(TableName tableName) {
        ++this.invocations;
        if (this.usecache) {
            TableDescriptor cachedtdm = this.cache.get(tableName);
            if (cachedtdm != null) {
                ++this.cachehits;
                return cachedtdm;
            }
            if (this.fsvisited) {
                return null;
            }
        }
        TableDescriptor tdmt = null;
        try {
            tdmt = FSTableDescriptors.getTableDescriptorFromFs(this.fs, this.getTableDir(tableName), this.fsreadonly).map(Pair::getSecond).orElse(null);
        }
        catch (IOException ioe) {
            LOG.debug("Exception during readTableDecriptor. Current table name = " + tableName, (Throwable)ioe);
        }
        if (this.usecache && tdmt != null) {
            this.cache.put(tableName, tdmt);
        }
        return tdmt;
    }

    @Override
    public Map<String, TableDescriptor> getAll() throws IOException {
        TreeMap<String, TableDescriptor> tds = new TreeMap<String, TableDescriptor>();
        if (this.fsvisited) {
            for (Map.Entry<TableName, TableDescriptor> entry : this.cache.entrySet()) {
                tds.put(entry.getKey().getNameWithNamespaceInclAsString(), entry.getValue());
            }
        } else {
            LOG.trace("Fetching table descriptors from the filesystem.");
            boolean allvisited = this.usecache;
            for (Path d : FSUtils.getTableDirs(this.fs, this.rootdir)) {
                TableDescriptor htd = this.get(CommonFSUtils.getTableName((Path)d));
                if (htd == null) {
                    allvisited = false;
                    continue;
                }
                tds.put(htd.getTableName().getNameWithNamespaceInclAsString(), htd);
            }
            this.fsvisited = allvisited;
        }
        return tds;
    }

    @Override
    public Map<String, TableDescriptor> getByNamespace(String name) throws IOException {
        TreeMap<String, TableDescriptor> htds = new TreeMap<String, TableDescriptor>();
        List<Path> tableDirs = FSUtils.getLocalTableDirs(this.fs, CommonFSUtils.getNamespaceDir((Path)this.rootdir, (String)name));
        for (Path d : tableDirs) {
            TableDescriptor htd = this.get(CommonFSUtils.getTableName((Path)d));
            if (htd == null) continue;
            htds.put(CommonFSUtils.getTableName((Path)d).getNameAsString(), htd);
        }
        return htds;
    }

    @Override
    public void update(TableDescriptor td, boolean cacheOnly) throws IOException {
        if (this.fsreadonly) {
            throw new UnsupportedOperationException("Cannot add a table descriptor - in read only mode");
        }
        if (!cacheOnly) {
            this.updateTableDescriptor(td);
        }
        if (this.usecache) {
            this.cache.put(td.getTableName(), td);
        }
    }

    @RestrictedApi(explanation="Should only be called in tests or self", link="", allowedOnPath=".*/src/test/.*|.*/FSTableDescriptors\\.java")
    Path updateTableDescriptor(TableDescriptor td) throws IOException {
        TableName tableName = td.getTableName();
        Path tableDir = this.getTableDir(tableName);
        Path p = FSTableDescriptors.writeTableDescriptor(this.fs, td, tableDir, FSTableDescriptors.getTableDescriptorFromFs(this.fs, tableDir, this.fsreadonly).map(Pair::getFirst).orElse(null));
        if (p == null) {
            throw new IOException("Failed update");
        }
        LOG.info("Updated tableinfo=" + p);
        return p;
    }

    @Override
    public TableDescriptor remove(TableName tablename) throws IOException {
        if (this.fsreadonly) {
            throw new NotImplementedException("Cannot remove a table descriptor - in read only mode");
        }
        Path tabledir = this.getTableDir(tablename);
        if (this.fs.exists(tabledir) && !this.fs.delete(tabledir, true)) {
            throw new IOException("Failed delete of " + tabledir.toString());
        }
        TableDescriptor descriptor = this.cache.remove(tablename);
        return descriptor;
    }

    public static boolean isTableDir(FileSystem fs, Path tableDir) throws IOException {
        return FSTableDescriptors.getTableDescriptorFromFs(fs, tableDir, true).isPresent();
    }

    private Path getTableDir(TableName tableName) {
        return CommonFSUtils.getTableDir((Path)this.rootdir, (TableName)tableName);
    }

    private static String formatTableInfoSequenceId(int number) {
        byte[] b = new byte[10];
        int d = Math.abs(number);
        for (int i = b.length - 1; i >= 0; --i) {
            b[i] = (byte)(d % 10 + 48);
            d /= 10;
        }
        return Bytes.toString((byte[])b);
    }

    @RestrictedApi(explanation="Should only be called in tests or self", link="", allowedOnPath=".*/src/test/.*|.*/FSTableDescriptors\\.java")
    static SequenceIdAndFileLength getTableInfoSequenceIdAndFileLength(Path p) {
        String name = p.getName();
        if (!name.startsWith(TABLEINFO_FILE_PREFIX)) {
            throw new IllegalArgumentException("Invalid table descriptor file name: " + name);
        }
        int firstDot = name.indexOf(46, TABLEINFO_FILE_PREFIX.length());
        if (firstDot < 0) {
            return new SequenceIdAndFileLength(0, 0);
        }
        int secondDot = name.indexOf(46, firstDot + 1);
        if (secondDot < 0) {
            int sequenceId = Integer.parseInt(name.substring(firstDot + 1));
            return new SequenceIdAndFileLength(sequenceId, 0);
        }
        int sequenceId = Integer.parseInt(name.substring(firstDot + 1, secondDot));
        int fileLength = Integer.parseInt(name.substring(secondDot + 1));
        return new SequenceIdAndFileLength(sequenceId, fileLength);
    }

    @RestrictedApi(explanation="Should only be called in tests or self", link="", allowedOnPath=".*/src/test/.*|.*/FSTableDescriptors\\.java")
    static String getTableInfoFileName(int sequenceId, byte[] content) {
        return ".tableinfo." + FSTableDescriptors.formatTableInfoSequenceId(sequenceId) + "." + content.length;
    }

    public static TableDescriptor getTableDescriptorFromFs(FileSystem fs, Path hbaseRootDir, TableName tableName) throws IOException {
        Path tableDir = CommonFSUtils.getTableDir((Path)hbaseRootDir, (TableName)tableName);
        return FSTableDescriptors.getTableDescriptorFromFs(fs, tableDir);
    }

    public static TableDescriptor getTableDescriptorFromFs(FileSystem fs, Path tableDir) throws IOException {
        return FSTableDescriptors.getTableDescriptorFromFs(fs, tableDir, true).map(Pair::getSecond).orElse(null);
    }

    private static void deleteMalformedFile(FileSystem fs, Path file) throws IOException {
        LOG.info("Delete malformed table descriptor file {}", (Object)file);
        if (!fs.delete(file, false)) {
            LOG.warn("Failed to delete malformed table descriptor file {}", (Object)file);
        }
    }

    private static Optional<Pair<FileStatus, TableDescriptor>> getTableDescriptorFromFs(FileSystem fs, Path tableDir, boolean readonly) throws IOException {
        Path file;
        int i;
        Path tableInfoDir = new Path(tableDir, TABLEINFO_DIR);
        FileStatus[] descFiles = CommonFSUtils.listStatus((FileSystem)fs, (Path)tableInfoDir, (PathFilter)TABLEINFO_PATHFILTER);
        if (descFiles == null || descFiles.length < 1) {
            return Optional.empty();
        }
        Arrays.sort(descFiles, TABLEINFO_FILESTATUS_COMPARATOR);
        TableDescriptor td = null;
        FileStatus descFile = null;
        for (i = 0; i < descFiles.length; ++i) {
            descFile = descFiles[i];
            file = descFile.getPath();
            int fileLength = FSTableDescriptors.getTableInfoSequenceIdAndFileLength((Path)file).fileLength;
            byte[] content = new byte[fileLength > 0 ? fileLength : Ints.checkedCast((long)descFile.getLen())];
            try (FSDataInputStream in = fs.open(file);){
                in.readFully(content);
            }
            catch (EOFException e) {
                LOG.info("Failed to load file {} due to EOF, it should be half written: {}", (Object)file, (Object)e.toString());
                if (readonly) continue;
                FSTableDescriptors.deleteMalformedFile(fs, file);
                continue;
            }
            try {
                td = TableDescriptorBuilder.parseFrom((byte[])content);
                break;
            }
            catch (DeserializationException e) {
                LOG.info("Failed to parse file {} due to malformed protobuf message: {}", (Object)file, (Object)e.toString());
                if (readonly) continue;
                FSTableDescriptors.deleteMalformedFile(fs, file);
            }
        }
        if (!readonly) {
            ++i;
            while (i < descFiles.length) {
                file = descFiles[i].getPath();
                LOG.info("Delete old table descriptor file {}", (Object)file);
                if (!fs.delete(file, false)) {
                    LOG.info("Failed to delete old table descriptor file {}", (Object)file);
                }
                ++i;
            }
        }
        return td != null ? Optional.of(Pair.newPair((Object)descFile, td)) : Optional.empty();
    }

    @RestrictedApi(explanation="Should only be called in tests", link="", allowedOnPath=".*/src/test/.*")
    public static void deleteTableDescriptors(FileSystem fs, Path tableDir) throws IOException {
        Path tableInfoDir = new Path(tableDir, TABLEINFO_DIR);
        FSTableDescriptors.deleteTableDescriptorFiles(fs, tableInfoDir, Integer.MAX_VALUE);
    }

    private static void deleteTableDescriptorFiles(FileSystem fs, Path dir, int maxSequenceId) throws IOException {
        FileStatus[] status;
        for (FileStatus file : status = CommonFSUtils.listStatus((FileSystem)fs, (Path)dir, (PathFilter)TABLEINFO_PATHFILTER)) {
            Path path = file.getPath();
            int sequenceId = FSTableDescriptors.getTableInfoSequenceIdAndFileLength((Path)path).sequenceId;
            if (sequenceId > maxSequenceId) continue;
            boolean success = CommonFSUtils.delete((FileSystem)fs, (Path)path, (boolean)false);
            if (success) {
                LOG.debug("Deleted {}", (Object)path);
                continue;
            }
            LOG.error("Failed to delete table descriptor at {}", (Object)path);
        }
    }

    private static Path writeTableDescriptor(FileSystem fs, TableDescriptor td, Path tableDir, FileStatus currentDescriptorFile) throws IOException {
        Path tableInfoDir = new Path(tableDir, TABLEINFO_DIR);
        int currentSequenceId = currentDescriptorFile == null ? 0 : FSTableDescriptors.getTableInfoSequenceIdAndFileLength((Path)currentDescriptorFile.getPath()).sequenceId;
        int maxAttempts = 10;
        int maxSequenceId = currentSequenceId + maxAttempts;
        byte[] bytes = TableDescriptorBuilder.toByteArray((TableDescriptor)td);
        for (int newSequenceId = currentSequenceId + 1; newSequenceId <= maxSequenceId; ++newSequenceId) {
            String fileName = FSTableDescriptors.getTableInfoFileName(newSequenceId, bytes);
            Path filePath = new Path(tableInfoDir, fileName);
            try (FSDataOutputStream out = fs.create(filePath, false);){
                out.write(bytes);
            }
            catch (FileAlreadyExistsException e) {
                LOG.debug("{} exists; retrying up to {} times", new Object[]{filePath, maxAttempts, e});
                continue;
            }
            catch (IOException e) {
                LOG.debug("Failed write {}; retrying up to {} times", new Object[]{filePath, maxAttempts, e});
                continue;
            }
            FSTableDescriptors.deleteTableDescriptorFiles(fs, tableInfoDir, newSequenceId - 1);
            return filePath;
        }
        return null;
    }

    public boolean createTableDescriptor(TableDescriptor htd) throws IOException {
        return this.createTableDescriptor(htd, false);
    }

    public boolean createTableDescriptor(TableDescriptor htd, boolean forceCreation) throws IOException {
        Path tableDir = this.getTableDir(htd.getTableName());
        return this.createTableDescriptorForTableDirectory(tableDir, htd, forceCreation);
    }

    public boolean createTableDescriptorForTableDirectory(Path tableDir, TableDescriptor htd, boolean forceCreation) throws IOException {
        if (this.fsreadonly) {
            throw new NotImplementedException("Cannot create a table descriptor - in read only mode");
        }
        return FSTableDescriptors.createTableDescriptorForTableDirectory(this.fs, tableDir, htd, forceCreation);
    }

    public static boolean createTableDescriptorForTableDirectory(FileSystem fs, Path tableDir, TableDescriptor htd, boolean forceCreation) throws IOException {
        Optional<Pair<FileStatus, TableDescriptor>> opt = FSTableDescriptors.getTableDescriptorFromFs(fs, tableDir, false);
        if (opt.isPresent()) {
            LOG.debug("Current path={}", opt.get().getFirst());
            if (!forceCreation && htd.equals(opt.get().getSecond())) {
                LOG.trace("TableInfo already exists.. Skipping creation");
                return false;
            }
        }
        return FSTableDescriptors.writeTableDescriptor(fs, htd, tableDir, opt.map(Pair::getFirst).orElse(null)) != null;
    }

    static final class SequenceIdAndFileLength {
        final int sequenceId;
        final int fileLength;

        SequenceIdAndFileLength(int sequenceId, int fileLength) {
            this.sequenceId = sequenceId;
            this.fileLength = fileLength;
        }
    }
}

