/*
 * Decompiled with CFR 0.152.
 */
package org.apache.polaris.core.storage.cache;

import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.Expiry;
import com.github.benmanes.caffeine.cache.LoadingCache;
import com.google.common.annotations.VisibleForTesting;
import jakarta.annotation.Nonnull;
import jakarta.annotation.Nullable;
import java.time.Duration;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import org.apache.iceberg.exceptions.UnprocessableEntityException;
import org.apache.polaris.core.PolarisDiagnostics;
import org.apache.polaris.core.config.FeatureConfiguration;
import org.apache.polaris.core.config.PolarisConfiguration;
import org.apache.polaris.core.config.RealmConfig;
import org.apache.polaris.core.context.RealmContext;
import org.apache.polaris.core.entity.PolarisEntity;
import org.apache.polaris.core.entity.PolarisEntityType;
import org.apache.polaris.core.persistence.dao.entity.ScopedCredentialsResult;
import org.apache.polaris.core.storage.StorageAccessConfig;
import org.apache.polaris.core.storage.StorageCredentialsVendor;
import org.apache.polaris.core.storage.cache.StorageCredentialCacheConfig;
import org.apache.polaris.core.storage.cache.StorageCredentialCacheEntry;
import org.apache.polaris.core.storage.cache.StorageCredentialCacheKey;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StorageCredentialCache {
    private static final Logger LOGGER = LoggerFactory.getLogger(StorageCredentialCache.class);
    private final PolarisDiagnostics diagnostics;
    private final LoadingCache<StorageCredentialCacheKey, StorageCredentialCacheEntry> cache;

    public StorageCredentialCache(PolarisDiagnostics diagnostics, StorageCredentialCacheConfig cacheConfig) {
        this.diagnostics = diagnostics;
        this.cache = Caffeine.newBuilder().maximumSize(cacheConfig.maxEntries()).expireAfter(Expiry.creating((key, entry) -> {
            long expireAfterMillis = Math.max(0L, Math.min((entry.getExpirationTime() - System.currentTimeMillis()) / 2L, entry.getMaxCacheDurationMs()));
            return Duration.ofMillis(expireAfterMillis);
        })).build(key -> null);
    }

    private long maxCacheDurationMs(RealmConfig realmConfig) {
        Integer cacheDurationSeconds = (Integer)realmConfig.getConfig((PolarisConfiguration)FeatureConfiguration.STORAGE_CREDENTIAL_CACHE_DURATION_SECONDS);
        Integer credentialDurationSeconds = (Integer)realmConfig.getConfig((PolarisConfiguration)FeatureConfiguration.STORAGE_CREDENTIAL_DURATION_SECONDS);
        if (cacheDurationSeconds >= credentialDurationSeconds) {
            throw new IllegalArgumentException(String.format("%s should be less than %s", FeatureConfiguration.STORAGE_CREDENTIAL_CACHE_DURATION_SECONDS.key(), FeatureConfiguration.STORAGE_CREDENTIAL_DURATION_SECONDS.key()));
        }
        return (long)cacheDurationSeconds.intValue() * 1000L;
    }

    public StorageAccessConfig getOrGenerateSubScopeCreds(@Nonnull StorageCredentialsVendor storageCredentialsVendor, @Nonnull PolarisEntity polarisEntity, boolean allowListOperation, @Nonnull Set<String> allowedReadLocations, @Nonnull Set<String> allowedWriteLocations, Optional<String> refreshCredentialsEndpoint) {
        RealmContext realmContext = storageCredentialsVendor.getRealmContext();
        RealmConfig realmConfig = storageCredentialsVendor.getRealmConfig();
        if (!this.isTypeSupported(polarisEntity.getType())) {
            this.diagnostics.fail("entity_type_not_suppported_to_scope_creds", "type={}", new Object[]{polarisEntity.getType()});
        }
        StorageCredentialCacheKey key = StorageCredentialCacheKey.of((String)realmContext.getRealmIdentifier(), (PolarisEntity)polarisEntity, (boolean)allowListOperation, allowedReadLocations, allowedWriteLocations, refreshCredentialsEndpoint);
        LOGGER.atDebug().addKeyValue("key", (Object)key).log("subscopedCredsCache");
        Function<StorageCredentialCacheKey, StorageCredentialCacheEntry> loader = k -> {
            LOGGER.atDebug().log("StorageCredentialCache::load");
            ScopedCredentialsResult scopedCredentialsResult = storageCredentialsVendor.getSubscopedCredsForEntity(polarisEntity, allowListOperation, allowedReadLocations, allowedWriteLocations, refreshCredentialsEndpoint);
            if (scopedCredentialsResult.isSuccess()) {
                long maxCacheDurationMs = this.maxCacheDurationMs(realmConfig);
                return new StorageCredentialCacheEntry(scopedCredentialsResult.getStorageAccessConfig(), maxCacheDurationMs);
            }
            LOGGER.atDebug().addKeyValue("errorMessage", (Object)scopedCredentialsResult.getExtraInformation()).log("Failed to get subscoped credentials");
            throw new UnprocessableEntityException("Failed to get subscoped credentials: %s", new Object[]{scopedCredentialsResult.getExtraInformation()});
        };
        return ((StorageCredentialCacheEntry)this.cache.get((Object)key, loader)).toAccessConfig();
    }

    @Nullable
    @VisibleForTesting
    Map<String, String> getIfPresent(StorageCredentialCacheKey key) {
        return this.getAccessConfig(key).map(StorageAccessConfig::credentials).orElse(null);
    }

    @VisibleForTesting
    Optional<StorageAccessConfig> getAccessConfig(StorageCredentialCacheKey key) {
        return Optional.ofNullable((StorageCredentialCacheEntry)this.cache.getIfPresent((Object)key)).map(StorageCredentialCacheEntry::toAccessConfig);
    }

    private boolean isTypeSupported(PolarisEntityType type) {
        return type == PolarisEntityType.CATALOG || type == PolarisEntityType.NAMESPACE || type == PolarisEntityType.TABLE_LIKE || type == PolarisEntityType.TASK;
    }

    @VisibleForTesting
    public long getEstimatedSize() {
        return this.cache.estimatedSize();
    }

    @VisibleForTesting
    public void invalidateAll() {
        this.cache.invalidateAll();
    }

    public StorageCredentialCache() {
    }
}

