/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.amoro.hive.utils;

import org.apache.amoro.properties.HiveTableProperties;
import org.apache.iceberg.util.PropertyUtil;

import java.util.Map;

/** PropertyUtil compatible with legacy hive properties */
public class CompatibleHivePropertyUtil {

  private CompatibleHivePropertyUtil() {}

  public static boolean propertyAsBoolean(
      Map<String, String> properties, String property, boolean defaultValue) {
    return PropertyUtil.propertyAsBoolean(
        properties, getCompatibleProperty(properties, property), defaultValue);
  }

  public static double propertyAsDouble(
      Map<String, String> properties, String property, double defaultValue) {
    return PropertyUtil.propertyAsDouble(
        properties, getCompatibleProperty(properties, property), defaultValue);
  }

  public static int propertyAsInt(
      Map<String, String> properties, String property, int defaultValue) {
    return PropertyUtil.propertyAsInt(
        properties, getCompatibleProperty(properties, property), defaultValue);
  }

  public static long propertyAsLong(
      Map<String, String> properties, String property, long defaultValue) {
    return PropertyUtil.propertyAsLong(
        properties, getCompatibleProperty(properties, property), defaultValue);
  }

  public static String propertyAsString(
      Map<String, String> properties, String property, String defaultValue) {
    return PropertyUtil.propertyAsString(
        properties, getCompatibleProperty(properties, property), defaultValue);
  }

  private static String getCompatibleProperty(Map<String, String> properties, String property) {
    String legacyProperty = getLegacyProperty(property);
    if (legacyProperty != null
        && properties.containsKey(legacyProperty)
        && !properties.containsKey(property)) {
      return legacyProperty;
    } else {
      return property;
    }
  }

  private static String getLegacyProperty(String property) {
    if (property == null) {
      return null;
    }
    switch (property) {
      case HiveTableProperties.MIXED_TABLE_FLAG:
        return HiveTableProperties.AMORO_TABLE_FLAG_LEGACY;
      default:
        return null;
    }
  }
}
