
// rsa_pss_vectors.js

// Data for testing RSA-PSS with a 2048-bit modulus and 65537 public exponent.

// The following function returns an array of test vectors
// for the subtleCrypto encrypt method.
//
// Each test vector has the following fields:
//     name - a unique name for this vector
//     publicKeyBuffer - an arrayBuffer with the key data
//     publicKeyFormat - "spki" "jwk"
//     publicKey - a CryptoKey object for the keyBuffer. INITIALLY null! You must fill this in first to use it!
//     privateKeyBuffer - an arrayBuffer with the key data
//     privateKeyFormat - "pkcs8" or "jwk"
//     privateKey - a CryptoKey object for the keyBuffer. INITIALLY null! You must fill this in first to use it!
//     algorithm - the value of the AlgorithmIdentifier parameter to provide to encrypt
//     plaintext - the text to encrypt
//     signature - the expected signature
function getTestVectors() {
    var pkcs8 = new Uint8Array([48, 130, 4, 191, 2, 1, 0, 48, 13, 6, 9, 42, 134, 72, 134, 247, 13, 1, 1, 1, 5, 0, 4, 130, 4, 169, 48, 130, 4, 165, 2, 1, 0, 2, 130, 1, 1, 0, 211, 87, 96, 146, 230, 41, 87, 54, 69, 68, 231, 228, 35, 59, 123, 219, 41, 61, 178, 8, 81, 34, 196, 121, 50, 133, 70, 249, 240, 247, 18, 246, 87, 196, 177, 120, 104, 201, 48, 144, 140, 197, 148, 247, 237, 0, 192, 20, 66, 193, 175, 4, 194, 246, 120, 164, 139, 162, 200, 15, 209, 113, 62, 48, 181, 172, 80, 120, 122, 195, 81, 101, 137, 241, 113, 150, 127, 99, 134, 173, 163, 73, 0, 166, 187, 4, 238, 206, 164, 43, 240, 67, 206, 217, 160, 249, 77, 12, 192, 158, 145, 155, 157, 113, 102, 192, 138, 182, 206, 32, 70, 64, 174, 164, 196, 146, 13, 182, 216, 110, 185, 22, 208, 220, 192, 244, 52, 26, 16, 56, 4, 41, 231, 225, 3, 33, 68, 234, 148, 157, 232, 246, 192, 204, 191, 149, 250, 142, 146, 141, 112, 216, 163, 140, 225, 104, 219, 69, 246, 241, 52, 102, 61, 111, 101, 111, 92, 234, 188, 114, 93, 168, 192, 42, 171, 234, 170, 19, 172, 54, 167, 92, 192, 186, 225, 53, 223, 49, 20, 182, 101, 137, 199, 237, 60, 182, 21, 89, 174, 90, 56, 79, 22, 43, 250, 128, 219, 228, 97, 127, 134, 195, 241, 208, 16, 201, 79, 226, 201, 191, 1, 154, 110, 99, 179, 239, 192, 40, 212, 60, 238, 97, 28, 133, 236, 38, 60, 144, 108, 70, 55, 114, 198, 145, 27, 25, 238, 192, 150, 202, 118, 236, 94, 49, 225, 227, 2, 3, 1, 0, 1, 2, 130, 1, 1, 0, 139, 55, 92, 203, 135, 200, 37, 197, 255, 61, 83, 208, 9, 145, 110, 150, 65, 5, 126, 24, 82, 114, 39, 160, 122, 178, 38, 190, 16, 136, 129, 58, 59, 56, 187, 123, 72, 243, 119, 5, 81, 101, 250, 42, 147, 57, 210, 77, 198, 103, 213, 197, 186, 52, 39, 230, 164, 129, 23, 110, 172, 21, 255, 212, 144, 104, 49, 30, 28, 40, 59, 159, 58, 142, 12, 184, 9, 180, 99, 12, 80, 170, 143, 62, 69, 166, 11, 53, 158, 25, 191, 140, 187, 94, 202, 214, 78, 118, 31, 16, 149, 116, 63, 243, 106, 175, 92, 240, 236, 185, 127, 237, 173, 221, 166, 11, 91, 243, 93, 129, 26, 117, 184, 34, 35, 12, 250, 160, 25, 47, 173, 64, 84, 126, 39, 84, 72, 170, 51, 22, 191, 142, 43, 76, 224, 133, 79, 199, 112, 139, 83, 123, 162, 45, 19, 33, 11, 9, 174, 195, 122, 39, 89, 239, 192, 130, 161, 83, 27, 35, 169, 23, 48, 3, 125, 222, 78, 242, 107, 95, 150, 239, 220, 195, 159, 211, 76, 52, 90, 213, 28, 187, 228, 79, 229, 139, 138, 59, 78, 201, 151, 134, 108, 8, 109, 255, 27, 136, 49, 239, 10, 31, 234, 38, 60, 247, 218, 205, 3, 192, 76, 188, 194, 178, 121, 229, 127, 165, 185, 83, 153, 107, 251, 29, 214, 136, 23, 175, 127, 180, 44, 222, 247, 165, 41, 74, 87, 250, 194, 184, 173, 115, 159, 27, 2, 153, 2, 129, 129, 0, 251, 248, 51, 194, 198, 49, 201, 112, 36, 12, 142, 116, 133, 240, 106, 62, 162, 168, 72, 34, 81, 26, 134, 39, 221, 70, 78, 248, 175, 175, 113, 72, 209, 164, 37, 182, 184, 101, 125, 221, 82, 70, 131, 43, 142, 83, 48, 32, 197, 187, 181, 104, 133, 90, 106, 236, 62, 66, 33, 215, 147, 241, 220, 91, 47, 37, 132, 226, 65, 94, 72, 233, 162, 189, 41, 43, 19, 64, 49, 249, 156, 142, 180, 47, 192, 188, 208, 68, 155, 242, 44, 230, 222, 201, 112, 20, 239, 229, 172, 147, 235, 232, 53, 135, 118, 86, 37, 44, 187, 177, 108, 65, 91, 103, 177, 132, 210, 40, 69, 104, 162, 119, 213, 147, 53, 88, 92, 253, 2, 129, 129, 0, 214, 184, 206, 39, 199, 41, 93, 93, 22, 252, 53, 112, 237, 100, 200, 218, 147, 3, 250, 210, 148, 136, 193, 166, 94, 154, 215, 17, 249, 3, 112, 24, 125, 187, 253, 129, 49, 109, 105, 100, 139, 200, 140, 197, 200, 53, 81, 175, 255, 69, 222, 186, 207, 182, 17, 5, 247, 9, 228, 195, 8, 9, 185, 0, 49, 235, 214, 134, 36, 68, 150, 198, 246, 158, 105, 46, 189, 200, 20, 246, 66, 57, 244, 173, 21, 117, 110, 203, 120, 197, 165, 176, 153, 49, 219, 24, 48, 119, 197, 70, 163, 140, 76, 116, 56, 137, 173, 61, 62, 208, 121, 181, 98, 46, 208, 18, 15, 160, 225, 249, 59, 89, 61, 183, 216, 82, 224, 95, 2, 129, 128, 56, 135, 75, 157, 131, 247, 129, 120, 206, 45, 158, 252, 23, 92, 131, 137, 127, 214, 127, 48, 107, 191, 166, 159, 100, 238, 52, 35, 104, 206, 212, 124, 128, 195, 241, 206, 23, 122, 117, 141, 100, 186, 251, 12, 151, 134, 164, 66, 133, 250, 1, 205, 236, 53, 7, 205, 238, 125, 201, 183, 226, 178, 29, 60, 187, 204, 16, 14, 238, 153, 103, 132, 59, 5, 115, 41, 253, 204, 166, 41, 152, 237, 15, 17, 179, 140, 232, 176, 171, 199, 222, 57, 1, 124, 113, 207, 208, 174, 87, 84, 108, 85, 145, 68, 205, 208, 175, 208, 100, 95, 126, 168, 255, 7, 185, 116, 209, 237, 68, 253, 31, 142, 0, 245, 96, 191, 109, 69, 2, 129, 129, 0, 133, 41, 239, 144, 115, 207, 143, 123, 95, 249, 226, 26, 186, 223, 58, 65, 115, 211, 144, 6, 112, 223, 175, 89, 66, 106, 188, 223, 4, 147, 193, 61, 47, 29, 27, 70, 184, 36, 166, 172, 24, 148, 179, 217, 37, 37, 12, 24, 30, 52, 114, 193, 96, 120, 5, 110, 177, 154, 141, 40, 247, 31, 48, 128, 146, 117, 52, 129, 212, 148, 68, 253, 247, 140, 158, 166, 194, 68, 7, 220, 1, 142, 119, 211, 175, 239, 56, 91, 47, 247, 67, 158, 150, 35, 121, 65, 51, 45, 212, 70, 206, 190, 255, 219, 68, 4, 254, 79, 113, 89, 81, 97, 208, 22, 64, 44, 51, 77, 15, 87, 198, 26, 190, 79, 249, 244, 203, 249, 2, 129, 129, 0, 135, 216, 119, 8, 212, 103, 99, 228, 204, 190, 178, 209, 233, 113, 46, 91, 240, 33, 109, 112, 222, 148, 32, 165, 178, 6, 155, 116, 89, 185, 159, 93, 159, 127, 47, 173, 124, 215, 154, 174, 230, 122, 127, 154, 52, 67, 126, 60, 121, 168, 74, 240, 205, 141, 233, 223, 242, 104, 235, 12, 71, 147, 245, 1, 249, 136, 213, 64, 246, 211, 71, 92, 32, 121, 184, 34, 122, 35, 217, 104, 222, 196, 227, 198, 101, 3, 24, 113, 147, 69, 150, 48, 71, 43, 253, 182, 186, 29, 231, 134, 199, 151, 250, 111, 78, 166, 90, 42, 132, 25, 38, 47, 41, 103, 136, 86, 203, 115, 201, 189, 75, 200, 155, 94, 4, 27, 34, 119]);
    var spki = new Uint8Array([48, 130, 1, 34, 48, 13, 6, 9, 42, 134, 72, 134, 247, 13, 1, 1, 1, 5, 0, 3, 130, 1, 15, 0, 48, 130, 1, 10, 2, 130, 1, 1, 0, 211, 87, 96, 146, 230, 41, 87, 54, 69, 68, 231, 228, 35, 59, 123, 219, 41, 61, 178, 8, 81, 34, 196, 121, 50, 133, 70, 249, 240, 247, 18, 246, 87, 196, 177, 120, 104, 201, 48, 144, 140, 197, 148, 247, 237, 0, 192, 20, 66, 193, 175, 4, 194, 246, 120, 164, 139, 162, 200, 15, 209, 113, 62, 48, 181, 172, 80, 120, 122, 195, 81, 101, 137, 241, 113, 150, 127, 99, 134, 173, 163, 73, 0, 166, 187, 4, 238, 206, 164, 43, 240, 67, 206, 217, 160, 249, 77, 12, 192, 158, 145, 155, 157, 113, 102, 192, 138, 182, 206, 32, 70, 64, 174, 164, 196, 146, 13, 182, 216, 110, 185, 22, 208, 220, 192, 244, 52, 26, 16, 56, 4, 41, 231, 225, 3, 33, 68, 234, 148, 157, 232, 246, 192, 204, 191, 149, 250, 142, 146, 141, 112, 216, 163, 140, 225, 104, 219, 69, 246, 241, 52, 102, 61, 111, 101, 111, 92, 234, 188, 114, 93, 168, 192, 42, 171, 234, 170, 19, 172, 54, 167, 92, 192, 186, 225, 53, 223, 49, 20, 182, 101, 137, 199, 237, 60, 182, 21, 89, 174, 90, 56, 79, 22, 43, 250, 128, 219, 228, 97, 127, 134, 195, 241, 208, 16, 201, 79, 226, 201, 191, 1, 154, 110, 99, 179, 239, 192, 40, 212, 60, 238, 97, 28, 133, 236, 38, 60, 144, 108, 70, 55, 114, 198, 145, 27, 25, 238, 192, 150, 202, 118, 236, 94, 49, 225, 227, 2, 3, 1, 0, 1]);

    // plaintext for RSA-PSS
    var plaintext = new Uint8Array([95, 77, 186, 79, 50, 12, 12, 232, 118, 114, 90, 252, 229, 251, 210, 91, 248, 62, 90, 113, 37, 160, 140, 175, 231, 60, 62, 186, 196, 33, 119, 157, 249, 213, 93, 24, 12, 58, 233, 148, 38, 69, 225, 216, 47, 238, 140, 157, 41, 75, 60, 177, 160, 138, 153, 49, 32, 27, 60, 14, 129, 252, 71, 202, 207, 131, 21, 162, 175, 102, 50, 65, 19, 195, 182, 98, 48, 195, 70, 8, 196, 244, 89, 54, 52, 206, 2, 178, 103, 54, 34, 119, 240, 168, 64, 202, 116, 188, 61, 26, 98, 54, 149, 44, 94, 215, 170, 248, 168, 254, 203, 221, 250, 117, 132, 230, 151, 140, 234, 93, 42, 91, 159, 183, 241, 180, 140, 139, 11, 229, 138, 48, 82, 2, 117, 77, 131, 118, 16, 115, 116, 121, 60, 240, 38, 170, 238, 83, 0, 114, 125, 131, 108, 215, 30, 113, 179, 69, 221, 178, 228, 68, 70, 255, 197, 185, 1, 99, 84, 19, 137, 13, 145, 14, 163, 128, 152, 74, 144, 25, 16, 49, 50, 63, 22, 219, 204, 157, 107, 225, 104, 184, 72, 133, 56, 76, 160, 62, 18, 96, 10, 193, 194, 72, 2, 138, 243, 114, 108, 201, 52, 99, 136, 46, 168, 192, 42, 171]);

    // For verification tests. Note that "salted" signatures use a salt length equal to the hash size
    var signatures = {
        "sha-1, no salt": new Uint8Array([31, 28, 216, 30, 203, 59, 179, 29, 242, 229, 240, 246, 76, 92, 10, 49, 12, 124, 248, 141, 25, 235, 81, 42, 80, 120, 225, 86, 216, 35, 114, 122, 248, 134, 138, 18, 229, 223, 191, 169, 118, 56, 103, 132, 186, 152, 44, 57, 146, 135, 137, 177, 52, 149, 42, 74, 40, 198, 36, 17, 119, 188, 242, 36, 143, 42, 219, 96, 7, 127, 84, 93, 209, 126, 79, 128, 155, 59, 133, 159, 212, 48, 209, 104, 30, 128, 71, 18, 109, 119, 54, 149, 25, 238, 213, 182, 24, 243, 41, 122, 87, 80, 133, 240, 201, 49, 237, 36, 140, 246, 11, 189, 126, 255, 250, 10, 140, 43, 135, 75, 167, 248, 30, 205, 107, 243, 145, 208, 31, 30, 136, 29, 130, 122, 123, 149, 223, 135, 77, 154, 218, 187, 123, 7, 241, 49, 171, 51, 20, 42, 139, 11, 109, 92, 169, 104, 86, 113, 212, 155, 152, 43, 103, 101, 25, 9, 234, 161, 123, 150, 179, 147, 224, 79, 179, 109, 151, 47, 155, 37, 143, 27, 121, 18, 61, 242, 18, 211, 153, 36, 164, 222, 174, 197, 6, 207, 100, 15, 29, 237, 208, 45, 40, 132, 95, 53, 72, 216, 72, 134, 82, 120, 142, 46, 33, 70, 243, 206, 138, 134, 165, 86, 216, 75, 69, 120, 241, 13, 162, 154, 189, 177, 118, 166, 135, 24, 204, 27, 34, 112, 176, 115, 92, 46, 92, 166, 198, 187, 10, 250, 194, 58, 91, 250, 129, 122]),
        "sha-256, no salt": new Uint8Array([97, 87, 214, 104, 237, 101, 93, 151, 139, 76, 21, 140, 132, 25, 235, 128, 113, 141, 253, 252, 125, 75, 52, 53, 127, 153, 23, 233, 225, 22, 182, 243, 182, 80, 64, 201, 209, 97, 85, 192, 129, 214, 136, 122, 188, 179, 186, 79, 250, 1, 145, 228, 128, 126, 226, 6, 104, 26, 161, 212, 128, 158, 162, 13, 229, 24, 107, 119, 227, 202, 206, 208, 127, 201, 179, 215, 27, 157, 240, 172, 129, 181, 195, 39, 63, 243, 247, 79, 50, 167, 173, 52, 198, 80, 98, 163, 21, 64, 206, 211, 5, 39, 239, 164, 183, 170, 45, 39, 255, 127, 128, 83, 95, 62, 101, 206, 53, 46, 185, 225, 139, 80, 84, 65, 109, 233, 89, 53, 74, 77, 204, 203, 37, 66, 227, 58, 131, 88, 237, 166, 32, 168, 101, 61, 214, 69, 143, 86, 171, 148, 254, 225, 220, 1, 239, 66, 251, 137, 88, 170, 19, 72, 16, 228, 216, 254, 29, 212, 254, 238, 106, 240, 71, 66, 248, 13, 165, 121, 56, 117, 167, 138, 42, 76, 192, 141, 78, 10, 104, 171, 3, 241, 192, 34, 160, 232, 167, 211, 9, 96, 137, 146, 210, 78, 205, 215, 232, 241, 137, 94, 62, 92, 211, 110, 73, 144, 107, 83, 25, 50, 217, 255, 149, 134, 24, 177, 165, 15, 152, 69, 95, 81, 94, 12, 99, 16, 61, 46, 78, 22, 81, 175, 197, 102, 235, 156, 173, 30, 126, 250, 225, 169, 117, 12, 56, 128]),
        "sha-384, no salt": new Uint8Array([123, 149, 170, 182, 179, 76, 9, 98, 210, 40, 64, 158, 48, 223, 155, 4, 60, 27, 11, 170, 218, 8, 231, 61, 136, 116, 34, 85, 43, 143, 21, 34, 226, 228, 43, 242, 185, 255, 44, 108, 154, 163, 235, 12, 210, 55, 6, 24, 232, 241, 163, 104, 115, 89, 94, 0, 189, 231, 90, 156, 224, 98, 236, 50, 181, 246, 57, 79, 34, 103, 163, 245, 193, 24, 64, 255, 146, 230, 225, 91, 243, 28, 197, 62, 145, 124, 168, 239, 192, 137, 95, 177, 18, 194, 239, 143, 104, 28, 187, 106, 65, 1, 82, 246, 233, 48, 202, 255, 31, 38, 14, 49, 249, 131, 84, 46, 104, 205, 21, 222, 161, 126, 211, 19, 156, 172, 115, 81, 6, 251, 5, 252, 22, 59, 46, 208, 90, 13, 237, 147, 144, 89, 161, 12, 92, 215, 97, 158, 33, 178, 210, 6, 144, 121, 148, 39, 75, 52, 164, 218, 239, 161, 206, 89, 182, 179, 25, 247, 57, 85, 160, 145, 138, 94, 35, 126, 27, 191, 218, 219, 69, 201, 7, 165, 0, 131, 87, 126, 113, 146, 129, 136, 69, 153, 91, 74, 109, 63, 241, 151, 142, 15, 154, 66, 105, 88, 83, 40, 46, 53, 195, 183, 129, 51, 179, 224, 198, 36, 18, 90, 255, 20, 161, 135, 61, 25, 143, 99, 4, 255, 236, 127, 193, 207, 42, 222, 204, 108, 209, 75, 31, 137, 177, 166, 55, 247, 46, 209, 255, 93, 231, 198, 180, 217, 101, 153]),
        "sha-512, no salt": new Uint8Array([175, 27, 192, 127, 167, 10, 221, 25, 243, 206, 31, 27, 239, 141, 252, 110, 36, 175, 67, 103, 28, 251, 151, 230, 184, 105, 232, 107, 126, 240, 53, 80, 166, 88, 19, 24, 255, 246, 68, 154, 250, 139, 103, 231, 62, 42, 106, 20, 226, 6, 119, 216, 176, 103, 20, 90, 132, 66, 37, 116, 174, 12, 253, 42, 93, 255, 112, 198, 215, 233, 127, 106, 14, 22, 101, 5, 7, 158, 180, 38, 74, 67, 196, 147, 242, 235, 63, 176, 111, 172, 192, 27, 230, 7, 116, 194, 119, 100, 106, 40, 8, 18, 71, 103, 150, 34, 178, 32, 34, 126, 146, 73, 117, 72, 103, 170, 143, 225, 128, 64, 21, 196, 249, 135, 0, 152, 46, 218, 64, 232, 77, 11, 160, 51, 108, 244, 79, 88, 47, 184, 120, 19, 116, 128, 78, 143, 180, 62, 185, 213, 119, 172, 244, 114, 53, 135, 163, 154, 43, 74, 158, 22, 139, 118, 118, 50, 183, 165, 84, 247, 123, 197, 39, 40, 33, 201, 56, 192, 153, 75, 22, 47, 116, 130, 99, 111, 127, 250, 197, 100, 161, 155, 215, 51, 244, 135, 120, 1, 220, 50, 77, 196, 113, 150, 239, 18, 202, 154, 143, 73, 33, 165, 73, 108, 214, 115, 121, 53, 202, 85, 91, 115, 70, 109, 221, 129, 126, 175, 240, 63, 237, 160, 235, 45, 97, 46, 60, 219, 89, 177, 152, 158, 239, 253, 193, 129, 1, 212, 110, 86, 185, 255, 92, 145, 249, 93]),
        "sha-1, salted": new Uint8Array([31, 96, 138, 113, 209, 136, 76, 254, 33, 131, 180, 144, 55, 170, 133, 85, 176, 19, 154, 138, 18, 103, 165, 197, 185, 204, 226, 7, 1, 242, 173, 75, 189, 91, 50, 151, 64, 191, 243, 26, 204, 195, 75, 249, 175, 209, 67, 154, 5, 54, 187, 50, 182, 212, 39, 210, 105, 104, 219, 201, 233, 200, 13, 33, 17, 217, 72, 196, 129, 203, 23, 49, 119, 138, 205, 49, 16, 70, 50, 65, 196, 242, 59, 62, 19, 184, 85, 209, 98, 203, 21, 56, 81, 41, 15, 217, 95, 120, 21, 25, 226, 206, 249, 55, 69, 164, 19, 207, 238, 200, 233, 79, 186, 120, 34, 183, 37, 212, 116, 67, 24, 69, 140, 246, 180, 169, 23, 182, 91, 21, 238, 111, 84, 185, 195, 145, 246, 6, 74, 158, 3, 31, 112, 9, 245, 146, 68, 156, 11, 70, 213, 69, 122, 39, 153, 203, 14, 189, 120, 161, 2, 160, 85, 238, 4, 112, 178, 96, 194, 179, 216, 255, 189, 238, 15, 212, 118, 68, 130, 32, 144, 236, 85, 174, 98, 51, 190, 16, 98, 244, 65, 196, 50, 237, 60, 39, 94, 116, 214, 32, 19, 38, 129, 236, 46, 128, 30, 155, 91, 106, 204, 26, 215, 31, 137, 53, 56, 143, 126, 44, 3, 55, 13, 18, 233, 68, 227, 65, 140, 42, 182, 59, 180, 42, 190, 27, 185, 230, 149, 48, 240, 36, 88, 186, 40, 64, 11, 54, 128, 111, 247, 141, 165, 121, 26, 206]),
        "sha-256, salted": new Uint8Array([140, 61, 3, 189, 232, 196, 45, 148, 83, 99, 27, 11, 170, 200, 158, 98, 150, 218, 32, 84, 55, 19, 192, 4, 223, 53, 188, 26, 111, 174, 32, 90, 178, 191, 88, 83, 105, 104, 144, 115, 205, 238, 52, 90, 214, 226, 120, 59, 45, 218, 24, 123, 73, 121, 234, 4, 87, 70, 55, 88, 21, 110, 16, 62, 237, 208, 239, 24, 52, 211, 91, 214, 173, 84, 13, 155, 139, 34, 95, 209, 119, 14, 81, 78, 160, 175, 53, 247, 7, 242, 231, 160, 56, 43, 230, 245, 237, 157, 107, 89, 29, 83, 108, 225, 33, 91, 23, 239, 62, 235, 69, 11, 180, 138, 0, 23, 73, 124, 103, 190, 2, 64, 71, 10, 221, 210, 137, 26, 129, 168, 241, 207, 110, 128, 227, 248, 55, 254, 66, 55, 98, 146, 223, 85, 91, 139, 5, 147, 27, 105, 83, 5, 151, 250, 227, 109, 205, 1, 177, 200, 23, 103, 212, 236, 212, 202, 240, 107, 239, 192, 53, 34, 75, 221, 42, 94, 107, 137, 213, 21, 57, 35, 90, 201, 85, 112, 231, 87, 219, 215, 15, 220, 21, 4, 0, 1, 176, 123, 147, 123, 240, 20, 140, 204, 0, 95, 76, 39, 42, 207, 95, 143, 192, 150, 163, 125, 38, 32, 142, 150, 172, 52, 28, 45, 29, 33, 44, 68, 214, 213, 21, 108, 147, 79, 102, 239, 66, 253, 186, 199, 122, 32, 134, 129, 85, 11, 4, 139, 70, 110, 50, 199, 108, 122, 123, 7]),
        "sha-384, salted": new Uint8Array([121, 247, 40, 75, 180, 33, 109, 230, 132, 41, 133, 78, 219, 66, 24, 239, 120, 173, 23, 64, 132, 133, 103, 55, 115, 21, 219, 136, 103, 161, 87, 51, 199, 4, 46, 139, 249, 7, 98, 230, 115, 201, 12, 14, 44, 88, 198, 197, 206, 244, 151, 86, 139, 217, 42, 109, 33, 150, 18, 196, 117, 108, 85, 250, 196, 85, 7, 248, 22, 8, 188, 39, 32, 218, 78, 237, 213, 178, 62, 31, 60, 135, 64, 198, 180, 205, 126, 76, 240, 224, 67, 66, 177, 132, 193, 17, 1, 153, 230, 80, 128, 215, 59, 152, 94, 97, 29, 102, 248, 233, 121, 144, 129, 110, 73, 23, 186, 219, 176, 66, 93, 217, 67, 131, 137, 46, 42, 169, 109, 228, 219, 13, 224, 147, 106, 238, 132, 213, 72, 42, 61, 163, 27, 39, 49, 159, 67, 131, 15, 196, 135, 3, 204, 125, 78, 174, 219, 32, 253, 48, 50, 61, 191, 63, 34, 96, 141, 181, 22, 55, 211, 179, 5, 179, 25, 121, 98, 101, 141, 128, 147, 92, 38, 109, 51, 204, 251, 41, 117, 144, 98, 31, 74, 150, 124, 114, 69, 233, 43, 1, 88, 192, 220, 234, 148, 62, 42, 206, 113, 158, 189, 177, 150, 169, 186, 231, 223, 62, 217, 204, 98, 118, 94, 39, 182, 53, 113, 116, 62, 40, 160, 83, 141, 176, 130, 92, 173, 37, 57, 235, 93, 229, 230, 163, 32, 168, 139, 87, 62, 193, 151, 44, 38, 64, 21, 48]),
        "sha-512, salted": new Uint8Array([183, 79, 48, 153, 216, 7, 135, 17, 139, 31, 157, 231, 159, 194, 7, 137, 62, 13, 45, 117, 196, 17, 15, 75, 21, 155, 133, 186, 7, 214, 58, 2, 86, 252, 60, 208, 246, 108, 232, 217, 162, 227, 207, 122, 61, 90, 123, 156, 11, 239, 172, 102, 56, 137, 74, 62, 54, 206, 117, 230, 73, 238, 6, 157, 216, 221, 152, 170, 139, 96, 36, 116, 201, 139, 20, 187, 3, 73, 45, 229, 81, 169, 232, 231, 121, 52, 239, 155, 104, 69, 131, 147, 79, 33, 141, 149, 118, 190, 36, 11, 92, 79, 54, 46, 175, 94, 1, 64, 200, 234, 146, 99, 144, 133, 166, 38, 150, 83, 80, 93, 207, 160, 4, 34, 109, 185, 246, 50, 119, 101, 58, 100, 161, 130, 110, 75, 171, 177, 122, 181, 77, 216, 84, 61, 207, 28, 232, 9, 112, 109, 104, 22, 230, 167, 95, 248, 70, 163, 212, 193, 141, 17, 189, 235, 31, 49, 177, 13, 85, 163, 121, 91, 100, 150, 49, 158, 110, 117, 21, 4, 216, 106, 78, 123, 182, 83, 91, 159, 4, 21, 232, 21, 216, 199, 137, 197, 177, 227, 135, 242, 168, 192, 15, 239, 110, 50, 116, 98, 203, 126, 82, 91, 143, 148, 91, 229, 177, 114, 72, 224, 224, 164, 216, 85, 211, 151, 226, 45, 6, 124, 228, 83, 147, 115, 223, 186, 70, 209, 121, 146, 80, 175, 199, 15, 83, 80, 6, 202, 205, 39, 102, 245, 221, 207, 143, 145])
    };

    var vectors = [
        {
            name: "RSA-PSS with SHA-1 and no salt",
            publicKeyBuffer: spki,
            publicKeyFormat: "spki",
            privateKey: null,
            privateKeyBuffer: pkcs8,
            privateKeyFormat: "pkcs8",
            publicKey: null,
            algorithm: {name: "RSA-PSS", saltLength: 0},
            hash: "SHA-1",
            plaintext: plaintext,
            signature: signatures["sha-1, no salt"]
        },
        {
            name: "RSA-PSS with SHA-256 and no salt",
            publicKeyBuffer: spki,
            publicKeyFormat: "spki",
            privateKey: null,
            privateKeyBuffer: pkcs8,
            privateKeyFormat: "pkcs8",
            publicKey: null,
            algorithm: {name: "RSA-PSS", saltLength: 0},
            hash: "SHA-256",
            plaintext: plaintext,
            signature: signatures["sha-256, no salt"]
        },
        {
            name: "RSA-PSS with SHA-384 and no salt",
            publicKeyBuffer: spki,
            publicKeyFormat: "spki",
            privateKey: null,
            privateKeyBuffer: pkcs8,
            privateKeyFormat: "pkcs8",
            publicKey: null,
            algorithm: {name: "RSA-PSS", saltLength: 0},
            hash: "SHA-384",
            plaintext: plaintext,
            signature: signatures["sha-384, no salt"]
        },
        {
            name: "RSA-PSS with SHA-512 and no salt",
            publicKeyBuffer: spki,
            publicKeyFormat: "spki",
            privateKey: null,
            privateKeyBuffer: pkcs8,
            privateKeyFormat: "pkcs8",
            publicKey: null,
            algorithm: {name: "RSA-PSS", saltLength: 0},
            hash: "SHA-512",
            plaintext: plaintext,
            signature: signatures["sha-512, no salt"]
        },
        {
            name: "RSA-PSS with SHA-1, salted",
            publicKeyBuffer: spki,
            publicKeyFormat: "spki",
            privateKey: null,
            privateKeyBuffer: pkcs8,
            privateKeyFormat: "pkcs8",
            publicKey: null,
            algorithm: {name: "RSA-PSS", saltLength: 20},
            hash: "SHA-1",
            plaintext: plaintext,
            signature: signatures["sha-1, salted"]
        },
        {
            name: "RSA-PSS with SHA-256, salted",
            publicKeyBuffer: spki,
            publicKeyFormat: "spki",
            privateKey: null,
            privateKeyBuffer: pkcs8,
            privateKeyFormat: "pkcs8",
            publicKey: null,
            algorithm: {name: "RSA-PSS", saltLength: 32},
            hash: "SHA-256",
            plaintext: plaintext,
            signature: signatures["sha-256, salted"]
        },
        {
            name: "RSA-PSS with SHA-384, salted",
            publicKeyBuffer: spki,
            publicKeyFormat: "spki",
            privateKey: null,
            privateKeyBuffer: pkcs8,
            privateKeyFormat: "pkcs8",
            publicKey: null,
            algorithm: {name: "RSA-PSS", saltLength: 48},
            hash: "SHA-384",
            plaintext: plaintext,
            signature: signatures["sha-384, salted"]
        },
        {
            name: "RSA-PSS with SHA-512, salted",
            publicKeyBuffer: spki,
            publicKeyFormat: "spki",
            privateKey: null,
            privateKeyBuffer: pkcs8,
            privateKeyFormat: "pkcs8",
            publicKey: null,
            algorithm: {name: "RSA-PSS", saltLength: 64},
            hash: "SHA-512",
            plaintext: plaintext,
            signature: signatures["sha-512, salted"]
        }
    ];

    return vectors;
}
