# @antora/user-require-helper

A Node.js module that provides helper functions to resolve the module name or path in relation to the specified context and require it (using `userRequire`) or return it (using `userRequire.resolve`).
The expanded path is system dependent.
Developed for use in Antora.

The purpose of this function is to require a JavaScript module (specified as a module name or path request) from the user’s environment.
The `require` function in Node.js resolves module names and paths in the environment of the calling script (i.e., the application code).
That means it will not find modules in the local _node_modules_ folder (unless the calling script is also located inside that folder) or local relative paths.
This assumption is not well suited for user-supplied code.

The described behavior can be altered using `require.resolve`, which is precisely the functionality this helper encapsulates.
Building on the behavior of `require.resolve`, this function honor prefixes recognized by the **@antora/expand-path-helper** module to anchor the location of the requested module name or path to the resolved context specified in the request.
See **Usage** for examples.

For example, the `.:` or `./` prefix will look for a module name or path, respectively, relative to the specified dot argument.
If a context is not specified, and the request argument is a module name, this function will use the function specified by the resolve argument (which defaults to the built-in `require.resolve` function) along with the paths argument, if specified.
In other words, if a bare module name is given, the helper resolves the module relative to the specified paths, if specified, otherwise to the environment of the calling script (thus ignoring the base and dot arguments).

The function inspects the request argument to differentiate between a module name and a path.
If the request argument has a file extension, it’s resolved as a path.
Additionally, if the request argument begins with a dot (`./`) or dot dot (`../`) segment, and it does not contain a colon (`:`), it’s resolved as a path, _even if it doesn’t have a file extension_.
Otherwise, it’s resolved as a module name.

**💡 TIP**\
If the request is for a module name that contains a file extension (e.g., `highlight.js`), append `/` to force it to be treated as a module name (e.g., `highlight.js/`).

## Install

```console
$ npm i @antora/user-require-helper
```

## API

```js
function userRequire (id[, context])
```

Resolves the module name or path using `userRequire.resolve` and requires it.
If the resolved module name or path cannot be found on the local filesystem, an Error with a `MODULE_NOT_FOUND` code is thrown.

* `request` &lt;string> - The module name or path to resolve.
This parameter is required and must not be empty.
* `context` &lt;Object> - Named parameters that control how the request is resolved.
All named parameters are optional.
**Default:** `{}`.
  * `base` &lt;string> - The base directory from which to resolve a relative module name or path.
  Passed on to `expandPath`, which assigns the default value `~+`.
  **Default:** `~+`.
  * `cwd` &lt;string> - The absolute directory to use as the working directory instead of the current working directory.
  Passed on to `expandPath`, which assigns the current working directory as the default value.
  **Default:** `process.cwd()`.
  * `dot` &lt;string> - The directory to use in place of a leading dot (`.`) segment in the module name or path.
  Passed on to `expandPath`, which assigns the default value `.`.
  **Default:** `.`.
  * `paths` &lt;string[]> - The paths to pass to `require.resolve` when a bare module name is specified.
  If this argument is empty, no paths are passed to `require.resolve` (which means it uses the built-in paths).
  **Default:** `[]`.
  * `context.resolve` &lt;Function> - The resolve function to use to resolve a module name or path request.
  Defaults to the built-in `require.resolve` function.
  **Default:** `require.resolve`.

```js
function userRequire.resolve (request[, context])
```

Resolves the module name or path in relation to the specified context and returns it.
If the resolved module name or path cannot be found on the local filesystem, an Error with the code `MODULE_NOT_FOUND` is thrown.

See `userRequire` for description of parameters.

## Usage

### userRequire.resolve

The `userRequire.resolve` function returns a system-dependent path.
Here we show the results when used on a Unix-like operating system.

```js
const userRequire = require('@antora/user-require-helper')

// request is a path

userRequire.resolve('/foo/bar.js')
//=> /foo/bar.js

userRequire.resolve('~/foo/bar.js')
//=> $HOME/foo/bar.js

userRequire.resolve('foo/bar.js')
//=> $PWD/foo/bar.js

userRequire.resolve('foo/bar.js', { base: '/base/dir' })
//=> /base/dir/foo/bar.js

userRequire.resolve('./foo/bar.js')
//=> $PWD/foo/bar.js

userRequire.resolve('./foo/bar.js', { dot: '/dot/dir' })
//=> /dot/dir/foo/bar.js

userRequire.resolve('~+/foo/bar.js')
//=> $PWD/foo/bar.js

// request is a module name

userRequire.resolve('/path/to/node_modules/foo-bar')
//=> /path/to/node_modules/foo-bar/main.js

userRequire.resolve('foo-bar')
//=> /calling/script/dir/node_modules/foo-bar/main.js

userRequire.resolve('foo-bar', { paths: ['/other/dir'] })
//=> /other/dir/node_modules/foo-bar/main.js

userRequire.resolve('^:foo-bar')
//=> /calling/script/dir/node_modules/foo-bar/main.js

userRequire.resolve('^:foo-bar', { paths: ['/other/dir'] })
//=> /calling/script/dir/node_modules/foo-bar/main.js

userRequire.resolve(':foo-bar')
//=> $PWD/node_modules/foo-bar/main.js

userRequire.resolve(':foo-bar', { base: '/base/dir' })
//=> /base/dir/node_modules/foo-bar/main.js

userRequire.resolve('.:foo-bar')
//=> $PWD/node_modules/foo-bar/main.js

userRequire.resolve('.:foo-bar', { dot: '/dot/dir' })
//=> /dot/dir/node_modules/foo-bar/main.js

userRequire.resolve('./subdir:foo-bar', { dot: '/dot/dir' })
//=> /dot/dir/subdir/node_modules/foo-bar/main.js

userRequire.resolve('~+:foo-bar')
//=> $PWD/node_modules/foo-bar/main.js

userRequire.resolve('foo-bar.js/')
//=> /calling/script/dir/node_modules/foo-bar.js/main.js
```

### userRequire

Uses `userRequire.resolve` to resolve the path and returns the object exported by that path by requiring it.

```js
const userRequire = require('@antora/user-require-helper')

const exportedObject = userRequire('./foo/bar.js', { dot: '/dot/dir' })
```

## Copyright and License

Copyright (C) 2021-present by OpenDevise Inc. and the individual contributors to Antora.

Use of this software is granted under the terms of the [Mozilla Public License Version 2.0](https://www.mozilla.org/en-US/MPL/2.0/) (MPL-2.0).
