//========================================================================
//
// PDF417Barcode.cc
//
// Copyright 2018 Glyph & Cog, LLC
//
//========================================================================

#include <aconf.h>

#include <math.h>
#include "gtypes.h"
#include "GString.h"
#include "GList.h"
#include "Error.h"
#include "PDF417Barcode.h"

//------------------------------------------------------------------------

#define maxDataCodewords 928
#define maxErrorCorrectionLevel 8
#define maxErrorCorrectionCodewords (1 << (maxErrorCorrectionLevel + 1))

// mode latches
#define textModeLatch         900
#define byteModeLatchMult6    924
#define byteModeLatchNonMult6 901
#define byteModeShift         913
#define numericModeLatch      902

// text submode latches and shifts
#define textAlphaLowerLatch 27
#define textAlphaMixedLatch 28
#define textAlphaPuncShift  29
#define textLowerAlphaShift 27
#define textLowerMixedLatch 28
#define textLowerPuncShift  29
#define textMixedPuncLatch  25
#define textMixedLowerLatch 27
#define textMixedAlphaLatch 28
#define textMixedPuncShift  29
#define textPuncAlphaLatch  29

#define textSubmodeAlpha      0
#define textSubmodeLower      1
#define textSubmodeMixed      2
#define textSubmodePunc       3
#define textSubmodeAlphaShift 4
#define textSubmodePuncShift  5

// textCompaction[char][submode] = base-30 value
//                                 or 30 if char isn't available
static char textCompaction[128][4] = {
  { 30, 30, 30, 30 },  //   0
  { 30, 30, 30, 30 },  //   1
  { 30, 30, 30, 30 },  //   2
  { 30, 30, 30, 30 },  //   3
  { 30, 30, 30, 30 },  //   4
  { 30, 30, 30, 30 },  //   5
  { 30, 30, 30, 30 },  //   6
  { 30, 30, 30, 30 },  //   7
  { 30, 30, 30, 30 },  //   8
  { 30, 30, 12, 12 },  //   9
  { 30, 30, 30, 15 },  //  10
  { 30, 30, 30, 30 },  //  11
  { 30, 30, 30, 30 },  //  12
  { 30, 30, 11, 11 },  //  13
  { 30, 30, 30, 30 },  //  14
  { 30, 30, 30, 30 },  //  15
  { 30, 30, 30, 30 },  //  16
  { 30, 30, 30, 30 },  //  17
  { 30, 30, 30, 30 },  //  18
  { 30, 30, 30, 30 },  //  19
  { 30, 30, 30, 30 },  //  20
  { 30, 30, 30, 30 },  //  21
  { 30, 30, 30, 30 },  //  22
  { 30, 30, 30, 30 },  //  23
  { 30, 30, 30, 30 },  //  24
  { 30, 30, 30, 30 },  //  25
  { 30, 30, 30, 30 },  //  26
  { 30, 30, 30, 30 },  //  27
  { 30, 30, 30, 30 },  //  28
  { 30, 30, 30, 30 },  //  29
  { 30, 30, 30, 30 },  //  30
  { 30, 30, 30, 30 },  //  31
  { 26, 26, 26, 30 },  //  32
  { 30, 30, 30, 10 },  //  33
  { 30, 30, 30, 20 },  //  34
  { 30, 30, 15, 30 },  //  35
  { 30, 30, 18, 18 },  //  36
  { 30, 30, 21, 30 },  //  37
  { 30, 30, 10, 30 },  //  38
  { 30, 30, 30, 28 },  //  39
  { 30, 30, 30, 23 },  //  40
  { 30, 30, 30, 24 },  //  41
  { 30, 30, 22, 22 },  //  42
  { 30, 30, 20, 30 },  //  43
  { 30, 30, 13, 13 },  //  44
  { 30, 30, 16, 16 },  //  45
  { 30, 30, 17, 17 },  //  46
  { 30, 30, 19, 19 },  //  47
  { 30, 30,  0, 30 },  //  48
  { 30, 30,  1, 30 },  //  49
  { 30, 30,  2, 30 },  //  50
  { 30, 30,  3, 30 },  //  51
  { 30, 30,  4, 30 },  //  52
  { 30, 30,  5, 30 },  //  53
  { 30, 30,  6, 30 },  //  54
  { 30, 30,  7, 30 },  //  55
  { 30, 30,  8, 30 },  //  56
  { 30, 30,  9, 30 },  //  57
  { 30, 30, 14, 14 },  //  58
  { 30, 30, 30,  0 },  //  59
  { 30, 30, 30,  1 },  //  60
  { 30, 30, 23, 30 },  //  61
  { 30, 30, 30,  2 },  //  62
  { 30, 30, 30, 25 },  //  63
  { 30, 30, 30,  3 },  //  64
  {  0, 30, 30, 30 },  //  65
  {  1, 30, 30, 30 },  //  66
  {  2, 30, 30, 30 },  //  67
  {  3, 30, 30, 30 },  //  68
  {  4, 30, 30, 30 },  //  69
  {  5, 30, 30, 30 },  //  70
  {  6, 30, 30, 30 },  //  71
  {  7, 30, 30, 30 },  //  72
  {  8, 30, 30, 30 },  //  73
  {  9, 30, 30, 30 },  //  74
  { 10, 30, 30, 30 },  //  75
  { 11, 30, 30, 30 },  //  76
  { 12, 30, 30, 30 },  //  77
  { 13, 30, 30, 30 },  //  78
  { 14, 30, 30, 30 },  //  79
  { 15, 30, 30, 30 },  //  80
  { 16, 30, 30, 30 },  //  81
  { 17, 30, 30, 30 },  //  82
  { 18, 30, 30, 30 },  //  83
  { 19, 30, 30, 30 },  //  84
  { 20, 30, 30, 30 },  //  85
  { 21, 30, 30, 30 },  //  86
  { 22, 30, 30, 30 },  //  87
  { 23, 30, 30, 30 },  //  88
  { 24, 30, 30, 30 },  //  89
  { 25, 30, 30, 30 },  //  90
  { 30, 30, 30,  4 },  //  91
  { 30, 30, 30,  5 },  //  92
  { 30, 30, 30,  6 },  //  93
  { 30, 30, 24, 30 },  //  94
  { 30, 30, 30,  7 },  //  95
  { 30, 30, 30,  8 },  //  96
  { 30,  0, 30, 30 },  //  97
  { 30,  1, 30, 30 },  //  98
  { 30,  2, 30, 30 },  //  30
  { 30,  3, 30, 30 },  // 100
  { 30,  4, 30, 30 },  // 101
  { 30,  5, 30, 30 },  // 102
  { 30,  6, 30, 30 },  // 103
  { 30,  7, 30, 30 },  // 104
  { 30,  8, 30, 30 },  // 105
  { 30,  9, 30, 30 },  // 106
  { 30, 10, 30, 30 },  // 107
  { 30, 11, 30, 30 },  // 108
  { 30, 12, 30, 30 },  // 109
  { 30, 13, 30, 30 },  // 110
  { 30, 14, 30, 30 },  // 111
  { 30, 15, 30, 30 },  // 112
  { 30, 16, 30, 30 },  // 113
  { 30, 17, 30, 30 },  // 114
  { 30, 18, 30, 30 },  // 115
  { 30, 19, 30, 30 },  // 116
  { 30, 20, 30, 30 },  // 117
  { 30, 21, 30, 30 },  // 118
  { 30, 22, 30, 30 },  // 119
  { 30, 23, 30, 30 },  // 120
  { 30, 24, 30, 30 },  // 121
  { 30, 25, 30, 30 },  // 122
  { 30, 30, 30, 26 },  // 123
  { 30, 30, 30, 21 },  // 124
  { 30, 30, 30, 27 },  // 125
  { 30, 30, 30,  9 },  // 126
  { 30, 30, 30, 30 }   // 127
};

static int errorCorrectionCoeffLevel0[2] = {
  27, 917
};

static int errorCorrectionCoeffLevel1[4] = {
  522, 568, 723, 809
};

static int errorCorrectionCoeffLevel2[8] = {
  237, 308, 436, 284, 646, 653, 428, 379
};

static int errorCorrectionCoeffLevel3[16] = {
  274, 562, 232, 755, 599, 524, 801, 132,
  295, 116, 442, 428, 295,  42, 176,  65
};

static int errorCorrectionCoeffLevel4[32] = {
  361, 575, 922, 525, 176, 586, 640, 321,
  536, 742, 677, 742, 687, 284, 193, 517,
  273, 494, 263, 147, 593, 800, 571, 320,
  803, 133, 231, 390, 685, 330,  63, 410
};

static int errorCorrectionCoeffLevel5[64] = {
  539, 422,   6,  93, 862, 771, 453, 106,
  610, 287, 107, 505, 733, 877, 381, 612,
  723, 476, 462, 172, 430, 609, 858, 822,
  543, 376, 511, 400, 672, 762, 283, 184,
  440,  35, 519,  31, 460, 594, 225, 535,
  517, 352, 605, 158, 651, 201, 488, 502,
  648, 733, 717,  83, 404,  97, 280, 771,
  840, 629,   4, 381, 843, 623, 264, 543
};

static int errorCorrectionCoeffLevel6[128] = {
  521, 310, 864, 547, 858, 580, 296, 379,
   53, 779, 897, 444, 400, 925, 749, 415,
  822,  93, 217, 208, 928, 244, 583, 620,
  246, 148, 447, 631, 292, 908, 490, 704,
  516, 258, 457, 907, 594, 723, 674, 292,
  272,  96, 684, 432, 686, 606, 860, 569,
  193, 219, 129, 186, 236, 287, 192, 775,
  278, 173,  40, 379, 712, 463, 646, 776,
  171, 491, 297, 763, 156, 732,  95, 270,
  447,  90, 507,  48, 228, 821, 808, 898,
  784, 663, 627, 378, 382, 262, 380, 602,
  754, 336,  89, 614,  87, 432, 670, 616,
  157, 374, 242, 726, 600, 269, 375, 898,
  845, 454, 354, 130, 814, 587, 804,  34,
  211, 330, 539, 297, 827, 865,  37, 517,
  834, 315, 550,  86, 801,   4, 108, 539
};

static int errorCorrectionCoeffLevel7[256] = {
  524, 894,  75, 766, 882, 857,  74, 204,
   82, 586, 708, 250, 905, 786, 138, 720,
  858, 194, 311, 913, 275, 190, 375, 850,
  438, 733, 194, 280, 201, 280, 828, 757,
  710, 814, 919,  89,  68, 569,  11, 204,
  796, 605, 540, 913, 801, 700, 799, 137,
  439, 418, 592, 668, 353, 859, 370, 694,
  325, 240, 216, 257, 284, 549, 209, 884,
  315,  70, 329, 793, 490, 274, 877, 162,
  749, 812, 684, 461, 334, 376, 849, 521,
  307, 291, 803, 712,  19, 358, 399, 908,
  103, 511,  51,   8, 517, 225, 289, 470,
  637, 731,  66, 255, 917, 269, 463, 830,
  730, 433, 848, 585, 136, 538, 906,  90,
    2, 290, 743, 199, 655, 903, 329,  49,
  802, 580, 355, 588, 188, 462,  10, 134,
  628, 320, 479, 130, 739,  71, 263, 318,
  374, 601, 192, 605, 142, 673, 687, 234,
  722, 384, 177, 752, 607, 640, 455, 193,
  689, 707, 805, 641,  48,  60, 732, 621,
  895, 544, 261, 852, 655, 309, 697, 755,
  756,  60, 231, 773, 434, 421, 726, 528,
  503, 118,  49, 795,  32, 144, 500, 238,
  836, 394, 280, 566, 319,   9, 647, 550,
   73, 914, 342, 126,  32, 681, 331, 792,
  620,  60, 609, 441, 180, 791, 893, 754,
  605, 383, 228, 749, 760, 213,  54, 297,
  134,  54, 834, 299, 922, 191, 910, 532,
  609, 829, 189,  20, 167,  29, 872, 449,
   83, 402,  41, 656, 505, 579, 481, 173,
  404, 251, 688,  95, 497, 555, 642, 543,
  307, 159, 924, 558, 648,  55, 497,  10
};

static int errorCorrectionCoeffLevel8[512] = {
  352,  77, 373, 504,  35, 599, 428, 207,
  409, 574, 118, 498, 285, 380, 350, 492,
  197, 265, 920, 155, 914, 299, 229, 643,
  294, 871, 306,  88,  87, 193, 352, 781,
  846,  75, 327, 520, 435, 543, 203, 666,
  249, 346, 781, 621, 640, 268, 794, 534,
  539, 781, 408, 390, 644, 102, 476, 499,
  290, 632, 545,  37, 858, 916, 552,  41,
  542, 289, 122, 272, 383, 800, 485,  98,
  752, 472, 761, 107, 784, 860, 658, 741,
  290, 204, 681, 407, 855,  85,  99,  62,
  482, 180,  20, 297, 451, 593, 913, 142,
  808, 684, 287, 536, 561,  76, 653, 899,
  729, 567, 744, 390, 513, 192, 516, 258,
  240, 518, 794, 395, 768, 848,  51, 610,
  384, 168, 190, 826, 328, 596, 786, 303,
  570, 381, 415, 641, 156, 237, 151, 429,
  531, 207, 676, 710,  89, 168, 304, 402,
   40, 708, 575, 162, 864, 229,  65, 861,
  841, 512, 164, 477, 221,  92, 358, 785,
  288, 357, 850, 836, 827, 736, 707,  94,
    8, 494, 114, 521,   2, 499, 851, 543,
  152, 729, 771,  95, 248, 361, 578, 323,
  856, 797, 289,  51, 684, 466, 533, 820,
  669,  45, 902, 452, 167, 342, 244, 173,
   35, 463, 651,  51, 699, 591, 452, 578,
   37, 124, 298, 332, 552,  43, 427, 119,
  662, 777, 475, 850, 764, 364, 578, 911,
  283, 711, 472, 420, 245, 288, 594, 394,
  511, 327, 589, 777, 699, 688,  43, 408,
  842, 383, 721, 521, 560, 644, 714, 559,
   62, 145, 873, 663, 713, 159, 672, 729,
  624,  59, 193, 417, 158, 209, 563, 564,
  343, 693, 109, 608, 563, 365, 181, 772,
  677, 310, 248, 353, 708, 410, 579, 870,
  617, 841, 632, 860, 289, 536,  35, 777,
  618, 586, 424, 833,  77, 597, 346, 269,
  757, 632, 695, 751, 331, 247, 184,  45,
  787, 680,  18,  66, 407, 369,  54, 492,
  228, 613, 830, 922, 437, 519, 644, 905,
  789, 420, 305, 441, 207, 300, 892, 827,
  141, 537, 381, 662, 513,  56, 252, 341,
  242, 797, 838, 837, 720, 224, 307, 631,
   61,  87, 560, 310, 756, 665, 397, 808,
  851, 309, 473, 795, 378,  31, 647, 915,
  459, 806, 590, 731, 425, 216, 548, 249,
  321, 881, 699, 535, 673, 782, 210, 815,
  905, 303, 843, 922, 281,  73, 469, 791,
  660, 162, 498, 308, 155, 422, 907, 817,
  187,  62,  16, 425, 535, 336, 286, 437,
  375, 273, 610, 296, 183, 923, 116, 667,
  751, 353,  62, 366, 691, 379, 687, 842,
   37, 357, 720, 742, 330,   5,  39, 923,
  311, 424, 242, 749, 321,  54, 669, 316,
  342, 299, 534, 105, 667, 488, 640, 672,
  576, 540, 316, 486, 721, 610,  46, 656,
  447, 171, 616, 464, 190, 531, 297, 321,
  762, 752, 533, 175, 134,  14, 381, 433,
  717,  45, 111,  20, 596, 284, 736, 138,
  646, 411, 877, 669, 141, 919,  45, 780,
  407, 164, 332, 899, 165, 726, 600, 325,
  498, 655, 357, 752, 768, 223, 849, 647,
   63, 310, 863, 251, 366, 304, 282, 738,
  675, 410, 389, 244,  31, 121, 303, 263
};

static int *errorCorrectionCoeff[maxErrorCorrectionLevel + 1] = {
  errorCorrectionCoeffLevel0,
  errorCorrectionCoeffLevel1,
  errorCorrectionCoeffLevel2,
  errorCorrectionCoeffLevel3,
  errorCorrectionCoeffLevel4,
  errorCorrectionCoeffLevel5,
  errorCorrectionCoeffLevel6,
  errorCorrectionCoeffLevel7,
  errorCorrectionCoeffLevel8
};

#define startPatternLength 8
static char startPattern[startPatternLength] = {8,1,1,1,1,1,1,3};

#define stopPatternLength 9
static char stopPattern[stopPatternLength] = {7,1,1,3,1,1,1,2,1};

#define patternLength 8

// patterns[codeword][cluster][element] = element (bar or space) width
static char patterns[929][3][patternLength] = {
  {{3,1,1,1,1,1,3,6}, {5,1,1,1,1,1,2,5}, {2,1,1,1,1,1,5,5}},
  {{4,1,1,1,1,1,4,4}, {6,1,1,1,1,1,3,3}, {3,1,1,1,1,1,6,3}},
  {{5,1,1,1,1,1,5,2}, {4,1,1,1,1,2,1,6}, {1,1,1,1,1,2,4,6}},
  {{3,1,1,1,1,2,3,5}, {5,1,1,1,1,2,2,4}, {2,1,1,1,1,2,5,4}},
  {{4,1,1,1,1,2,4,3}, {6,1,1,1,1,2,3,2}, {3,1,1,1,1,2,6,2}},
  {{5,1,1,1,1,2,5,1}, {4,1,1,1,1,3,1,5}, {1,1,1,1,1,3,4,5}},
  {{2,1,1,1,1,3,2,6}, {5,1,1,1,1,3,2,3}, {2,1,1,1,1,3,5,3}},
  {{3,1,1,1,1,3,3,4}, {6,1,1,1,1,3,3,1}, {3,1,1,1,1,3,6,1}},
  {{2,1,1,1,1,4,2,5}, {4,1,1,1,1,4,1,4}, {1,1,1,1,1,4,4,4}},
  {{1,1,1,1,1,5,1,6}, {5,1,1,1,1,4,2,2}, {2,1,1,1,1,4,5,2}},
  {{2,1,1,1,1,5,2,4}, {4,1,1,1,1,5,1,3}, {1,1,1,1,1,5,4,3}},
  {{1,1,1,1,1,6,1,5}, {5,1,1,1,1,5,2,1}, {6,1,1,1,2,1,1,4}},
  {{2,1,1,1,2,1,3,6}, {4,1,1,1,1,6,1,2}, {1,1,1,1,2,1,5,5}},
  {{3,1,1,1,2,1,4,4}, {4,1,1,1,2,1,2,5}, {2,1,1,1,2,1,6,3}},
  {{4,1,1,1,2,1,5,2}, {5,1,1,1,2,1,3,3}, {6,1,1,1,2,2,1,3}},
  {{2,1,1,1,2,2,3,5}, {6,1,1,1,2,1,4,1}, {1,1,1,1,2,2,5,4}},
  {{3,1,1,1,2,2,4,3}, {3,1,1,1,2,2,1,6}, {2,1,1,1,2,2,6,2}},
  {{4,1,1,1,2,2,5,1}, {4,1,1,1,2,2,2,4}, {6,1,1,1,2,3,1,2}},
  {{1,1,1,1,2,3,2,6}, {5,1,1,1,2,2,3,2}, {1,1,1,1,2,3,5,3}},
  {{2,1,1,1,2,3,3,4}, {3,1,1,1,2,3,1,5}, {2,1,1,1,2,3,6,1}},
  {{1,1,1,1,2,4,2,5}, {4,1,1,1,2,3,2,3}, {6,1,1,1,2,4,1,1}},
  {{1,1,1,1,3,1,3,6}, {5,1,1,1,2,3,3,1}, {1,1,1,1,2,4,5,2}},
  {{2,1,1,1,3,1,4,4}, {3,1,1,1,2,4,1,4}, {5,1,1,1,3,1,1,4}},
  {{3,1,1,1,3,1,5,2}, {4,1,1,1,2,4,2,2}, {6,1,1,1,3,1,2,2}},
  {{1,1,1,1,3,2,3,5}, {3,1,1,1,2,5,1,3}, {1,1,1,1,3,1,6,3}},
  {{2,1,1,1,3,2,4,3}, {4,1,1,1,2,5,2,1}, {5,1,1,1,3,2,1,3}},
  {{3,1,1,1,3,2,5,1}, {3,1,1,1,2,6,1,2}, {6,1,1,1,3,2,2,1}},
  {{1,1,1,1,3,3,3,4}, {3,1,1,1,3,1,2,5}, {1,1,1,1,3,2,6,2}},
  {{2,1,1,1,3,3,4,2}, {4,1,1,1,3,1,3,3}, {5,1,1,1,3,3,1,2}},
  {{1,1,1,1,4,1,4,4}, {5,1,1,1,3,1,4,1}, {1,1,1,1,3,3,6,1}},
  {{2,1,1,1,4,1,5,2}, {2,1,1,1,3,2,1,6}, {5,1,1,1,3,4,1,1}},
  {{1,1,1,1,4,2,4,3}, {3,1,1,1,3,2,2,4}, {4,1,1,1,4,1,1,4}},
  {{2,1,1,1,4,2,5,1}, {4,1,1,1,3,2,3,2}, {5,1,1,1,4,1,2,2}},
  {{1,1,1,1,5,1,5,2}, {2,1,1,1,3,3,1,5}, {4,1,1,1,4,2,1,3}},
  {{5,1,1,1,6,1,1,1}, {3,1,1,1,3,3,2,3}, {5,1,1,1,4,2,2,1}},
  {{3,1,1,2,1,1,3,5}, {4,1,1,1,3,3,3,1}, {4,1,1,1,4,3,1,2}},
  {{4,1,1,2,1,1,4,3}, {2,1,1,1,3,4,1,4}, {4,1,1,1,4,4,1,1}},
  {{5,1,1,2,1,1,5,1}, {3,1,1,1,3,4,2,2}, {3,1,1,1,5,1,1,4}},
  {{2,1,1,2,1,2,2,6}, {2,1,1,1,3,5,1,3}, {4,1,1,1,5,1,2,2}},
  {{3,1,1,2,1,2,3,4}, {3,1,1,1,3,5,2,1}, {3,1,1,1,5,2,1,3}},
  {{4,1,1,2,1,2,4,2}, {2,1,1,1,3,6,1,2}, {4,1,1,1,5,2,2,1}},
  {{2,1,1,2,1,3,2,5}, {2,1,1,1,4,1,2,5}, {3,1,1,1,5,3,1,2}},
  {{3,1,1,2,1,3,3,3}, {3,1,1,1,4,1,3,3}, {3,1,1,1,5,4,1,1}},
  {{1,1,1,2,1,4,1,6}, {4,1,1,1,4,1,4,1}, {2,1,1,1,6,1,1,4}},
  {{2,1,1,2,1,4,2,4}, {1,1,1,1,4,2,1,6}, {3,1,1,1,6,1,2,2}},
  {{3,1,1,2,1,4,3,2}, {2,1,1,1,4,2,2,4}, {2,1,1,1,6,2,1,3}},
  {{1,1,1,2,1,5,1,5}, {3,1,1,1,4,2,3,2}, {3,1,1,1,6,2,2,1}},
  {{2,1,1,2,1,5,2,3}, {1,1,1,1,4,3,1,5}, {2,1,1,1,6,3,1,2}},
  {{1,1,1,2,1,6,1,4}, {2,1,1,1,4,3,2,3}, {1,1,1,2,1,1,4,6}},
  {{2,1,1,2,2,1,3,5}, {3,1,1,1,4,3,3,1}, {2,1,1,2,1,1,5,4}},
  {{3,1,1,2,2,1,4,3}, {1,1,1,1,4,4,1,4}, {3,1,1,2,1,1,6,2}},
  {{4,1,1,2,2,1,5,1}, {2,1,1,1,4,4,2,2}, {1,1,1,2,1,2,4,5}},
  {{1,1,1,2,2,2,2,6}, {1,1,1,1,4,5,1,3}, {2,1,1,2,1,2,5,3}},
  {{2,1,1,2,2,2,3,4}, {2,1,1,1,4,5,2,1}, {3,1,1,2,1,2,6,1}},
  {{3,1,1,2,2,2,4,2}, {1,1,1,1,5,1,2,5}, {1,1,1,2,1,3,4,4}},
  {{1,1,1,2,2,3,2,5}, {2,1,1,1,5,1,3,3}, {2,1,1,2,1,3,5,2}},
  {{2,1,1,2,2,3,3,3}, {3,1,1,1,5,1,4,1}, {1,1,1,2,1,4,4,3}},
  {{3,1,1,2,2,3,4,1}, {1,1,1,1,5,2,2,4}, {2,1,1,2,1,4,5,1}},
  {{1,1,1,2,2,4,2,4}, {2,1,1,1,5,2,3,2}, {1,1,1,2,1,5,4,2}},
  {{2,1,1,2,2,4,3,2}, {1,1,1,1,5,3,2,3}, {6,1,1,2,2,1,1,3}},
  {{1,1,1,2,3,1,3,5}, {2,1,1,1,5,3,3,1}, {1,1,1,2,2,1,5,4}},
  {{2,1,1,2,3,1,4,3}, {1,1,1,1,5,4,2,2}, {2,1,1,2,2,1,6,2}},
  {{3,1,1,2,3,1,5,1}, {1,1,1,1,6,1,3,3}, {6,1,1,2,2,2,1,2}},
  {{1,1,1,2,3,2,3,4}, {2,1,1,1,6,1,4,1}, {1,1,1,2,2,2,5,3}},
  {{2,1,1,2,3,2,4,2}, {1,1,1,1,6,2,3,2}, {2,1,1,2,2,2,6,1}},
  {{1,1,1,2,3,3,3,3}, {1,1,1,1,6,3,3,1}, {6,1,1,2,2,3,1,1}},
  {{2,1,1,2,3,3,4,1}, {4,1,1,2,1,1,1,6}, {1,1,1,2,2,3,5,2}},
  {{1,1,1,2,4,1,4,3}, {5,1,1,2,1,1,2,4}, {1,1,1,2,2,4,5,1}},
  {{2,1,1,2,4,1,5,1}, {6,1,1,2,1,1,3,2}, {5,1,1,2,3,1,1,3}},
  {{1,1,1,2,4,2,4,2}, {4,1,1,2,1,2,1,5}, {6,1,1,2,3,1,2,1}},
  {{1,1,1,2,4,3,4,1}, {5,1,1,2,1,2,2,3}, {1,1,1,2,3,1,6,2}},
  {{2,1,1,3,1,1,2,6}, {6,1,1,2,1,2,3,1}, {5,1,1,2,3,2,1,2}},
  {{3,1,1,3,1,1,3,4}, {4,1,1,2,1,3,1,4}, {1,1,1,2,3,2,6,1}},
  {{4,1,1,3,1,1,4,2}, {5,1,1,2,1,3,2,2}, {5,1,1,2,3,3,1,1}},
  {{2,1,1,3,1,2,2,5}, {4,1,1,2,1,4,1,3}, {4,1,1,2,4,1,1,3}},
  {{3,1,1,3,1,2,3,3}, {5,1,1,2,1,4,2,1}, {5,1,1,2,4,1,2,1}},
  {{4,1,1,3,1,2,4,1}, {4,1,1,2,1,5,1,2}, {4,1,1,2,4,2,1,2}},
  {{1,1,1,3,1,3,1,6}, {4,1,1,2,1,6,1,1}, {4,1,1,2,4,3,1,1}},
  {{2,1,1,3,1,3,2,4}, {3,1,1,2,2,1,1,6}, {3,1,1,2,5,1,1,3}},
  {{3,1,1,3,1,3,3,2}, {4,1,1,2,2,1,2,4}, {4,1,1,2,5,1,2,1}},
  {{1,1,1,3,1,4,1,5}, {5,1,1,2,2,1,3,2}, {3,1,1,2,5,2,1,2}},
  {{2,1,1,3,1,4,2,3}, {3,1,1,2,2,2,1,5}, {3,1,1,2,5,3,1,1}},
  {{1,1,1,3,1,5,1,4}, {4,1,1,2,2,2,2,3}, {2,1,1,2,6,1,1,3}},
  {{1,1,1,3,1,6,1,3}, {5,1,1,2,2,2,3,1}, {3,1,1,2,6,1,2,1}},
  {{1,1,1,3,2,1,2,6}, {3,1,1,2,2,3,1,4}, {2,1,1,2,6,2,1,2}},
  {{2,1,1,3,2,1,3,4}, {4,1,1,2,2,3,2,2}, {2,1,1,2,6,3,1,1}},
  {{3,1,1,3,2,1,4,2}, {3,1,1,2,2,4,1,3}, {1,1,1,3,1,1,4,5}},
  {{1,1,1,3,2,2,2,5}, {4,1,1,2,2,4,2,1}, {2,1,1,3,1,1,5,3}},
  {{2,1,1,3,2,2,3,3}, {3,1,1,2,2,5,1,2}, {3,1,1,3,1,1,6,1}},
  {{3,1,1,3,2,2,4,1}, {3,1,1,2,2,6,1,1}, {1,1,1,3,1,2,4,4}},
  {{1,1,1,3,2,3,2,4}, {2,1,1,2,3,1,1,6}, {2,1,1,3,1,2,5,2}},
  {{2,1,1,3,2,3,3,2}, {3,1,1,2,3,1,2,4}, {1,1,1,3,1,3,4,3}},
  {{1,1,1,3,2,4,2,3}, {4,1,1,2,3,1,3,2}, {2,1,1,3,1,3,5,1}},
  {{1,1,1,3,2,5,2,2}, {2,1,1,2,3,2,1,5}, {1,1,1,3,1,4,4,2}},
  {{1,1,1,3,3,1,3,4}, {3,1,1,2,3,2,2,3}, {1,1,1,3,1,5,4,1}},
  {{2,1,1,3,3,1,4,2}, {4,1,1,2,3,2,3,1}, {6,1,1,3,2,1,1,2}},
  {{1,1,1,3,3,2,3,3}, {2,1,1,2,3,3,1,4}, {1,1,1,3,2,1,5,3}},
  {{2,1,1,3,3,2,4,1}, {3,1,1,2,3,3,2,2}, {2,1,1,3,2,1,6,1}},
  {{1,1,1,3,3,3,3,2}, {2,1,1,2,3,4,1,3}, {6,1,1,3,2,2,1,1}},
  {{1,1,1,3,4,1,4,2}, {3,1,1,2,3,4,2,1}, {1,1,1,3,2,2,5,2}},
  {{2,1,1,4,1,1,2,5}, {2,1,1,2,3,5,1,2}, {1,1,1,3,2,3,5,1}},
  {{3,1,1,4,1,1,3,3}, {2,1,1,2,3,6,1,1}, {5,1,1,3,3,1,1,2}},
  {{4,1,1,4,1,1,4,1}, {1,1,1,2,4,1,1,6}, {1,1,1,3,3,1,6,1}},
  {{1,1,1,4,1,2,1,6}, {2,1,1,2,4,1,2,4}, {5,1,1,3,3,2,1,1}},
  {{2,1,1,4,1,2,2,4}, {3,1,1,2,4,1,3,2}, {4,1,1,3,4,1,1,2}},
  {{3,1,1,4,1,2,3,2}, {1,1,1,2,4,2,1,5}, {4,1,1,3,4,2,1,1}},
  {{1,1,1,4,1,3,1,5}, {2,1,1,2,4,2,2,3}, {3,1,1,3,5,1,1,2}},
  {{2,1,1,4,1,3,2,3}, {3,1,1,2,4,2,3,1}, {3,1,1,3,5,2,1,1}},
  {{3,1,1,4,1,3,3,1}, {1,1,1,2,4,3,1,4}, {2,1,1,3,6,1,1,2}},
  {{1,1,1,4,1,4,1,4}, {2,1,1,2,4,3,2,2}, {2,1,1,3,6,2,1,1}},
  {{2,1,1,4,1,4,2,2}, {1,1,1,2,4,4,1,3}, {1,1,1,4,1,1,4,4}},
  {{1,1,1,4,1,5,1,3}, {2,1,1,2,4,4,2,1}, {2,1,1,4,1,1,5,2}},
  {{2,1,1,4,1,5,2,1}, {1,1,1,2,4,5,1,2}, {1,1,1,4,1,2,4,3}},
  {{1,1,1,4,2,1,2,5}, {1,1,1,2,5,1,2,4}, {2,1,1,4,1,2,5,1}},
  {{2,1,1,4,2,1,3,3}, {2,1,1,2,5,1,3,2}, {1,1,1,4,1,3,4,2}},
  {{3,1,1,4,2,1,4,1}, {1,1,1,2,5,2,2,3}, {1,1,1,4,1,4,4,1}},
  {{1,1,1,4,2,2,2,4}, {2,1,1,2,5,2,3,1}, {6,1,1,4,2,1,1,1}},
  {{2,1,1,4,2,2,3,2}, {1,1,1,2,5,3,2,2}, {1,1,1,4,2,1,5,2}},
  {{1,1,1,4,2,3,2,3}, {1,1,1,2,5,4,2,1}, {1,1,1,4,2,2,5,1}},
  {{2,1,1,4,2,3,3,1}, {1,1,1,2,6,1,3,2}, {5,1,1,4,3,1,1,1}},
  {{1,1,1,4,2,4,2,2}, {1,1,1,2,6,2,3,1}, {4,1,1,4,4,1,1,1}},
  {{1,1,1,4,2,5,2,1}, {4,1,1,3,1,1,1,5}, {3,1,1,4,5,1,1,1}},
  {{2,1,1,4,3,1,4,1}, {5,1,1,3,1,1,2,3}, {1,1,1,5,1,1,4,3}},
  {{1,1,1,4,3,3,3,1}, {6,1,1,3,1,1,3,1}, {2,1,1,5,1,1,5,1}},
  {{1,1,1,5,1,1,1,6}, {4,1,1,3,1,2,1,4}, {1,1,1,5,1,2,4,2}},
  {{2,1,1,5,1,1,2,4}, {5,1,1,3,1,2,2,2}, {1,1,1,5,1,3,4,1}},
  {{3,1,1,5,1,1,3,2}, {4,1,1,3,1,3,1,3}, {1,1,1,5,2,1,5,1}},
  {{1,1,1,5,1,2,1,5}, {5,1,1,3,1,3,2,1}, {1,1,1,6,1,1,4,2}},
  {{2,1,1,5,1,2,2,3}, {4,1,1,3,1,4,1,2}, {1,1,1,6,1,2,4,1}},
  {{3,1,1,5,1,2,3,1}, {4,1,1,3,1,5,1,1}, {1,2,1,1,1,1,4,6}},
  {{1,1,1,5,1,3,1,4}, {3,1,1,3,2,1,1,5}, {2,2,1,1,1,1,5,4}},
  {{2,1,1,5,1,3,2,2}, {4,1,1,3,2,1,2,3}, {3,2,1,1,1,1,6,2}},
  {{1,1,1,5,1,4,1,3}, {5,1,1,3,2,1,3,1}, {1,2,1,1,1,2,4,5}},
  {{2,1,1,5,1,4,2,1}, {3,1,1,3,2,2,1,4}, {2,2,1,1,1,2,5,3}},
  {{1,1,1,5,1,5,1,2}, {4,1,1,3,2,2,2,2}, {3,2,1,1,1,2,6,1}},
  {{1,1,1,5,2,1,2,4}, {3,1,1,3,2,3,1,3}, {1,2,1,1,1,3,4,4}},
  {{1,1,1,5,2,2,2,3}, {4,1,1,3,2,3,2,1}, {2,2,1,1,1,3,5,2}},
  {{1,1,1,5,2,3,2,2}, {3,1,1,3,2,4,1,2}, {1,2,1,1,1,4,4,3}},
  {{1,1,1,6,1,1,1,5}, {3,1,1,3,2,5,1,1}, {2,2,1,1,1,4,5,1}},
  {{3,1,1,6,1,1,3,1}, {2,1,1,3,3,1,1,5}, {1,2,1,1,1,5,4,2}},
  {{2,1,1,6,1,2,2,2}, {3,1,1,3,3,1,2,3}, {6,2,1,1,2,1,1,3}},
  {{2,1,1,6,1,3,2,1}, {4,1,1,3,3,1,3,1}, {1,2,1,1,2,1,5,4}},
  {{1,1,1,6,1,5,1,1}, {2,1,1,3,3,2,1,4}, {2,2,1,1,2,1,6,2}},
  {{3,2,1,1,1,1,3,5}, {3,1,1,3,3,2,2,2}, {6,2,1,1,2,2,1,2}},
  {{4,2,1,1,1,1,4,3}, {2,1,1,3,3,3,1,3}, {1,2,1,1,2,2,5,3}},
  {{5,2,1,1,1,1,5,1}, {3,1,1,3,3,3,2,1}, {2,2,1,1,2,2,6,1}},
  {{2,2,1,1,1,2,2,6}, {2,1,1,3,3,4,1,2}, {6,2,1,1,2,3,1,1}},
  {{3,2,1,1,1,2,3,4}, {2,1,1,3,3,5,1,1}, {1,2,1,1,2,3,5,2}},
  {{4,2,1,1,1,2,4,2}, {1,1,1,3,4,1,1,5}, {1,2,1,1,2,4,5,1}},
  {{2,2,1,1,1,3,2,5}, {2,1,1,3,4,1,2,3}, {5,2,1,1,3,1,1,3}},
  {{3,2,1,1,1,3,3,3}, {3,1,1,3,4,1,3,1}, {6,2,1,1,3,1,2,1}},
  {{4,2,1,1,1,3,4,1}, {1,1,1,3,4,2,1,4}, {1,2,1,1,3,1,6,2}},
  {{1,2,1,1,1,4,1,6}, {2,1,1,3,4,2,2,2}, {5,2,1,1,3,2,1,2}},
  {{2,2,1,1,1,4,2,4}, {1,1,1,3,4,3,1,3}, {1,2,1,1,3,2,6,1}},
  {{1,2,1,1,1,5,1,5}, {2,1,1,3,4,3,2,1}, {5,2,1,1,3,3,1,1}},
  {{2,2,1,1,2,1,3,5}, {1,1,1,3,4,4,1,2}, {4,2,1,1,4,1,1,3}},
  {{3,2,1,1,2,1,4,3}, {1,1,1,3,4,5,1,1}, {5,2,1,1,4,1,2,1}},
  {{4,2,1,1,2,1,5,1}, {1,1,1,3,5,1,2,3}, {4,2,1,1,4,2,1,2}},
  {{1,2,1,1,2,2,2,6}, {2,1,1,3,5,1,3,1}, {4,2,1,1,4,3,1,1}},
  {{2,2,1,1,2,2,3,4}, {1,1,1,3,5,2,2,2}, {3,2,1,1,5,1,1,3}},
  {{3,2,1,1,2,2,4,2}, {1,1,1,3,5,3,2,1}, {4,2,1,1,5,1,2,1}},
  {{1,2,1,1,2,3,2,5}, {1,1,1,3,6,1,3,1}, {3,2,1,1,5,2,1,2}},
  {{2,2,1,1,2,3,3,3}, {4,1,1,4,1,1,1,4}, {3,2,1,1,5,3,1,1}},
  {{1,2,1,1,2,4,2,4}, {5,1,1,4,1,1,2,2}, {2,2,1,1,6,1,1,3}},
  {{1,2,1,1,2,5,2,3}, {4,1,1,4,1,2,1,3}, {3,2,1,1,6,1,2,1}},
  {{1,2,1,1,3,1,3,5}, {5,1,1,4,1,2,2,1}, {2,2,1,1,6,2,1,2}},
  {{2,2,1,1,3,1,4,3}, {4,1,1,4,1,3,1,2}, {2,2,1,1,6,3,1,1}},
  {{3,2,1,1,3,1,5,1}, {4,1,1,4,1,4,1,1}, {2,1,2,1,1,1,4,5}},
  {{1,2,1,1,3,2,3,4}, {3,1,1,4,2,1,1,4}, {3,1,2,1,1,1,5,3}},
  {{2,2,1,1,3,2,4,2}, {4,1,1,4,2,1,2,2}, {4,1,2,1,1,1,6,1}},
  {{1,2,1,1,3,3,3,3}, {3,1,1,4,2,2,1,3}, {1,1,2,1,1,2,3,6}},
  {{1,2,1,1,3,4,3,2}, {4,1,1,4,2,2,2,1}, {2,1,2,1,1,2,4,4}},
  {{1,2,1,1,4,1,4,3}, {3,1,1,4,2,3,1,2}, {3,1,2,1,1,2,5,2}},
  {{2,2,1,1,4,1,5,1}, {3,1,1,4,2,4,1,1}, {1,1,2,1,1,3,3,5}},
  {{1,2,1,1,4,2,4,2}, {2,1,1,4,3,1,1,4}, {2,1,2,1,1,3,4,3}},
  {{1,2,1,1,5,1,5,1}, {3,1,1,4,3,1,2,2}, {3,1,2,1,1,3,5,1}},
  {{3,1,2,1,1,1,2,6}, {2,1,1,4,3,2,1,3}, {1,1,2,1,1,4,3,4}},
  {{4,1,2,1,1,1,3,4}, {3,1,1,4,3,2,2,1}, {2,1,2,1,1,4,4,2}},
  {{5,1,2,1,1,1,4,2}, {2,1,1,4,3,3,1,2}, {1,1,2,1,1,5,3,3}},
  {{3,1,2,1,1,2,2,5}, {2,1,1,4,3,4,1,1}, {2,1,2,1,1,5,4,1}},
  {{4,1,2,1,1,2,3,3}, {1,1,1,4,4,1,1,4}, {1,1,2,1,1,6,3,2}},
  {{5,1,2,1,1,2,4,1}, {2,1,1,4,4,1,2,2}, {1,2,1,2,1,1,4,5}},
  {{2,1,2,1,1,3,1,6}, {1,1,1,4,4,2,1,3}, {2,2,1,2,1,1,5,3}},
  {{3,1,2,1,1,3,2,4}, {2,1,1,4,4,2,2,1}, {3,2,1,2,1,1,6,1}},
  {{4,1,2,1,1,3,3,2}, {1,1,1,4,4,3,1,2}, {1,1,2,1,2,1,4,5}},
  {{2,1,2,1,1,4,1,5}, {1,1,1,4,4,4,1,1}, {1,2,1,2,1,2,4,4}},
  {{3,1,2,1,1,4,2,3}, {1,1,1,4,5,1,2,2}, {2,2,1,2,1,2,5,2}},
  {{4,1,2,1,1,4,3,1}, {1,1,1,4,5,2,2,1}, {1,1,2,1,2,2,4,4}},
  {{2,1,2,1,1,5,1,4}, {4,1,1,5,1,1,1,3}, {2,1,2,1,2,2,5,2}},
  {{3,1,2,1,1,5,2,2}, {5,1,1,5,1,1,2,1}, {2,2,1,2,1,3,5,1}},
  {{2,2,1,2,1,1,2,6}, {4,1,1,5,1,2,1,2}, {1,1,2,1,2,3,4,3}},
  {{3,2,1,2,1,1,3,4}, {4,1,1,5,1,3,1,1}, {1,2,1,2,1,4,4,2}},
  {{4,2,1,2,1,1,4,2}, {3,1,1,5,2,1,1,3}, {1,1,2,1,2,4,4,2}},
  {{2,1,2,1,2,1,2,6}, {4,1,1,5,2,1,2,1}, {1,2,1,2,1,5,4,1}},
  {{2,2,1,2,1,2,2,5}, {3,1,1,5,2,2,1,2}, {1,1,2,1,2,5,4,1}},
  {{3,2,1,2,1,2,3,3}, {3,1,1,5,2,3,1,1}, {6,2,1,2,2,1,1,2}},
  {{4,2,1,2,1,2,4,1}, {2,1,1,5,3,1,1,3}, {1,2,1,2,2,1,5,3}},
  {{2,1,2,1,2,2,2,5}, {3,1,1,5,3,1,2,1}, {2,2,1,2,2,1,6,1}},
  {{3,1,2,1,2,2,3,3}, {2,1,1,5,3,2,1,2}, {6,1,2,1,3,1,1,2}},
  {{4,1,2,1,2,2,4,1}, {2,1,1,5,3,3,1,1}, {6,2,1,2,2,2,1,1}},
  {{1,1,2,1,2,3,1,6}, {1,1,1,5,4,1,1,3}, {1,1,2,1,3,1,5,3}},
  {{1,2,1,2,1,4,1,5}, {2,1,1,5,4,1,2,1}, {1,2,1,2,2,2,5,2}},
  {{2,2,1,2,1,4,2,3}, {1,1,1,5,4,2,1,2}, {6,1,2,1,3,2,1,1}},
  {{3,2,1,2,1,4,3,1}, {1,1,1,5,4,3,1,1}, {1,1,2,1,3,2,5,2}},
  {{1,1,2,1,2,4,1,5}, {4,1,1,6,1,1,1,2}, {1,2,1,2,2,3,5,1}},
  {{2,1,2,1,2,4,2,3}, {4,1,1,6,1,2,1,1}, {1,1,2,1,3,3,5,1}},
  {{1,1,2,1,2,5,1,4}, {3,1,1,6,2,1,1,2}, {5,2,1,2,3,1,1,2}},
  {{1,2,1,2,2,1,2,6}, {3,1,1,6,2,2,1,1}, {1,2,1,2,3,1,6,1}},
  {{2,2,1,2,2,1,3,4}, {2,1,1,6,3,1,1,2}, {5,1,2,1,4,1,1,2}},
  {{3,2,1,2,2,1,4,2}, {2,1,1,6,3,2,1,1}, {5,2,1,2,3,2,1,1}},
  {{1,1,2,1,3,1,2,6}, {4,2,1,1,1,1,1,6}, {1,1,2,1,4,1,6,1}},
  {{1,2,1,2,2,2,2,5}, {5,2,1,1,1,1,2,4}, {5,1,2,1,4,2,1,1}},
  {{2,2,1,2,2,2,3,3}, {6,2,1,1,1,1,3,2}, {4,2,1,2,4,1,1,2}},
  {{3,2,1,2,2,2,4,1}, {4,2,1,1,1,2,1,5}, {4,1,2,1,5,1,1,2}},
  {{1,1,2,1,3,2,2,5}, {5,2,1,1,1,2,2,3}, {4,2,1,2,4,2,1,1}},
  {{2,1,2,1,3,2,3,3}, {6,2,1,1,1,2,3,1}, {4,1,2,1,5,2,1,1}},
  {{3,1,2,1,3,2,4,1}, {4,2,1,1,1,3,1,4}, {3,2,1,2,5,1,1,2}},
  {{1,1,2,1,3,3,2,4}, {5,2,1,1,1,3,2,2}, {3,1,2,1,6,1,1,2}},
  {{1,2,1,2,2,4,2,3}, {4,2,1,1,1,4,1,3}, {3,2,1,2,5,2,1,1}},
  {{1,1,2,1,3,4,2,3}, {5,2,1,1,1,4,2,1}, {3,1,2,1,6,2,1,1}},
  {{1,2,1,2,3,1,3,4}, {4,2,1,1,1,5,1,2}, {2,2,1,2,6,1,1,2}},
  {{2,2,1,2,3,1,4,2}, {4,2,1,1,1,6,1,1}, {2,2,1,2,6,2,1,1}},
  {{1,1,2,1,4,1,3,4}, {3,2,1,1,2,1,1,6}, {1,1,2,2,1,1,3,6}},
  {{1,2,1,2,3,2,3,3}, {4,2,1,1,2,1,2,4}, {2,1,2,2,1,1,4,4}},
  {{2,2,1,2,3,2,4,1}, {5,2,1,1,2,1,3,2}, {3,1,2,2,1,1,5,2}},
  {{1,1,2,1,4,2,3,3}, {3,2,1,1,2,2,1,5}, {1,1,2,2,1,2,3,5}},
  {{2,1,2,1,4,2,4,1}, {4,2,1,1,2,2,2,3}, {2,1,2,2,1,2,4,3}},
  {{1,1,2,1,4,3,3,2}, {5,2,1,1,2,2,3,1}, {3,1,2,2,1,2,5,1}},
  {{1,2,1,2,4,1,4,2}, {3,2,1,1,2,3,1,4}, {1,1,2,2,1,3,3,4}},
  {{1,1,2,1,5,1,4,2}, {4,2,1,1,2,3,2,2}, {2,1,2,2,1,3,4,2}},
  {{1,2,1,2,4,2,4,1}, {3,2,1,1,2,4,1,3}, {1,1,2,2,1,4,3,3}},
  {{1,1,2,1,5,2,4,1}, {4,2,1,1,2,4,2,1}, {2,1,2,2,1,4,4,1}},
  {{3,1,2,2,1,1,2,5}, {3,2,1,1,2,5,1,2}, {1,1,2,2,1,5,3,2}},
  {{4,1,2,2,1,1,3,3}, {3,2,1,1,2,6,1,1}, {1,1,2,2,1,6,3,1}},
  {{5,1,2,2,1,1,4,1}, {2,2,1,1,3,1,1,6}, {1,2,1,3,1,1,4,4}},
  {{2,1,2,2,1,2,1,6}, {3,2,1,1,3,1,2,4}, {2,2,1,3,1,1,5,2}},
  {{3,1,2,2,1,2,2,4}, {4,2,1,1,3,1,3,2}, {1,1,2,2,2,1,4,4}},
  {{4,1,2,2,1,2,3,2}, {2,2,1,1,3,2,1,5}, {1,2,1,3,1,2,4,3}},
  {{2,1,2,2,1,3,1,5}, {3,2,1,1,3,2,2,3}, {2,2,1,3,1,2,5,1}},
  {{3,1,2,2,1,3,2,3}, {4,2,1,1,3,2,3,1}, {1,1,2,2,2,2,4,3}},
  {{4,1,2,2,1,3,3,1}, {2,2,1,1,3,3,1,4}, {2,1,2,2,2,2,5,1}},
  {{2,1,2,2,1,4,1,4}, {3,2,1,1,3,3,2,2}, {1,1,2,2,2,3,4,2}},
  {{3,1,2,2,1,4,2,2}, {2,2,1,1,3,4,1,3}, {1,2,1,3,1,4,4,1}},
  {{2,1,2,2,1,5,1,3}, {3,2,1,1,3,4,2,1}, {1,1,2,2,2,4,4,1}},
  {{2,1,2,2,1,6,1,2}, {2,2,1,1,3,5,1,2}, {6,2,1,3,2,1,1,1}},
  {{2,2,1,3,1,1,2,5}, {2,2,1,1,3,6,1,1}, {1,2,1,3,2,1,5,2}},
  {{3,2,1,3,1,1,3,3}, {1,2,1,1,4,1,1,6}, {6,1,2,2,3,1,1,1}},
  {{4,2,1,3,1,1,4,1}, {2,2,1,1,4,1,2,4}, {1,1,2,2,3,1,5,2}},
  {{2,1,2,2,2,1,2,5}, {3,2,1,1,4,1,3,2}, {1,2,1,3,2,2,5,1}},
  {{2,2,1,3,1,2,2,4}, {1,2,1,1,4,2,1,5}, {1,1,2,2,3,2,5,1}},
  {{3,2,1,3,1,2,3,2}, {2,2,1,1,4,2,2,3}, {5,2,1,3,3,1,1,1}},
  {{1,1,2,2,2,2,1,6}, {3,2,1,1,4,2,3,1}, {5,1,2,2,4,1,1,1}},
  {{1,2,1,3,1,3,1,5}, {1,2,1,1,4,3,1,4}, {4,2,1,3,4,1,1,1}},
  {{3,1,2,2,2,2,3,2}, {2,2,1,1,4,3,2,2}, {4,1,2,2,5,1,1,1}},
  {{3,2,1,3,1,3,3,1}, {1,2,1,1,4,4,1,3}, {3,2,1,3,5,1,1,1}},
  {{1,1,2,2,2,3,1,5}, {2,2,1,1,4,4,2,1}, {3,1,2,2,6,1,1,1}},
  {{1,2,1,3,1,4,1,4}, {1,2,1,1,4,5,1,2}, {2,2,1,3,6,1,1,1}},
  {{2,2,1,3,1,4,2,2}, {1,2,1,1,5,1,2,4}, {1,1,2,3,1,1,3,5}},
  {{1,1,2,2,2,4,1,4}, {2,2,1,1,5,1,3,2}, {2,1,2,3,1,1,4,3}},
  {{2,1,2,2,2,4,2,2}, {1,2,1,1,5,2,2,3}, {3,1,2,3,1,1,5,1}},
  {{2,2,1,3,1,5,2,1}, {2,2,1,1,5,2,3,1}, {1,1,2,3,1,2,3,4}},
  {{1,2,1,3,1,6,1,2}, {1,2,1,1,5,3,2,2}, {2,1,2,3,1,2,4,2}},
  {{1,2,1,3,2,1,2,5}, {1,2,1,1,5,4,2,1}, {1,1,2,3,1,3,3,3}},
  {{2,2,1,3,2,1,3,3}, {1,2,1,1,6,1,3,2}, {2,1,2,3,1,3,4,1}},
  {{3,2,1,3,2,1,4,1}, {1,2,1,1,6,2,3,1}, {1,1,2,3,1,4,3,2}},
  {{1,1,2,2,3,1,2,5}, {5,1,2,1,1,1,1,5}, {1,1,2,3,1,5,3,1}},
  {{1,2,1,3,2,2,2,4}, {6,1,2,1,1,1,2,3}, {1,2,1,4,1,1,4,3}},
  {{2,2,1,3,2,2,3,2}, {1,1,2,1,1,1,6,4}, {2,2,1,4,1,1,5,1}},
  {{1,1,2,2,3,2,2,4}, {5,1,2,1,1,2,1,4}, {1,1,2,3,2,1,4,3}},
  {{2,1,2,2,3,2,3,2}, {6,1,2,1,1,2,2,2}, {1,2,1,4,1,2,4,2}},
  {{2,2,1,3,2,3,3,1}, {1,1,2,1,1,2,6,3}, {1,1,2,3,2,2,4,2}},
  {{1,1,2,2,3,3,2,3}, {5,1,2,1,1,3,1,3}, {1,2,1,4,1,3,4,1}},
  {{1,2,1,3,2,4,2,2}, {6,1,2,1,1,3,2,1}, {1,1,2,3,2,3,4,1}},
  {{1,2,1,3,2,5,2,1}, {1,1,2,1,1,3,6,2}, {1,2,1,4,2,1,5,1}},
  {{1,2,1,3,3,1,3,3}, {5,1,2,1,1,4,1,2}, {1,1,2,3,3,1,5,1}},
  {{2,2,1,3,3,1,4,1}, {5,1,2,1,1,5,1,1}, {1,1,2,4,1,1,3,4}},
  {{1,1,2,2,4,1,3,3}, {4,2,1,2,1,1,1,5}, {2,1,2,4,1,1,4,2}},
  {{1,2,1,3,3,2,3,2}, {5,2,1,2,1,1,2,3}, {1,1,2,4,1,2,3,3}},
  {{1,1,2,2,4,2,3,2}, {6,2,1,2,1,1,3,1}, {2,1,2,4,1,2,4,1}},
  {{1,2,1,3,3,3,3,1}, {4,1,2,1,2,1,1,5}, {1,1,2,4,1,3,3,2}},
  {{1,1,2,2,4,3,3,1}, {4,2,1,2,1,2,1,4}, {1,1,2,4,1,4,3,1}},
  {{1,1,2,2,5,1,4,1}, {6,1,2,1,2,1,3,1}, {1,2,1,5,1,1,4,2}},
  {{2,1,2,3,1,1,1,6}, {4,1,2,1,2,2,1,4}, {1,1,2,4,2,1,4,2}},
  {{3,1,2,3,1,1,2,4}, {5,1,2,1,2,2,2,2}, {1,2,1,5,1,2,4,1}},
  {{4,1,2,3,1,1,3,2}, {5,2,1,2,1,3,2,1}, {1,1,2,4,2,2,4,1}},
  {{2,1,2,3,1,2,1,5}, {4,1,2,1,2,3,1,3}, {1,1,2,5,1,1,3,3}},
  {{3,1,2,3,1,2,2,3}, {4,2,1,2,1,4,1,2}, {2,1,2,5,1,1,4,1}},
  {{4,1,2,3,1,2,3,1}, {4,1,2,1,2,4,1,2}, {1,1,2,5,1,2,3,2}},
  {{2,1,2,3,1,3,1,4}, {4,2,1,2,1,5,1,1}, {1,1,2,5,1,3,3,1}},
  {{3,1,2,3,1,3,2,2}, {4,1,2,1,2,5,1,1}, {1,2,1,6,1,1,4,1}},
  {{2,1,2,3,1,4,1,3}, {3,2,1,2,2,1,1,5}, {1,1,2,5,2,1,4,1}},
  {{3,1,2,3,1,4,2,1}, {4,2,1,2,2,1,2,3}, {1,1,2,6,1,1,3,2}},
  {{2,1,2,3,1,5,1,2}, {5,2,1,2,2,1,3,1}, {1,1,2,6,1,2,3,1}},
  {{2,1,2,3,1,6,1,1}, {3,1,2,1,3,1,1,5}, {1,3,1,1,1,1,4,5}},
  {{1,2,1,4,1,1,1,6}, {3,2,1,2,2,2,1,4}, {2,3,1,1,1,1,5,3}},
  {{2,2,1,4,1,1,2,4}, {4,2,1,2,2,2,2,2}, {3,3,1,1,1,1,6,1}},
  {{3,2,1,4,1,1,3,2}, {3,1,2,1,3,2,1,4}, {1,3,1,1,1,2,4,4}},
  {{1,1,2,3,2,1,1,6}, {4,1,2,1,3,2,2,2}, {2,3,1,1,1,2,5,2}},
  {{1,2,1,4,1,2,1,5}, {4,2,1,2,2,3,2,1}, {1,3,1,1,1,3,4,3}},
  {{2,2,1,4,1,2,2,3}, {3,1,2,1,3,3,1,3}, {2,3,1,1,1,3,5,1}},
  {{3,2,1,4,1,2,3,1}, {3,2,1,2,2,4,1,2}, {1,3,1,1,1,4,4,2}},
  {{1,1,2,3,2,2,1,5}, {3,1,2,1,3,4,1,2}, {1,3,1,1,1,5,4,1}},
  {{2,1,2,3,2,2,2,3}, {3,2,1,2,2,5,1,1}, {6,3,1,1,2,1,1,2}},
  {{3,1,2,3,2,2,3,1}, {3,1,2,1,3,5,1,1}, {1,3,1,1,2,1,5,3}},
  {{1,1,2,3,2,3,1,4}, {2,2,1,2,3,1,1,5}, {2,3,1,1,2,1,6,1}},
  {{1,2,1,4,1,4,1,3}, {3,2,1,2,3,1,2,3}, {6,3,1,1,2,2,1,1}},
  {{2,2,1,4,1,4,2,1}, {4,2,1,2,3,1,3,1}, {1,3,1,1,2,2,5,2}},
  {{1,1,2,3,2,4,1,3}, {2,1,2,1,4,1,1,5}, {1,3,1,1,2,3,5,1}},
  {{2,1,2,3,2,4,2,1}, {2,2,1,2,3,2,1,4}, {5,3,1,1,3,1,1,2}},
  {{1,1,2,3,2,5,1,2}, {3,2,1,2,3,2,2,2}, {1,3,1,1,3,1,6,1}},
  {{1,2,1,4,2,1,2,4}, {2,1,2,1,4,2,1,4}, {5,3,1,1,3,2,1,1}},
  {{2,2,1,4,2,1,3,2}, {3,1,2,1,4,2,2,2}, {4,3,1,1,4,1,1,2}},
  {{1,1,2,3,3,1,2,4}, {3,2,1,2,3,3,2,1}, {4,3,1,1,4,2,1,1}},
  {{1,2,1,4,2,2,2,3}, {2,1,2,1,4,3,1,3}, {3,3,1,1,5,1,1,2}},
  {{2,2,1,4,2,2,3,1}, {2,2,1,2,3,4,1,2}, {3,3,1,1,5,2,1,1}},
  {{1,1,2,3,3,2,2,3}, {2,1,2,1,4,4,1,2}, {2,3,1,1,6,1,1,2}},
  {{2,1,2,3,3,2,3,1}, {2,2,1,2,3,5,1,1}, {2,3,1,1,6,2,1,1}},
  {{1,1,2,3,3,3,2,2}, {2,1,2,1,4,5,1,1}, {1,2,2,1,1,1,3,6}},
  {{1,2,1,4,2,4,2,1}, {1,2,1,2,4,1,1,5}, {2,2,2,1,1,1,4,4}},
  {{1,1,2,3,3,4,2,1}, {2,2,1,2,4,1,2,3}, {3,2,2,1,1,1,5,2}},
  {{1,1,2,3,4,1,3,2}, {3,2,1,2,4,1,3,1}, {1,2,2,1,1,2,3,5}},
  {{1,1,2,3,4,2,3,1}, {1,1,2,1,5,1,1,5}, {2,2,2,1,1,2,4,3}},
  {{2,1,2,4,1,1,1,5}, {1,2,1,2,4,2,1,4}, {3,2,2,1,1,2,5,1}},
  {{3,1,2,4,1,1,2,3}, {2,2,1,2,4,2,2,2}, {1,2,2,1,1,3,3,4}},
  {{4,1,2,4,1,1,3,1}, {1,1,2,1,5,2,1,4}, {2,2,2,1,1,3,4,2}},
  {{2,1,2,4,1,2,1,4}, {2,1,2,1,5,2,2,2}, {1,2,2,1,1,4,3,3}},
  {{3,1,2,4,1,2,2,2}, {2,2,1,2,4,3,2,1}, {2,2,2,1,1,4,4,1}},
  {{2,1,2,4,1,3,1,3}, {1,1,2,1,5,3,1,3}, {1,2,2,1,1,5,3,2}},
  {{3,1,2,4,1,3,2,1}, {1,2,1,2,4,4,1,2}, {1,2,2,1,1,6,3,1}},
  {{2,1,2,4,1,4,1,2}, {1,1,2,1,5,4,1,2}, {1,3,1,2,1,1,4,4}},
  {{2,1,2,4,1,5,1,1}, {1,2,1,2,4,5,1,1}, {2,3,1,2,1,1,5,2}},
  {{1,2,1,5,1,1,1,5}, {1,2,1,2,5,1,2,3}, {1,2,2,1,2,1,4,4}},
  {{2,2,1,5,1,1,2,3}, {2,2,1,2,5,1,3,1}, {1,3,1,2,1,2,4,3}},
  {{3,2,1,5,1,1,3,1}, {1,1,2,1,6,1,2,3}, {2,3,1,2,1,2,5,1}},
  {{1,1,2,4,2,1,1,5}, {1,2,1,2,5,2,2,2}, {1,2,2,1,2,2,4,3}},
  {{1,2,1,5,1,2,1,4}, {1,1,2,1,6,2,2,2}, {2,2,2,1,2,2,5,1}},
  {{2,2,1,5,1,2,2,2}, {1,2,1,2,5,3,2,1}, {1,2,2,1,2,3,4,2}},
  {{1,1,2,4,2,2,1,4}, {1,1,2,1,6,3,2,1}, {1,3,1,2,1,4,4,1}},
  {{2,1,2,4,2,2,2,2}, {1,2,1,2,6,1,3,1}, {1,2,2,1,2,4,4,1}},
  {{2,2,1,5,1,3,2,1}, {5,1,2,2,1,1,1,4}, {6,3,1,2,2,1,1,1}},
  {{1,1,2,4,2,3,1,3}, {6,1,2,2,1,1,2,2}, {1,3,1,2,2,1,5,2}},
  {{1,2,1,5,1,4,1,2}, {1,1,2,2,1,1,6,3}, {6,2,2,1,3,1,1,1}},
  {{1,1,2,4,2,4,1,2}, {5,1,2,2,1,2,1,3}, {1,2,2,1,3,1,5,2}},
  {{1,2,1,5,1,5,1,1}, {6,1,2,2,1,2,2,1}, {1,3,1,2,2,2,5,1}},
  {{1,2,1,5,2,1,2,3}, {1,1,2,2,1,2,6,2}, {1,2,2,1,3,2,5,1}},
  {{1,1,2,4,3,1,2,3}, {5,1,2,2,1,3,1,2}, {5,3,1,2,3,1,1,1}},
  {{1,1,2,4,3,2,2,2}, {1,1,2,2,1,3,6,1}, {5,2,2,1,4,1,1,1}},
  {{1,1,2,4,3,3,2,1}, {5,1,2,2,1,4,1,1}, {4,3,1,2,4,1,1,1}},
  {{3,1,2,5,1,1,2,2}, {4,2,1,3,1,1,1,4}, {4,2,2,1,5,1,1,1}},
  {{3,1,2,5,1,2,2,1}, {5,2,1,3,1,1,2,2}, {3,3,1,2,5,1,1,1}},
  {{2,1,2,5,1,4,1,1}, {4,1,2,2,2,1,1,4}, {3,2,2,1,6,1,1,1}},
  {{2,2,1,6,1,1,2,2}, {4,2,1,3,1,2,1,3}, {2,3,1,2,6,1,1,1}},
  {{1,2,1,6,1,2,1,3}, {5,2,1,3,1,2,2,1}, {2,1,3,1,1,1,3,5}},
  {{1,1,2,5,2,2,1,3}, {4,1,2,2,2,2,1,3}, {3,1,3,1,1,1,4,3}},
  {{1,1,2,5,2,3,1,2}, {5,1,2,2,2,2,2,1}, {4,1,3,1,1,1,5,1}},
  {{1,1,2,5,2,4,1,1}, {4,1,2,2,2,3,1,2}, {1,1,3,1,1,2,2,6}},
  {{2,3,1,1,1,1,2,6}, {4,2,1,3,1,4,1,1}, {2,1,3,1,1,2,3,4}},
  {{3,3,1,1,1,1,3,4}, {4,1,2,2,2,4,1,1}, {3,1,3,1,1,2,4,2}},
  {{4,3,1,1,1,1,4,2}, {3,2,1,3,2,1,1,4}, {1,1,3,1,1,3,2,5}},
  {{2,3,1,1,1,2,2,5}, {4,2,1,3,2,1,2,2}, {2,1,3,1,1,3,3,3}},
  {{3,3,1,1,1,2,3,3}, {3,1,2,2,3,1,1,4}, {3,1,3,1,1,3,4,1}},
  {{1,3,1,1,1,3,1,6}, {3,2,1,3,2,2,1,3}, {1,1,3,1,1,4,2,4}},
  {{2,3,1,1,1,3,2,4}, {4,2,1,3,2,2,2,1}, {2,1,3,1,1,4,3,2}},
  {{3,3,1,1,1,3,3,2}, {3,1,2,2,3,2,1,3}, {1,1,3,1,1,5,2,3}},
  {{1,3,1,1,1,4,1,5}, {4,1,2,2,3,2,2,1}, {2,1,3,1,1,5,3,1}},
  {{2,3,1,1,1,4,2,3}, {3,1,2,2,3,3,1,2}, {1,1,3,1,1,6,2,2}},
  {{1,3,1,1,1,5,1,4}, {3,2,1,3,2,4,1,1}, {1,2,2,2,1,1,3,5}},
  {{1,3,1,1,1,6,1,3}, {3,1,2,2,3,4,1,1}, {2,2,2,2,1,1,4,3}},
  {{1,3,1,1,2,1,2,6}, {2,2,1,3,3,1,1,4}, {3,2,2,2,1,1,5,1}},
  {{2,3,1,1,2,1,3,4}, {3,2,1,3,3,1,2,2}, {1,1,3,1,2,1,3,5}},
  {{3,3,1,1,2,1,4,2}, {2,1,2,2,4,1,1,4}, {1,2,2,2,1,2,3,4}},
  {{1,3,1,1,2,2,2,5}, {2,2,1,3,3,2,1,3}, {2,2,2,2,1,2,4,2}},
  {{2,3,1,1,2,2,3,3}, {3,2,1,3,3,2,2,1}, {1,1,3,1,2,2,3,4}},
  {{3,3,1,1,2,2,4,1}, {2,1,2,2,4,2,1,3}, {2,1,3,1,2,2,4,2}},
  {{1,3,1,1,2,3,2,4}, {3,1,2,2,4,2,2,1}, {2,2,2,2,1,3,4,1}},
  {{2,3,1,1,2,3,3,2}, {2,1,2,2,4,3,1,2}, {1,1,3,1,2,3,3,3}},
  {{1,3,1,1,2,4,2,3}, {2,2,1,3,3,4,1,1}, {1,2,2,2,1,4,3,2}},
  {{1,3,1,1,2,5,2,2}, {2,1,2,2,4,4,1,1}, {1,1,3,1,2,4,3,2}},
  {{1,3,1,1,3,1,3,4}, {1,2,1,3,4,1,1,4}, {1,2,2,2,1,5,3,1}},
  {{2,3,1,1,3,1,4,2}, {2,2,1,3,4,1,2,2}, {1,1,3,1,2,5,3,1}},
  {{1,3,1,1,3,2,3,3}, {1,1,2,2,5,1,1,4}, {1,3,1,3,1,1,4,3}},
  {{2,3,1,1,3,2,4,1}, {1,2,1,3,4,2,1,3}, {2,3,1,3,1,1,5,1}},
  {{1,3,1,1,3,3,3,2}, {2,2,1,3,4,2,2,1}, {1,2,2,2,2,1,4,3}},
  {{1,3,1,1,4,1,4,2}, {1,1,2,2,5,2,1,3}, {1,3,1,3,1,2,4,2}},
  {{1,3,1,1,4,2,4,1}, {2,1,2,2,5,2,2,1}, {1,1,3,1,3,1,4,3}},
  {{3,2,2,1,1,1,2,5}, {1,1,2,2,5,3,1,2}, {1,2,2,2,2,2,4,2}},
  {{4,2,2,1,1,1,3,3}, {1,2,1,3,4,4,1,1}, {1,3,1,3,1,3,4,1}},
  {{5,2,2,1,1,1,4,1}, {1,1,2,2,5,4,1,1}, {1,1,3,1,3,2,4,2}},
  {{2,2,2,1,1,2,1,6}, {1,2,1,3,5,1,2,2}, {1,2,2,2,2,3,4,1}},
  {{3,2,2,1,1,2,2,4}, {1,1,2,2,6,1,2,2}, {1,1,3,1,3,3,4,1}},
  {{4,2,2,1,1,2,3,2}, {1,2,1,3,5,2,2,1}, {1,3,1,3,2,1,5,1}},
  {{2,2,2,1,1,3,1,5}, {1,1,2,2,6,2,2,1}, {1,2,2,2,3,1,5,1}},
  {{3,2,2,1,1,3,2,3}, {5,1,2,3,1,1,1,3}, {1,1,3,1,4,1,5,1}},
  {{4,2,2,1,1,3,3,1}, {6,1,2,3,1,1,2,1}, {1,1,3,2,1,1,2,6}},
  {{2,2,2,1,1,4,1,4}, {1,1,2,3,1,1,6,2}, {2,1,3,2,1,1,3,4}},
  {{3,2,2,1,1,4,2,2}, {5,1,2,3,1,2,1,2}, {3,1,3,2,1,1,4,2}},
  {{2,2,2,1,1,5,1,3}, {1,1,2,3,1,2,6,1}, {1,1,3,2,1,2,2,5}},
  {{3,2,2,1,1,5,2,1}, {5,1,2,3,1,3,1,1}, {2,1,3,2,1,2,3,3}},
  {{2,3,1,2,1,1,2,5}, {4,2,1,4,1,1,1,3}, {3,1,3,2,1,2,4,1}},
  {{3,3,1,2,1,1,3,3}, {5,2,1,4,1,1,2,1}, {1,1,3,2,1,3,2,4}},
  {{4,3,1,2,1,1,4,1}, {4,1,2,3,2,1,1,3}, {2,1,3,2,1,3,3,2}},
  {{2,2,2,1,2,1,2,5}, {5,1,2,3,2,1,2,1}, {1,1,3,2,1,4,2,3}},
  {{2,3,1,2,1,2,2,4}, {4,1,2,3,2,2,1,2}, {2,1,3,2,1,4,3,1}},
  {{3,3,1,2,1,2,3,2}, {4,2,1,4,1,3,1,1}, {1,1,3,2,1,5,2,2}},
  {{1,2,2,1,2,2,1,6}, {4,1,2,3,2,3,1,1}, {1,1,3,2,1,6,2,1}},
  {{1,3,1,2,1,3,1,5}, {3,2,1,4,2,1,1,3}, {1,2,2,3,1,1,3,4}},
  {{3,2,2,1,2,2,3,2}, {4,2,1,4,2,1,2,1}, {2,2,2,3,1,1,4,2}},
  {{3,3,1,2,1,3,3,1}, {3,1,2,3,3,1,1,3}, {1,1,3,2,2,1,3,4}},
  {{1,2,2,1,2,3,1,5}, {3,2,1,4,2,2,1,2}, {1,2,2,3,1,2,3,3}},
  {{2,2,2,1,2,3,2,3}, {3,1,2,3,3,2,1,2}, {2,2,2,3,1,2,4,1}},
  {{2,3,1,2,1,4,2,2}, {3,2,1,4,2,3,1,1}, {1,1,3,2,2,2,3,3}},
  {{1,2,2,1,2,4,1,4}, {3,1,2,3,3,3,1,1}, {2,1,3,2,2,2,4,1}},
  {{1,3,1,2,1,5,1,3}, {2,2,1,4,3,1,1,3}, {1,1,3,2,2,3,3,2}},
  {{1,2,2,1,2,5,1,3}, {3,2,1,4,3,1,2,1}, {1,2,2,3,1,4,3,1}},
  {{1,3,1,2,2,1,2,5}, {2,1,2,3,4,1,1,3}, {1,1,3,2,2,4,3,1}},
  {{2,3,1,2,2,1,3,3}, {3,1,2,3,4,1,2,1}, {1,3,1,4,1,1,4,2}},
  {{3,3,1,2,2,1,4,1}, {2,1,2,3,4,2,1,2}, {1,2,2,3,2,1,4,2}},
  {{1,2,2,1,3,1,2,5}, {2,2,1,4,3,3,1,1}, {1,3,1,4,1,2,4,1}},
  {{1,3,1,2,2,2,2,4}, {2,1,2,3,4,3,1,1}, {1,1,3,2,3,1,4,2}},
  {{3,2,2,1,3,1,4,1}, {1,2,1,4,4,1,1,3}, {1,2,2,3,2,2,4,1}},
  {{1,2,2,1,3,2,2,4}, {2,2,1,4,4,1,2,1}, {1,1,3,2,3,2,4,1}},
  {{2,2,2,1,3,2,3,2}, {1,1,2,3,5,1,1,3}, {1,1,3,3,1,1,2,5}},
  {{2,3,1,2,2,3,3,1}, {1,2,1,4,4,2,1,2}, {2,1,3,3,1,1,3,3}},
  {{1,2,2,1,3,3,2,3}, {1,1,2,3,5,2,1,2}, {3,1,3,3,1,1,4,1}},
  {{1,3,1,2,2,4,2,2}, {1,2,1,4,4,3,1,1}, {1,1,3,3,1,2,2,4}},
  {{1,2,2,1,3,4,2,2}, {1,1,2,3,5,3,1,1}, {2,1,3,3,1,2,3,2}},
  {{1,3,1,2,3,1,3,3}, {1,2,1,4,5,1,2,1}, {1,1,3,3,1,3,2,3}},
  {{2,3,1,2,3,1,4,1}, {1,1,2,3,6,1,2,1}, {2,1,3,3,1,3,3,1}},
  {{1,2,2,1,4,1,3,3}, {5,1,2,4,1,1,1,2}, {1,1,3,3,1,4,2,2}},
  {{1,3,1,2,3,2,3,2}, {1,1,2,4,1,1,6,1}, {1,1,3,3,1,5,2,1}},
  {{1,2,2,1,4,2,3,2}, {5,1,2,4,1,2,1,1}, {1,2,2,4,1,1,3,3}},
  {{1,3,1,2,3,3,3,1}, {4,2,1,5,1,1,1,2}, {2,2,2,4,1,1,4,1}},
  {{1,3,1,2,4,1,4,1}, {4,1,2,4,2,1,1,2}, {1,1,3,3,2,1,3,3}},
  {{1,2,2,1,5,1,4,1}, {4,2,1,5,1,2,1,1}, {1,2,2,4,1,2,3,2}},
  {{3,1,3,1,1,1,1,6}, {4,1,2,4,2,2,1,1}, {1,1,3,3,2,2,3,2}},
  {{4,1,3,1,1,1,2,4}, {3,2,1,5,2,1,1,2}, {1,2,2,4,1,3,3,1}},
  {{5,1,3,1,1,1,3,2}, {3,1,2,4,3,1,1,2}, {1,1,3,3,2,3,3,1}},
  {{3,1,3,1,1,2,1,5}, {3,2,1,5,2,2,1,1}, {1,3,1,5,1,1,4,1}},
  {{4,1,3,1,1,2,2,3}, {3,1,2,4,3,2,1,1}, {1,2,2,4,2,1,4,1}},
  {{5,1,3,1,1,2,3,1}, {2,2,1,5,3,1,1,2}, {1,1,3,3,3,1,4,1}},
  {{3,1,3,1,1,3,1,4}, {2,1,2,4,4,1,1,2}, {1,1,3,4,1,1,2,4}},
  {{4,1,3,1,1,3,2,2}, {2,2,1,5,3,2,1,1}, {2,1,3,4,1,1,3,2}},
  {{3,1,3,1,1,4,1,3}, {2,1,2,4,4,2,1,1}, {1,1,3,4,1,2,2,3}},
  {{4,1,3,1,1,4,2,1}, {1,2,1,5,4,1,1,2}, {2,1,3,4,1,2,3,1}},
  {{3,1,3,1,1,5,1,2}, {1,1,2,4,5,1,1,2}, {1,1,3,4,1,3,2,2}},
  {{2,2,2,2,1,1,1,6}, {1,2,1,5,4,2,1,1}, {1,1,3,4,1,4,2,1}},
  {{3,2,2,2,1,1,2,4}, {1,1,2,4,5,2,1,1}, {1,2,2,5,1,1,3,2}},
  {{4,2,2,2,1,1,3,2}, {5,1,2,5,1,1,1,1}, {1,1,3,4,2,1,3,2}},
  {{2,1,3,1,2,1,1,6}, {4,2,1,6,1,1,1,1}, {1,2,2,5,1,2,3,1}},
  {{2,2,2,2,1,2,1,5}, {4,1,2,5,2,1,1,1}, {1,1,3,4,2,2,3,1}},
  {{4,1,3,1,2,1,3,2}, {3,2,1,6,2,1,1,1}, {1,1,3,5,1,1,2,3}},
  {{4,2,2,2,1,2,3,1}, {3,1,2,5,3,1,1,1}, {2,1,3,5,1,1,3,1}},
  {{2,1,3,1,2,2,1,5}, {2,2,1,6,3,1,1,1}, {1,1,3,5,1,2,2,2}},
  {{3,1,3,1,2,2,2,3}, {2,1,2,5,4,1,1,1}, {1,1,3,5,1,3,2,1}},
  {{4,1,3,1,2,2,3,1}, {4,3,1,1,1,1,1,5}, {1,2,2,6,1,1,3,1}},
  {{2,1,3,1,2,3,1,4}, {5,3,1,1,1,1,2,3}, {1,1,3,5,2,1,3,1}},
  {{2,2,2,2,1,4,1,3}, {6,3,1,1,1,1,3,1}, {1,1,3,6,1,1,2,2}},
  {{3,2,2,2,1,4,2,1}, {4,3,1,1,1,2,1,4}, {1,1,3,6,1,2,2,1}},
  {{2,1,3,1,2,4,1,3}, {5,3,1,1,1,2,2,2}, {1,4,1,1,1,1,4,4}},
  {{3,1,3,1,2,4,2,1}, {4,3,1,1,1,3,1,3}, {2,4,1,1,1,1,5,2}},
  {{2,2,2,2,1,6,1,1}, {5,3,1,1,1,3,2,1}, {1,4,1,1,1,2,4,3}},
  {{1,3,1,3,1,1,1,6}, {4,3,1,1,1,4,1,2}, {2,4,1,1,1,2,5,1}},
  {{2,3,1,3,1,1,2,4}, {4,3,1,1,1,5,1,1}, {1,4,1,1,1,3,4,2}},
  {{3,3,1,3,1,1,3,2}, {3,3,1,1,2,1,1,5}, {1,4,1,1,1,4,4,1}},
  {{1,2,2,2,2,1,1,6}, {4,3,1,1,2,1,2,3}, {1,4,1,1,2,1,5,2}},
  {{1,3,1,3,1,2,1,5}, {5,3,1,1,2,1,3,1}, {1,4,1,1,2,2,5,1}},
  {{2,3,1,3,1,2,2,3}, {3,3,1,1,2,2,1,4}, {5,4,1,1,3,1,1,1}},
  {{3,3,1,3,1,2,3,1}, {4,3,1,1,2,2,2,2}, {4,4,1,1,4,1,1,1}},
  {{1,1,3,1,3,1,1,6}, {3,3,1,1,2,3,1,3}, {3,4,1,1,5,1,1,1}},
  {{1,2,2,2,2,2,1,5}, {4,3,1,1,2,3,2,1}, {2,4,1,1,6,1,1,1}},
  {{2,2,2,2,2,2,2,3}, {3,3,1,1,2,4,1,2}, {1,3,2,1,1,1,3,5}},
  {{3,2,2,2,2,2,3,1}, {3,3,1,1,2,5,1,1}, {2,3,2,1,1,1,4,3}},
  {{1,1,3,1,3,2,1,5}, {2,3,1,1,3,1,1,5}, {3,3,2,1,1,1,5,1}},
  {{2,1,3,1,3,2,2,3}, {3,3,1,1,3,1,2,3}, {1,3,2,1,1,2,3,4}},
  {{3,1,3,1,3,2,3,1}, {4,3,1,1,3,1,3,1}, {2,3,2,1,1,2,4,2}},
  {{2,3,1,3,1,4,2,1}, {2,3,1,1,3,2,1,4}, {1,3,2,1,1,3,3,3}},
  {{1,1,3,1,3,3,1,4}, {3,3,1,1,3,2,2,2}, {2,3,2,1,1,3,4,1}},
  {{1,2,2,2,2,4,1,3}, {2,3,1,1,3,3,1,3}, {1,3,2,1,1,4,3,2}},
  {{2,2,2,2,2,4,2,1}, {3,3,1,1,3,3,2,1}, {1,3,2,1,1,5,3,1}},
  {{1,1,3,1,3,4,1,3}, {2,3,1,1,3,4,1,2}, {1,4,1,2,1,1,4,3}},
  {{1,3,1,3,1,6,1,1}, {2,3,1,1,3,5,1,1}, {2,4,1,2,1,1,5,1}},
  {{1,3,1,3,2,1,2,4}, {1,3,1,1,4,1,1,5}, {1,3,2,1,2,1,4,3}},
  {{2,3,1,3,2,1,3,2}, {2,3,1,1,4,1,2,3}, {1,4,1,2,1,2,4,2}},
  {{1,2,2,2,3,1,2,4}, {3,3,1,1,4,1,3,1}, {1,3,2,1,2,2,4,2}},
  {{1,3,1,3,2,2,2,3}, {1,3,1,1,4,2,1,4}, {1,4,1,2,1,3,4,1}},
  {{2,3,1,3,2,2,3,1}, {2,3,1,1,4,2,2,2}, {1,3,2,1,2,3,4,1}},
  {{1,1,3,1,4,1,2,4}, {1,3,1,1,4,3,1,3}, {1,4,1,2,2,1,5,1}},
  {{1,2,2,2,3,2,2,3}, {2,3,1,1,4,3,2,1}, {1,3,2,1,3,1,5,1}},
  {{2,2,2,2,3,2,3,1}, {1,3,1,1,4,4,1,2}, {1,2,3,1,1,1,2,6}},
  {{1,1,3,1,4,2,2,3}, {1,3,1,1,4,5,1,1}, {2,2,3,1,1,1,3,4}},
  {{2,1,3,1,4,2,3,1}, {1,3,1,1,5,1,2,3}, {3,2,3,1,1,1,4,2}},
  {{1,3,1,3,2,4,2,1}, {2,3,1,1,5,1,3,1}, {1,2,3,1,1,2,2,5}},
  {{1,2,2,2,3,4,2,1}, {1,3,1,1,5,2,2,2}, {2,2,3,1,1,2,3,3}},
  {{1,3,1,3,3,1,3,2}, {1,3,1,1,5,3,2,1}, {3,2,3,1,1,2,4,1}},
  {{1,2,2,2,4,1,3,2}, {1,3,1,1,6,1,3,1}, {1,2,3,1,1,3,2,4}},
  {{1,3,1,3,3,2,3,1}, {5,2,2,1,1,1,1,4}, {2,2,3,1,1,3,3,2}},
  {{1,1,3,1,5,1,3,2}, {6,2,2,1,1,1,2,2}, {1,2,3,1,1,4,2,3}},
  {{1,2,2,2,4,2,3,1}, {1,2,2,1,1,1,6,3}, {2,2,3,1,1,4,3,1}},
  {{3,1,3,2,1,1,1,5}, {5,2,2,1,1,2,1,3}, {1,2,3,1,1,5,2,2}},
  {{4,1,3,2,1,1,2,3}, {6,2,2,1,1,2,2,1}, {1,2,3,1,1,6,2,1}},
  {{5,1,3,2,1,1,3,1}, {1,2,2,1,1,2,6,2}, {1,3,2,2,1,1,3,4}},
  {{3,1,3,2,1,2,1,4}, {5,2,2,1,1,3,1,2}, {2,3,2,2,1,1,4,2}},
  {{4,1,3,2,1,2,2,2}, {1,2,2,1,1,3,6,1}, {1,2,3,1,2,1,3,4}},
  {{3,1,3,2,1,3,1,3}, {5,2,2,1,1,4,1,1}, {1,3,2,2,1,2,3,3}},
  {{4,1,3,2,1,3,2,1}, {4,3,1,2,1,1,1,4}, {2,3,2,2,1,2,4,1}},
  {{3,1,3,2,1,4,1,2}, {5,3,1,2,1,1,2,2}, {1,2,3,1,2,2,3,3}},
  {{3,1,3,2,1,5,1,1}, {4,2,2,1,2,1,1,4}, {1,3,2,2,1,3,3,2}},
  {{2,2,2,3,1,1,1,5}, {4,3,1,2,1,2,1,3}, {1,2,3,1,2,3,3,2}},
  {{3,2,2,3,1,1,2,3}, {5,3,1,2,1,2,2,1}, {1,3,2,2,1,4,3,1}},
  {{4,2,2,3,1,1,3,1}, {4,2,2,1,2,2,1,3}, {1,2,3,1,2,4,3,1}},
  {{2,1,3,2,2,1,1,5}, {5,2,2,1,2,2,2,1}, {1,4,1,3,1,1,4,2}},
  {{2,2,2,3,1,2,1,4}, {4,2,2,1,2,3,1,2}, {1,3,2,2,2,1,4,2}},
  {{4,1,3,2,2,1,3,1}, {4,3,1,2,1,4,1,1}, {1,4,1,3,1,2,4,1}},
  {{2,1,3,2,2,2,1,4}, {4,2,2,1,2,4,1,1}, {1,2,3,1,3,1,4,2}},
  {{3,1,3,2,2,2,2,2}, {3,3,1,2,2,1,1,4}, {1,3,2,2,2,2,4,1}},
  {{3,2,2,3,1,3,2,1}, {4,3,1,2,2,1,2,2}, {1,2,3,1,3,2,4,1}},
  {{2,1,3,2,2,3,1,3}, {3,2,2,1,3,1,1,4}, {2,1,4,1,1,1,2,5}},
  {{2,2,2,3,1,4,1,2}, {3,3,1,2,2,2,1,3}, {3,1,4,1,1,1,3,3}},
  {{2,1,3,2,2,4,1,2}, {4,3,1,2,2,2,2,1}, {4,1,4,1,1,1,4,1}},
  {{2,2,2,3,1,5,1,1}, {3,2,2,1,3,2,1,3}, {1,1,4,1,1,2,1,6}},
  {{2,1,3,2,2,5,1,1}, {4,2,2,1,3,2,2,1}, {2,1,4,1,1,2,2,4}},
  {{1,3,1,4,1,1,1,5}, {3,2,2,1,3,3,1,2}, {3,1,4,1,1,2,3,2}},
  {{2,3,1,4,1,1,2,3}, {3,3,1,2,2,4,1,1}, {1,1,4,1,1,3,1,5}},
  {{3,3,1,4,1,1,3,1}, {3,2,2,1,3,4,1,1}, {2,1,4,1,1,3,2,3}},
  {{1,2,2,3,2,1,1,5}, {2,3,1,2,3,1,1,4}, {3,1,4,1,1,3,3,1}},
  {{1,3,1,4,1,2,1,4}, {3,3,1,2,3,1,2,2}, {1,1,4,1,1,4,1,4}},
  {{2,3,1,4,1,2,2,2}, {2,2,2,1,4,1,1,4}, {2,1,4,1,1,4,2,2}},
  {{1,1,3,2,3,1,1,5}, {2,3,1,2,3,2,1,3}, {1,1,4,1,1,5,1,3}},
  {{1,2,2,3,2,2,1,4}, {3,3,1,2,3,2,2,1}, {2,1,4,1,1,5,2,1}},
  {{2,2,2,3,2,2,2,2}, {2,2,2,1,4,2,1,3}, {1,1,4,1,1,6,1,2}},
  {{2,3,1,4,1,3,2,1}, {3,2,2,1,4,2,2,1}, {1,2,3,2,1,1,2,5}},
  {{1,1,3,2,3,2,1,4}, {2,2,2,1,4,3,1,2}, {2,2,3,2,1,1,3,3}},
  {{2,1,3,2,3,2,2,2}, {2,3,1,2,3,4,1,1}, {3,2,3,2,1,1,4,1}},
  {{1,3,1,4,1,4,1,2}, {2,2,2,1,4,4,1,1}, {1,1,4,1,2,1,2,5}},
  {{1,1,3,2,3,3,1,3}, {1,3,1,2,4,1,1,4}, {1,2,3,2,1,2,2,4}},
  {{1,2,2,3,2,4,1,2}, {2,3,1,2,4,1,2,2}, {2,2,3,2,1,2,3,2}},
  {{1,3,1,4,1,5,1,1}, {1,2,2,1,5,1,1,4}, {1,1,4,1,2,2,2,4}},
  {{1,2,2,3,2,5,1,1}, {1,3,1,2,4,2,1,3}, {2,1,4,1,2,2,3,2}},
  {{1,3,1,4,2,1,2,3}, {2,3,1,2,4,2,2,1}, {2,2,3,2,1,3,3,1}},
  {{2,3,1,4,2,1,3,1}, {1,2,2,1,5,2,1,3}, {1,1,4,1,2,3,2,3}},
  {{1,2,2,3,3,1,2,3}, {2,2,2,1,5,2,2,1}, {1,2,3,2,1,4,2,2}},
  {{1,3,1,4,2,2,2,2}, {1,2,2,1,5,3,1,2}, {1,1,4,1,2,4,2,2}},
  {{1,1,3,2,4,1,2,3}, {1,3,1,2,4,4,1,1}, {1,2,3,2,1,5,2,1}},
  {{1,2,2,3,3,2,2,2}, {1,2,2,1,5,4,1,1}, {1,1,4,1,2,5,2,1}},
  {{1,3,1,4,2,3,2,1}, {1,3,1,2,5,1,2,2}, {1,3,2,3,1,1,3,3}},
  {{1,1,3,2,4,2,2,2}, {1,2,2,1,6,1,2,2}, {2,3,2,3,1,1,4,1}},
  {{1,2,2,3,3,3,2,1}, {1,3,1,2,5,2,2,1}, {1,2,3,2,2,1,3,3}},
  {{1,3,1,4,3,1,3,1}, {1,2,2,1,6,2,2,1}, {1,3,2,3,1,2,3,2}},
  {{1,1,3,2,5,1,3,1}, {6,1,3,1,1,1,1,3}, {1,1,4,1,3,1,3,3}},
  {{3,1,3,3,1,1,1,4}, {1,1,3,1,1,1,5,4}, {1,2,3,2,2,2,3,2}},
  {{4,1,3,3,1,1,2,2}, {2,1,3,1,1,1,6,2}, {1,3,2,3,1,3,3,1}},
  {{3,1,3,3,1,2,1,3}, {6,1,3,1,1,2,1,2}, {1,1,4,1,3,2,3,2}},
  {{4,1,3,3,1,2,2,1}, {1,1,3,1,1,2,5,3}, {1,2,3,2,2,3,3,1}},
  {{3,1,3,3,1,3,1,2}, {2,1,3,1,1,2,6,1}, {1,1,4,1,3,3,3,1}},
  {{3,1,3,3,1,4,1,1}, {6,1,3,1,1,3,1,1}, {1,4,1,4,1,1,4,1}},
  {{2,2,2,4,1,1,1,4}, {1,1,3,1,1,3,5,2}, {1,3,2,3,2,1,4,1}},
  {{3,2,2,4,1,1,2,2}, {1,1,3,1,1,4,5,1}, {1,2,3,2,3,1,4,1}},
  {{2,1,3,3,2,1,1,4}, {5,2,2,2,1,1,1,3}, {1,1,4,1,4,1,4,1}},
  {{2,2,2,4,1,2,1,3}, {6,2,2,2,1,1,2,1}, {1,1,4,2,1,1,1,6}},
  {{3,2,2,4,1,2,2,1}, {1,2,2,2,1,1,6,2}, {2,1,4,2,1,1,2,4}},
  {{2,1,3,3,2,2,1,3}, {5,1,3,1,2,1,1,3}, {3,1,4,2,1,1,3,2}},
  {{3,1,3,3,2,2,2,1}, {6,1,3,1,2,1,2,1}, {1,1,4,2,1,2,1,5}},
  {{2,1,3,3,2,3,1,2}, {1,1,3,1,2,1,6,2}, {2,1,4,2,1,2,2,3}},
  {{2,2,2,4,1,4,1,1}, {1,2,2,2,1,2,6,1}, {3,1,4,2,1,2,3,1}},
  {{2,1,3,3,2,4,1,1}, {5,1,3,1,2,2,1,2}, {1,1,4,2,1,3,1,4}},
  {{1,3,1,5,1,1,1,4}, {5,2,2,2,1,3,1,1}, {2,1,4,2,1,3,2,2}},
  {{2,3,1,5,1,1,2,2}, {1,1,3,1,2,2,6,1}, {1,1,4,2,1,4,1,3}},
  {{1,2,2,4,2,1,1,4}, {5,1,3,1,2,3,1,1}, {2,1,4,2,1,4,2,1}},
  {{1,3,1,5,1,2,1,3}, {4,3,1,3,1,1,1,3}, {1,1,4,2,1,5,1,2}},
  {{2,3,1,5,1,2,2,1}, {5,3,1,3,1,1,2,1}, {1,1,4,2,1,6,1,1}},
  {{1,1,3,3,3,1,1,4}, {4,2,2,2,2,1,1,3}, {1,2,3,3,1,1,2,4}},
  {{1,2,2,4,2,2,1,3}, {4,3,1,3,1,2,1,2}, {2,2,3,3,1,1,3,2}},
  {{2,2,2,4,2,2,2,1}, {4,1,3,1,3,1,1,3}, {1,1,4,2,2,1,2,4}},
  {{1,1,3,3,3,2,1,3}, {5,1,3,1,3,1,2,1}, {1,2,3,3,1,2,2,3}},
  {{2,1,3,3,3,2,2,1}, {4,3,1,3,1,3,1,1}, {2,2,3,3,1,2,3,1}},
  {{1,3,1,5,1,4,1,1}, {4,1,3,1,3,2,1,2}, {1,1,4,2,2,2,2,3}},
  {{1,1,3,3,3,3,1,2}, {4,2,2,2,2,3,1,1}, {2,1,4,2,2,2,3,1}},
  {{1,2,2,4,2,4,1,1}, {4,1,3,1,3,3,1,1}, {1,1,4,2,2,3,2,2}},
  {{1,1,3,3,3,4,1,1}, {3,3,1,3,2,1,1,3}, {1,2,3,3,1,4,2,1}},
  {{1,2,2,4,3,1,2,2}, {4,3,1,3,2,1,2,1}, {1,1,4,2,2,4,2,1}},
  {{1,1,3,3,4,1,2,2}, {3,2,2,2,3,1,1,3}, {1,3,2,4,1,1,3,2}},
  {{1,1,3,3,4,2,2,1}, {3,3,1,3,2,2,1,2}, {1,2,3,3,2,1,3,2}},
  {{4,1,3,4,1,1,2,1}, {3,1,3,1,4,1,1,3}, {1,3,2,4,1,2,3,1}},
  {{3,1,3,4,1,3,1,1}, {3,2,2,2,3,2,1,2}, {1,1,4,2,3,1,3,2}},
  {{3,2,2,5,1,1,2,1}, {3,3,1,3,2,3,1,1}, {1,2,3,3,2,2,3,1}},
  {{2,2,2,5,1,2,1,2}, {3,1,3,1,4,2,1,2}, {1,1,4,2,3,2,3,1}},
  {{2,2,2,5,1,3,1,1}, {3,2,2,2,3,3,1,1}, {1,1,4,3,1,1,1,5}},
  {{1,3,1,6,1,1,1,3}, {3,1,3,1,4,3,1,1}, {2,1,4,3,1,1,2,3}},
  {{1,2,2,5,2,1,1,3}, {2,3,1,3,3,1,1,3}, {3,1,4,3,1,1,3,1}},
  {{1,1,3,4,3,1,1,3}, {3,3,1,3,3,1,2,1}, {1,1,4,3,1,2,1,4}},
  {{1,3,1,6,1,3,1,1}, {2,2,2,2,4,1,1,3}, {2,1,4,3,1,2,2,2}},
  {{1,2,2,5,2,3,1,1}, {2,3,1,3,3,2,1,2}, {1,1,4,3,1,3,1,3}},
  {{2,4,1,1,1,1,2,5}, {2,1,3,1,5,1,1,3}, {2,1,4,3,1,3,2,1}},
  {{1,4,1,1,1,2,1,6}, {2,2,2,2,4,2,1,2}, {1,1,4,3,1,4,1,2}},
  {{2,4,1,1,1,2,2,4}, {2,3,1,3,3,3,1,1}, {1,1,4,3,1,5,1,1}},
  {{1,4,1,1,1,3,1,5}, {2,1,3,1,5,2,1,2}, {1,2,3,4,1,1,2,3}},
  {{2,4,1,1,1,3,2,3}, {2,2,2,2,4,3,1,1}, {2,2,3,4,1,1,3,1}},
  {{3,4,1,1,1,3,3,1}, {2,1,3,1,5,3,1,1}, {1,1,4,3,2,1,2,3}},
  {{1,4,1,1,1,4,1,4}, {1,3,1,3,4,1,1,3}, {1,2,3,4,1,2,2,2}},
  {{2,4,1,1,1,4,2,2}, {2,3,1,3,4,1,2,1}, {1,1,4,3,2,2,2,2}},
  {{1,4,1,1,1,5,1,3}, {1,2,2,2,5,1,1,3}, {1,2,3,4,1,3,2,1}},
  {{2,4,1,1,1,5,2,1}, {1,3,1,3,4,2,1,2}, {1,1,4,3,2,3,2,1}},
  {{1,4,1,1,2,1,2,5}, {1,1,3,1,6,1,1,3}, {1,3,2,5,1,1,3,1}},
  {{2,4,1,1,2,1,3,3}, {1,2,2,2,5,2,1,2}, {1,2,3,4,2,1,3,1}},
  {{3,4,1,1,2,1,4,1}, {1,3,1,3,4,3,1,1}, {1,1,4,3,3,1,3,1}},
  {{1,4,1,1,2,2,2,4}, {1,1,3,1,6,2,1,2}, {1,1,4,4,1,1,1,4}},
  {{2,4,1,1,2,2,3,2}, {1,2,2,2,5,3,1,1}, {2,1,4,4,1,1,2,2}},
  {{1,4,1,1,2,3,2,3}, {1,1,3,1,6,3,1,1}, {1,1,4,4,1,2,1,3}},
  {{2,4,1,1,2,3,3,1}, {1,3,1,3,5,1,2,1}, {2,1,4,4,1,2,2,1}},
  {{1,4,1,1,2,4,2,2}, {1,2,2,2,6,1,2,1}, {1,1,4,4,1,3,1,2}},
  {{1,4,1,1,2,5,2,1}, {6,1,3,2,1,1,1,2}, {1,1,4,4,1,4,1,1}},
  {{1,4,1,1,3,1,3,3}, {1,1,3,2,1,1,5,3}, {1,2,3,5,1,1,2,2}},
  {{2,4,1,1,3,1,4,1}, {2,1,3,2,1,1,6,1}, {1,1,4,4,2,1,2,2}},
  {{1,4,1,1,3,2,3,2}, {6,1,3,2,1,2,1,1}, {1,2,3,5,1,2,2,1}},
  {{1,4,1,1,3,3,3,1}, {1,1,3,2,1,2,5,2}, {1,1,4,4,2,2,2,1}},
  {{1,4,1,1,4,1,4,1}, {1,1,3,2,1,3,5,1}, {1,1,4,5,1,1,1,3}},
  {{2,3,2,1,1,1,1,6}, {5,2,2,3,1,1,1,2}, {2,1,4,5,1,1,2,1}},
  {{3,3,2,1,1,1,2,4}, {1,2,2,3,1,1,6,1}, {1,1,4,5,1,2,1,2}},
  {{4,3,2,1,1,1,3,2}, {5,1,3,2,2,1,1,2}, {1,1,4,5,1,3,1,1}},
  {{2,3,2,1,1,2,1,5}, {5,2,2,3,1,2,1,1}, {1,2,3,6,1,1,2,1}},
  {{3,3,2,1,1,2,2,3}, {1,1,3,2,2,1,6,1}, {1,1,4,5,2,1,2,1}},
  {{2,3,2,1,1,3,1,4}, {5,1,3,2,2,2,1,1}, {1,5,1,1,1,1,4,3}},
  {{3,3,2,1,1,3,2,2}, {4,3,1,4,1,1,1,2}, {2,5,1,1,1,1,5,1}},
  {{2,3,2,1,1,4,1,3}, {4,2,2,3,2,1,1,2}, {1,5,1,1,1,2,4,2}},
  {{3,3,2,1,1,4,2,1}, {4,3,1,4,1,2,1,1}, {1,5,1,1,1,3,4,1}},
  {{2,3,2,1,1,5,1,2}, {4,1,3,2,3,1,1,2}, {1,5,1,1,2,1,5,1}},
  {{1,4,1,2,1,1,1,6}, {4,2,2,3,2,2,1,1}, {1,4,2,1,1,1,3,4}},
  {{2,4,1,2,1,1,2,4}, {4,1,3,2,3,2,1,1}, {2,4,2,1,1,1,4,2}},
  {{3,4,1,2,1,1,3,2}, {3,3,1,4,2,1,1,2}, {1,4,2,1,1,2,3,3}},
  {{1,3,2,1,2,1,1,6}, {3,2,2,3,3,1,1,2}, {2,4,2,1,1,2,4,1}},
  {{1,4,1,2,1,2,1,5}, {3,3,1,4,2,2,1,1}, {1,4,2,1,1,3,3,2}},
  {{3,3,2,1,2,1,3,2}, {3,1,3,2,4,1,1,2}, {1,4,2,1,1,4,3,1}},
  {{3,4,1,2,1,2,3,1}, {3,2,2,3,3,2,1,1}, {1,5,1,2,1,1,4,2}},
  {{1,3,2,1,2,2,1,5}, {3,1,3,2,4,2,1,1}, {1,4,2,1,2,1,4,2}},
  {{2,3,2,1,2,2,2,3}, {2,3,1,4,3,1,1,2}, {1,5,1,2,1,2,4,1}},
  {{3,3,2,1,2,2,3,1}, {2,2,2,3,4,1,1,2}, {1,4,2,1,2,2,4,1}},
  {{1,3,2,1,2,3,1,4}, {2,3,1,4,3,2,1,1}, {1,3,3,1,1,1,2,5}},
  {{1,4,1,2,1,4,1,3}, {2,1,3,2,5,1,1,2}, {2,3,3,1,1,1,3,3}},
  {{2,4,1,2,1,4,2,1}, {2,2,2,3,4,2,1,1}, {3,3,3,1,1,1,4,1}},
  {{1,3,2,1,2,4,1,3}, {2,1,3,2,5,2,1,1}, {1,3,3,1,1,2,2,4}},
  {{2,3,2,1,2,4,2,1}, {1,3,1,4,4,1,1,2}, {2,3,3,1,1,2,3,2}},
  {{1,4,1,2,1,6,1,1}, {1,2,2,3,5,1,1,2}, {1,3,3,1,1,3,2,3}},
  {{1,4,1,2,2,1,2,4}, {1,3,1,4,4,2,1,1}, {2,3,3,1,1,3,3,1}},
  {{2,4,1,2,2,1,3,2}, {1,1,3,2,6,1,1,2}, {1,3,3,1,1,4,2,2}},
  {{1,3,2,1,3,1,2,4}, {1,2,2,3,5,2,1,1}, {1,3,3,1,1,5,2,1}},
  {{1,4,1,2,2,2,2,3}, {1,1,3,2,6,2,1,1}, {1,4,2,2,1,1,3,3}},
  {{2,4,1,2,2,2,3,1}, {6,1,3,3,1,1,1,1}, {2,4,2,2,1,1,4,1}},
  {{1,3,2,1,3,2,2,3}, {1,1,3,3,1,1,5,2}, {1,3,3,1,2,1,3,3}},
  {{2,3,2,1,3,2,3,1}, {1,1,3,3,1,2,5,1}, {1,4,2,2,1,2,3,2}},
  {{1,3,2,1,3,3,2,2}, {5,2,2,4,1,1,1,1}, {1,3,3,1,2,2,3,2}},
  {{1,4,1,2,2,4,2,1}, {5,1,3,3,2,1,1,1}, {1,4,2,2,1,3,3,1}},
  {{1,4,1,2,3,1,3,2}, {4,3,1,5,1,1,1,1}, {1,3,3,1,2,3,3,1}},
  {{1,3,2,1,4,1,3,2}, {4,2,2,4,2,1,1,1}, {1,5,1,3,1,1,4,1}},
  {{1,4,1,2,3,2,3,1}, {4,1,3,3,3,1,1,1}, {1,4,2,2,2,1,4,1}},
  {{1,3,2,1,4,2,3,1}, {3,3,1,5,2,1,1,1}, {1,3,3,1,3,1,4,1}},
  {{3,2,3,1,1,1,1,5}, {3,2,2,4,3,1,1,1}, {1,2,4,1,1,1,1,6}},
  {{4,2,3,1,1,1,2,3}, {3,1,3,3,4,1,1,1}, {2,2,4,1,1,1,2,4}},
  {{5,2,3,1,1,1,3,1}, {2,3,1,5,3,1,1,1}, {3,2,4,1,1,1,3,2}},
  {{3,2,3,1,1,2,1,4}, {2,2,2,4,4,1,1,1}, {1,2,4,1,1,2,1,5}},
  {{4,2,3,1,1,2,2,2}, {2,1,3,3,5,1,1,1}, {2,2,4,1,1,2,2,3}},
  {{3,2,3,1,1,3,1,3}, {1,3,1,5,4,1,1,1}, {3,2,4,1,1,2,3,1}},
  {{4,2,3,1,1,3,2,1}, {1,2,2,4,5,1,1,1}, {1,2,4,1,1,3,1,4}},
  {{3,2,3,1,1,4,1,2}, {1,1,3,3,6,1,1,1}, {2,2,4,1,1,3,2,2}},
  {{3,2,3,1,1,5,1,1}, {1,1,3,4,1,1,5,1}, {1,2,4,1,1,4,1,3}},
  {{2,3,2,2,1,1,1,5}, {4,4,1,1,1,1,1,4}, {2,2,4,1,1,4,2,1}},
  {{3,3,2,2,1,1,2,3}, {5,4,1,1,1,1,2,2}, {1,2,4,1,1,5,1,2}},
  {{2,2,3,1,2,1,1,5}, {4,4,1,1,1,2,1,3}, {1,2,4,1,1,6,1,1}},
  {{2,3,2,2,1,2,1,4}, {5,4,1,1,1,2,2,1}, {1,3,3,2,1,1,2,4}},
  {{3,3,2,2,1,2,2,2}, {4,4,1,1,1,3,1,2}, {2,3,3,2,1,1,3,2}},
  {{2,2,3,1,2,2,1,4}, {4,4,1,1,1,4,1,1}, {1,2,4,1,2,1,2,4}},
  {{3,2,3,1,2,2,2,2}, {3,4,1,1,2,1,1,4}, {1,3,3,2,1,2,2,3}},
  {{3,3,2,2,1,3,2,1}, {4,4,1,1,2,1,2,2}, {2,3,3,2,1,2,3,1}},
  {{2,2,3,1,2,3,1,3}, {3,4,1,1,2,2,1,3}, {1,2,4,1,2,2,2,3}},
  {{2,3,2,2,1,4,1,2}, {4,4,1,1,2,2,2,1}, {2,2,4,1,2,2,3,1}},
  {{2,2,3,1,2,4,1,2}, {3,4,1,1,2,3,1,2}, {1,2,4,1,2,3,2,2}},
  {{2,3,2,2,1,5,1,1}, {3,4,1,1,2,4,1,1}, {1,3,3,2,1,4,2,1}},
  {{2,2,3,1,2,5,1,1}, {2,4,1,1,3,1,1,4}, {1,2,4,1,2,4,2,1}},
  {{1,4,1,3,1,1,1,5}, {3,4,1,1,3,1,2,2}, {1,4,2,3,1,1,3,2}},
  {{2,4,1,3,1,1,2,3}, {2,4,1,1,3,2,1,3}, {1,3,3,2,2,1,3,2}},
  {{1,3,2,2,2,1,1,5}, {3,4,1,1,3,2,2,1}, {1,4,2,3,1,2,3,1}},
  {{1,4,1,3,1,2,1,4}, {2,4,1,1,3,3,1,2}, {1,2,4,1,3,1,3,2}},
  {{3,3,2,2,2,1,3,1}, {2,4,1,1,3,4,1,1}, {1,3,3,2,2,2,3,1}},
  {{1,2,3,1,3,1,1,5}, {1,4,1,1,4,1,1,4}, {1,2,4,1,3,2,3,1}},
  {{1,3,2,2,2,2,1,4}, {2,4,1,1,4,1,2,2}, {2,1,5,1,1,1,1,5}},
  {{2,3,2,2,2,2,2,2}, {1,4,1,1,4,2,1,3}, {3,1,5,1,1,1,2,3}},
  {{2,4,1,3,1,3,2,1}, {2,4,1,1,4,2,2,1}, {4,1,5,1,1,1,3,1}},
  {{1,2,3,1,3,2,1,4}, {1,4,1,1,4,3,1,2}, {2,1,5,1,1,2,1,4}},
  {{2,2,3,1,3,2,2,2}, {1,4,1,1,4,4,1,1}, {3,1,5,1,1,2,2,2}},
  {{1,4,1,3,1,4,1,2}, {1,4,1,1,5,1,2,2}, {2,1,5,1,1,3,1,3}},
  {{1,2,3,1,3,3,1,3}, {1,4,1,1,5,2,2,1}, {3,1,5,1,1,3,2,1}},
  {{1,3,2,2,2,4,1,2}, {5,3,2,1,1,1,1,3}, {2,1,5,1,1,4,1,2}},
  {{1,4,1,3,1,5,1,1}, {6,3,2,1,1,1,2,1}, {2,1,5,1,1,5,1,1}},
  {{1,3,2,2,2,5,1,1}, {1,3,2,1,1,1,6,2}, {1,2,4,2,1,1,1,5}},
  {{1,4,1,3,2,1,2,3}, {5,3,2,1,1,2,1,2}, {2,2,4,2,1,1,2,3}},
  {{2,4,1,3,2,1,3,1}, {1,3,2,1,1,2,6,1}, {3,2,4,2,1,1,3,1}},
  {{1,3,2,2,3,1,2,3}, {5,3,2,1,1,3,1,1}, {1,1,5,1,2,1,1,5}},
  {{1,4,1,3,2,2,2,2}, {4,4,1,2,1,1,1,3}, {1,2,4,2,1,2,1,4}},
  {{1,2,3,1,4,1,2,3}, {5,4,1,2,1,1,2,1}, {2,2,4,2,1,2,2,2}},
  {{1,3,2,2,3,2,2,2}, {4,3,2,1,2,1,1,3}, {1,1,5,1,2,2,1,4}},
  {{1,4,1,3,2,3,2,1}, {4,4,1,2,1,2,1,2}, {2,1,5,1,2,2,2,2}},
  {{1,2,3,1,4,2,2,2}, {4,3,2,1,2,2,1,2}, {2,2,4,2,1,3,2,1}},
  {{1,3,2,2,3,3,2,1}, {4,4,1,2,1,3,1,1}, {1,1,5,1,2,3,1,3}},
  {{1,4,1,3,3,1,3,1}, {4,3,2,1,2,3,1,1}, {1,2,4,2,1,4,1,2}},
  {{1,3,2,2,4,1,3,1}, {3,4,1,2,2,1,1,3}, {1,1,5,1,2,4,1,2}},
  {{1,2,3,1,5,1,3,1}, {4,4,1,2,2,1,2,1}, {1,2,4,2,1,5,1,1}},
  {{4,1,4,1,1,1,1,4}, {3,3,2,1,3,1,1,3}, {1,1,5,1,2,5,1,1}},
  {{5,1,4,1,1,1,2,2}, {3,4,1,2,2,2,1,2}, {1,3,3,3,1,1,2,3}},
  {{4,1,4,1,1,2,1,3}, {3,3,2,1,3,2,1,2}, {2,3,3,3,1,1,3,1}},
  {{5,1,4,1,1,2,2,1}, {3,4,1,2,2,3,1,1}, {1,2,4,2,2,1,2,3}},
  {{4,1,4,1,1,3,1,2}, {3,3,2,1,3,3,1,1}, {1,3,3,3,1,2,2,2}},
  {{4,1,4,1,1,4,1,1}, {2,4,1,2,3,1,1,3}, {1,1,5,1,3,1,2,3}},
  {{3,2,3,2,1,1,1,4}, {3,4,1,2,3,1,2,1}, {1,2,4,2,2,2,2,2}},
  {{4,2,3,2,1,1,2,2}, {2,3,2,1,4,1,1,3}, {1,3,3,3,1,3,2,1}},
  {{3,1,4,1,2,1,1,4}, {2,4,1,2,3,2,1,2}, {1,1,5,1,3,2,2,2}},
  {{4,1,4,1,2,1,2,2}, {2,3,2,1,4,2,1,2}, {1,2,4,2,2,3,2,1}},
  {{4,2,3,2,1,2,2,1}, {2,4,1,2,3,3,1,1}, {1,1,5,1,3,3,2,1}},
  {{3,1,4,1,2,2,1,3}, {2,3,2,1,4,3,1,1}, {1,4,2,4,1,1,3,1}},
  {{4,1,4,1,2,2,2,1}, {1,4,1,2,4,1,1,3}, {1,3,3,3,2,1,3,1}},
  {{3,1,4,1,2,3,1,2}, {2,4,1,2,4,1,2,1}, {1,2,4,2,3,1,3,1}},
  {{3,2,3,2,1,4,1,1}, {1,3,2,1,5,1,1,3}, {1,1,5,1,4,1,3,1}},
  {{3,1,4,1,2,4,1,1}, {1,4,1,2,4,2,1,2}, {2,1,5,2,1,1,1,4}},
  {{2,3,2,3,1,1,1,4}, {1,3,2,1,5,2,1,2}, {3,1,5,2,1,1,2,2}},
  {{3,3,2,3,1,1,2,2}, {1,4,1,2,4,3,1,1}, {2,1,5,2,1,2,1,3}},
  {{2,2,3,2,2,1,1,4}, {1,3,2,1,5,3,1,1}, {3,1,5,2,1,2,2,1}},
  {{2,3,2,3,1,2,1,3}, {1,4,1,2,5,1,2,1}, {2,1,5,2,1,3,1,2}},
  {{3,3,2,3,1,2,2,1}, {1,3,2,1,6,1,2,1}, {2,1,5,2,1,4,1,1}},
  {{2,1,4,1,3,1,1,4}, {6,2,3,1,1,1,1,2}, {1,2,4,3,1,1,1,4}},
  {{2,2,3,2,2,2,1,3}, {1,2,3,1,1,1,5,3}, {2,2,4,3,1,1,2,2}},
  {{3,2,3,2,2,2,2,1}, {2,2,3,1,1,1,6,1}, {1,1,5,2,2,1,1,4}},
  {{2,1,4,1,3,2,1,3}, {6,2,3,1,1,2,1,1}, {1,2,4,3,1,2,1,3}},
  {{3,1,4,1,3,2,2,1}, {1,2,3,1,1,2,5,2}, {2,2,4,3,1,2,2,1}},
  {{2,3,2,3,1,4,1,1}, {1,2,3,1,1,3,5,1}, {1,1,5,2,2,2,1,3}},
  {{2,1,4,1,3,3,1,2}, {5,3,2,2,1,1,1,2}, {2,1,5,2,2,2,2,1}},
  {{2,2,3,2,2,4,1,1}, {1,3,2,2,1,1,6,1}, {1,1,5,2,2,3,1,2}},
  {{2,1,4,1,3,4,1,1}, {5,2,3,1,2,1,1,2}, {1,2,4,3,1,4,1,1}},
  {{1,4,1,4,1,1,1,4}, {5,3,2,2,1,2,1,1}, {1,1,5,2,2,4,1,1}},
  {{2,4,1,4,1,1,2,2}, {1,2,3,1,2,1,6,1}, {1,3,3,4,1,1,2,2}},
  {{1,3,2,3,2,1,1,4}, {5,2,3,1,2,2,1,1}, {1,2,4,3,2,1,2,2}},
  {{1,4,1,4,1,2,1,3}, {4,4,1,3,1,1,1,2}, {1,3,3,4,1,2,2,1}},
  {{2,4,1,4,1,2,2,1}, {4,3,2,2,2,1,1,2}, {1,1,5,2,3,1,2,2}},
  {{1,2,3,2,3,1,1,4}, {4,4,1,3,1,2,1,1}, {1,2,4,3,2,2,2,1}},
  {{1,3,2,3,2,2,1,3}, {4,2,3,1,3,1,1,2}, {1,1,5,2,3,2,2,1}},
  {{2,3,2,3,2,2,2,1}, {4,3,2,2,2,2,1,1}, {2,1,5,3,1,1,1,3}},
  {{1,1,4,1,4,1,1,4}, {4,2,3,1,3,2,1,1}, {3,1,5,3,1,1,2,1}},
  {{1,2,3,2,3,2,1,3}, {3,4,1,3,2,1,1,2}, {2,1,5,3,1,2,1,2}},
  {{2,2,3,2,3,2,2,1}, {3,3,2,2,3,1,1,2}, {2,1,5,3,1,3,1,1}},
  {{1,4,1,4,1,4,1,1}, {3,4,1,3,2,2,1,1}, {1,2,4,4,1,1,1,3}},
  {{1,1,4,1,4,2,1,3}, {3,2,3,1,4,1,1,2}, {2,2,4,4,1,1,2,1}},
  {{2,1,4,1,4,2,2,1}, {3,3,2,2,3,2,1,1}, {1,1,5,3,2,1,1,3}},
  {{1,3,2,3,2,4,1,1}, {3,2,3,1,4,2,1,1}, {1,2,4,4,1,2,1,2}},
  {{1,1,4,1,4,3,1,2}, {2,4,1,3,3,1,1,2}, {1,1,5,3,2,2,1,2}},
  {{1,4,1,4,2,1,2,2}, {2,3,2,2,4,1,1,2}, {1,2,4,4,1,3,1,1}},
  {{1,3,2,3,3,1,2,2}, {2,4,1,3,3,2,1,1}, {1,1,5,3,2,3,1,1}},
  {{1,4,1,4,2,2,2,1}, {2,2,3,1,5,1,1,2}, {1,3,3,5,1,1,2,1}},
  {{1,2,3,2,4,1,2,2}, {2,3,2,2,4,2,1,1}, {1,2,4,4,2,1,2,1}},
  {{1,3,2,3,3,2,2,1}, {2,2,3,1,5,2,1,1}, {1,1,5,3,3,1,2,1}},
  {{1,1,4,1,5,1,2,2}, {1,4,1,3,4,1,1,2}, {2,1,5,4,1,1,1,2}},
  {{1,2,3,2,4,2,2,1}, {1,3,2,2,5,1,1,2}, {2,1,5,4,1,2,1,1}},
  {{1,1,4,1,5,2,2,1}, {1,4,1,3,4,2,1,1}, {1,2,4,5,1,1,1,2}},
  {{4,1,4,2,1,1,1,3}, {1,2,3,1,6,1,1,2}, {1,1,5,4,2,1,1,2}},
  {{5,1,4,2,1,1,2,1}, {1,3,2,2,5,2,1,1}, {1,2,4,5,1,2,1,1}},
  {{4,1,4,2,1,2,1,2}, {1,2,3,1,6,2,1,1}, {1,1,5,4,2,2,1,1}},
  {{4,1,4,2,1,3,1,1}, {1,1,4,1,1,1,4,4}, {1,6,1,1,1,1,4,2}},
  {{3,2,3,3,1,1,1,3}, {2,1,4,1,1,1,5,2}, {1,6,1,1,1,2,4,1}},
  {{4,2,3,3,1,1,2,1}, {1,1,4,1,1,2,4,3}, {1,5,2,1,1,1,3,3}},
  {{3,1,4,2,2,1,1,3}, {2,1,4,1,1,2,5,1}, {2,5,2,1,1,1,4,1}},
  {{4,1,4,2,2,1,2,1}, {1,1,4,1,1,3,4,2}, {1,5,2,1,1,2,3,2}},
  {{3,1,4,2,2,2,1,2}, {1,1,4,1,1,4,4,1}, {1,5,2,1,1,3,3,1}},
  {{3,2,3,3,1,3,1,1}, {6,2,3,2,1,1,1,1}, {1,6,1,2,1,1,4,1}},
  {{3,1,4,2,2,3,1,1}, {1,2,3,2,1,1,5,2}, {1,5,2,1,2,1,4,1}},
  {{2,3,2,4,1,1,1,3}, {6,1,4,1,2,1,1,1}, {1,4,3,1,1,1,2,4}},
  {{3,3,2,4,1,1,2,1}, {1,1,4,1,2,1,5,2}, {2,4,3,1,1,1,3,2}},
  {{2,2,3,3,2,1,1,3}, {1,2,3,2,1,2,5,1}, {1,4,3,1,1,2,2,3}},
  {{2,3,2,4,1,2,1,2}, {1,1,4,1,2,2,5,1}, {2,4,3,1,1,2,3,1}},
  {{2,1,4,2,3,1,1,3}, {5,3,2,3,1,1,1,1}, {1,4,3,1,1,3,2,2}},
  {{2,2,3,3,2,2,1,2}, {5,2,3,2,2,1,1,1}, {1,4,3,1,1,4,2,1}},
  {{2,3,2,4,1,3,1,1}, {5,1,4,1,3,1,1,1}, {1,5,2,2,1,1,3,2}},
  {{2,1,4,2,3,2,1,2}, {4,4,1,4,1,1,1,1}, {1,4,3,1,2,1,3,2}},
  {{2,2,3,3,2,3,1,1}, {4,3,2,3,2,1,1,1}, {1,5,2,2,1,2,3,1}},
  {{2,1,4,2,3,3,1,1}, {4,2,3,2,3,1,1,1}, {1,4,3,1,2,2,3,1}},
  {{1,4,1,5,1,1,1,3}, {4,1,4,1,4,1,1,1}, {1,3,4,1,1,1,1,5}},
  {{2,4,1,5,1,1,2,1}, {3,4,1,4,2,1,1,1}, {2,3,4,1,1,1,2,3}},
  {{1,3,2,4,2,1,1,3}, {3,3,2,3,3,1,1,1}, {3,3,4,1,1,1,3,1}},
  {{2,3,2,4,2,1,2,1}, {3,2,3,2,4,1,1,1}, {1,3,4,1,1,2,1,4}},
  {{1,2,3,3,3,1,1,3}, {3,1,4,1,5,1,1,1}, {2,3,4,1,1,2,2,2}},
  {{1,3,2,4,2,2,1,2}, {2,4,1,4,3,1,1,1}, {1,3,4,1,1,3,1,3}},
  {{1,4,1,5,1,3,1,1}, {2,3,2,3,4,1,1,1}, {2,3,4,1,1,3,2,1}},
  {{1,1,4,2,4,1,1,3}, {2,2,3,2,5,1,1,1}, {1,3,4,1,1,4,1,2}},
  {{1,2,3,3,3,2,1,2}, {2,1,4,1,6,1,1,1}, {1,3,4,1,1,5,1,1}},
  {{1,3,2,4,2,3,1,1}, {1,4,1,4,4,1,1,1}, {1,4,3,2,1,1,2,3}},
  {{1,1,4,2,4,2,1,2}, {1,3,2,3,5,1,1,1}, {2,4,3,2,1,1,3,1}},
  {{1,2,3,3,3,3,1,1}, {1,2,3,2,6,1,1,1}, {1,3,4,1,2,1,2,3}},
  {{1,1,4,2,4,3,1,1}, {1,1,4,2,1,1,4,3}, {2,3,4,1,2,1,3,1}},
  {{1,3,2,4,3,1,2,1}, {2,1,4,2,1,1,5,1}, {1,3,4,1,2,2,2,2}},
  {{1,1,4,2,5,1,2,1}, {1,1,4,2,1,2,4,2}, {1,4,3,2,1,3,2,1}},
  {{4,1,4,3,1,2,1,1}, {1,1,4,2,1,3,4,1}, {1,3,4,1,2,3,2,1}},
  {{3,1,4,3,2,1,1,2}, {1,2,3,3,1,1,5,1}, {1,5,2,3,1,1,3,1}},
  {{3,1,4,3,2,2,1,1}, {1,1,4,2,2,1,5,1}, {1,4,3,2,2,1,3,1}},
  {{2,2,3,4,2,1,1,2}, {1,1,4,3,1,1,4,2}, {1,3,4,1,3,1,3,1}},
  {{2,1,4,3,3,1,1,2}, {1,1,4,3,1,2,4,1}, {2,2,5,1,1,1,1,4}},
  {{2,1,4,3,3,2,1,1}, {1,1,4,4,1,1,4,1}, {3,2,5,1,1,1,2,2}},
  {{1,3,2,5,2,1,1,2}, {4,5,1,1,1,1,1,3}, {2,2,5,1,1,2,1,3}},
  {{1,2,3,4,3,1,1,2}, {4,5,1,1,1,2,1,2}, {3,2,5,1,1,2,2,1}},
  {{1,1,4,3,4,1,1,2}, {4,5,1,1,1,3,1,1}, {2,2,5,1,1,3,1,2}},
  {{1,1,4,3,4,2,1,1}, {3,5,1,1,2,1,1,3}, {2,2,5,1,1,4,1,1}},
  {{1,5,1,1,1,1,1,6}, {4,5,1,1,2,1,2,1}, {1,3,4,2,1,1,1,4}},
  {{1,5,1,1,1,2,1,5}, {3,5,1,1,2,2,1,2}, {2,3,4,2,1,1,2,2}},
  {{2,5,1,1,1,2,2,3}, {3,5,1,1,2,3,1,1}, {1,2,5,1,2,1,1,4}},
  {{1,5,1,1,1,3,1,4}, {2,5,1,1,3,1,1,3}, {2,2,5,1,2,1,2,2}},
  {{1,5,1,1,1,4,1,3}, {3,5,1,1,3,1,2,1}, {2,3,4,2,1,2,2,1}},
  {{1,5,1,1,1,5,1,2}, {2,5,1,1,3,2,1,2}, {1,2,5,1,2,2,1,3}},
  {{1,5,1,1,2,1,2,4}, {2,5,1,1,3,3,1,1}, {1,3,4,2,1,3,1,2}},
  {{1,5,1,1,2,2,2,3}, {1,5,1,1,4,1,1,3}, {1,2,5,1,2,3,1,2}},
  {{1,5,1,1,2,3,2,2}, {2,5,1,1,4,1,2,1}, {1,3,4,2,1,4,1,1}},
  {{1,5,1,1,2,4,2,1}, {1,5,1,1,4,2,1,2}, {1,2,5,1,2,4,1,1}},
  {{1,5,1,1,3,1,3,2}, {1,5,1,1,4,3,1,1}, {1,4,3,3,1,1,2,2}},
  {{1,5,1,1,3,2,3,1}, {1,5,1,1,5,1,2,1}, {1,3,4,2,2,1,2,2}},
  {{2,4,2,1,1,1,1,5}, {5,4,2,1,1,1,1,2}, {1,4,3,3,1,2,2,1}},
  {{2,4,2,1,1,2,1,4}, {1,4,2,1,1,1,6,1}, {1,2,5,1,3,1,2,2}},
  {{3,4,2,1,1,2,2,2}, {5,4,2,1,1,2,1,1}, {1,3,4,2,2,2,2,1}},
  {{2,4,2,1,1,3,1,3}, {4,5,1,2,1,1,1,2}, {1,2,5,1,3,2,2,1}},
  {{3,4,2,1,1,3,2,1}, {4,4,2,1,2,1,1,2}, {3,1,6,1,1,1,1,3}},
  {{2,4,2,1,1,4,1,2}, {4,5,1,2,1,2,1,1}, {4,1,6,1,1,1,2,1}},
  {{2,4,2,1,1,5,1,1}, {4,4,2,1,2,2,1,1}, {3,1,6,1,1,2,1,2}},
  {{1,5,1,2,1,1,1,5}, {3,5,1,2,2,1,1,2}, {3,1,6,1,1,3,1,1}},
  {{2,5,1,2,1,1,2,3}, {3,4,2,1,3,1,1,2}, {2,2,5,2,1,1,1,3}},
  {{1,4,2,1,2,1,1,5}, {3,5,1,2,2,2,1,1}, {3,2,5,2,1,1,2,1}},
  {{2,4,2,1,2,1,2,3}, {3,4,2,1,3,2,1,1}, {2,1,6,1,2,1,1,3}},
  {{2,5,1,2,1,2,2,2}, {2,5,1,2,3,1,1,2}, {2,2,5,2,1,2,1,2}},
  {{1,4,2,1,2,2,1,4}, {2,4,2,1,4,1,1,2}, {2,1,6,1,2,2,1,2}},
  {{2,4,2,1,2,2,2,2}, {2,5,1,2,3,2,1,1}, {2,2,5,2,1,3,1,1}},
  {{1,4,2,1,2,3,1,3}, {2,4,2,1,4,2,1,1}, {2,1,6,1,2,3,1,1}},
  {{2,4,2,1,2,3,2,1}, {1,5,1,2,4,1,1,2}, {1,3,4,3,1,1,1,3}},
  {{1,4,2,1,2,4,1,2}, {1,4,2,1,5,1,1,2}, {2,3,4,3,1,1,2,1}},
  {{1,5,1,2,1,5,1,1}, {1,5,1,2,4,2,1,1}, {1,2,5,2,2,1,1,3}},
  {{1,4,2,1,2,5,1,1}, {1,4,2,1,5,2,1,1}, {1,3,4,3,1,2,1,2}},
  {{1,5,1,2,2,1,2,3}, {6,3,3,1,1,1,1,1}, {1,1,6,1,3,1,1,3}},
  {{2,5,1,2,2,1,3,1}, {1,3,3,1,1,1,5,2}, {1,2,5,2,2,2,1,2}},
  {{1,4,2,1,3,1,2,3}, {1,3,3,1,1,2,5,1}, {1,3,4,3,1,3,1,1}},
  {{2,4,2,1,3,1,3,1}, {5,4,2,2,1,1,1,1}, {1,1,6,1,3,2,1,2}},
  {{1,4,2,1,3,2,2,2}, {5,3,3,1,2,1,1,1}, {1,2,5,2,2,3,1,1}},
  {{1,5,1,2,2,3,2,1}, {4,5,1,3,1,1,1,1}, {1,1,6,1,3,3,1,1}},
  {{1,4,2,1,3,3,2,1}, {4,4,2,2,2,1,1,1}, {1,4,3,4,1,1,2,1}},
  {{1,5,1,2,3,1,3,1}, {4,3,3,1,3,1,1,1}, {1,3,4,3,2,1,2,1}},
  {{1,4,2,1,4,1,3,1}, {3,5,1,3,2,1,1,1}, {1,2,5,2,3,1,2,1}},
  {{3,3,3,1,1,1,1,4}, {3,4,2,2,3,1,1,1}, {1,1,6,1,4,1,2,1}},
  {{3,3,3,1,1,2,1,3}, {3,3,3,1,4,1,1,1}, {3,1,6,2,1,1,1,2}},
  {{3,3,3,1,1,3,1,2}, {2,5,1,3,3,1,1,1}, {3,1,6,2,1,2,1,1}},
  {{3,3,3,1,1,4,1,1}, {2,4,2,2,4,1,1,1}, {2,2,5,3,1,1,1,2}},
  {{2,4,2,2,1,1,1,4}, {2,3,3,1,5,1,1,1}, {2,1,6,2,2,1,1,2}},
  {{2,3,3,1,2,1,1,4}, {1,5,1,3,4,1,1,1}, {2,2,5,3,1,2,1,1}},
  {{3,3,3,1,2,1,2,2}, {1,4,2,2,5,1,1,1}, {2,1,6,2,2,2,1,1}},
  {{3,4,2,2,1,2,2,1}, {1,3,3,1,6,1,1,1}, {1,3,4,4,1,1,1,2}},
  {{2,3,3,1,2,2,1,3}, {1,2,4,1,1,1,4,3}, {1,2,5,3,2,1,1,2}},
  {{3,3,3,1,2,2,2,1}, {2,2,4,1,1,1,5,1}, {1,3,4,4,1,2,1,1}},
  {{2,3,3,1,2,3,1,2}, {1,2,4,1,1,2,4,2}, {1,1,6,2,3,1,1,2}},
  {{2,4,2,2,1,4,1,1}, {1,2,4,1,1,3,4,1}, {1,2,5,3,2,2,1,1}},
  {{2,3,3,1,2,4,1,1}, {1,3,3,2,1,1,5,1}, {1,1,6,2,3,2,1,1}},
  {{1,5,1,3,1,1,1,4}, {1,2,4,1,2,1,5,1}, {3,1,6,3,1,1,1,1}},
  {{1,4,2,2,2,1,1,4}, {1,1,5,1,1,1,3,4}, {2,2,5,4,1,1,1,1}},
  {{1,5,1,3,1,2,1,3}, {2,1,5,1,1,1,4,2}, {2,1,6,3,2,1,1,1}},
  {{2,5,1,3,1,2,2,1}, {1,1,5,1,1,2,3,3}, {1,3,4,5,1,1,1,1}},
  {{1,3,3,1,3,1,1,4}, {2,1,5,1,1,2,4,1}, {1,2,5,4,2,1,1,1}},
  {{1,4,2,2,2,2,1,3}, {1,1,5,1,1,3,3,2}, {1,1,6,3,3,1,1,1}},
  {{1,5,1,3,1,3,1,2}, {1,1,5,1,1,4,3,1}, {1,6,2,1,1,1,3,2}},
  {{1,3,3,1,3,2,1,3}, {1,2,4,2,1,1,4,2}, {1,6,2,1,1,2,3,1}},
  {{1,4,2,2,2,3,1,2}, {1,1,5,1,2,1,4,2}, {1,5,3,1,1,1,2,3}},
  {{1,5,1,3,1,4,1,1}, {1,2,4,2,1,2,4,1}, {2,5,3,1,1,1,3,1}},
  {{1,3,3,1,3,3,1,2}, {1,1,5,1,2,2,4,1}, {1,5,3,1,1,2,2,2}},
  {{1,4,2,2,2,4,1,1}, {1,1,5,2,1,1,3,3}, {1,5,3,1,1,3,2,1}},
  {{1,5,1,3,2,1,2,2}, {2,1,5,2,1,1,4,1}, {1,6,2,2,1,1,3,1}},
  {{1,4,2,2,3,1,2,2}, {1,1,5,2,1,2,3,2}, {1,5,3,1,2,1,3,1}},
  {{1,5,1,3,2,2,2,1}, {1,1,5,2,1,3,3,1}, {1,4,4,1,1,1,1,4}},
  {{1,3,3,1,4,1,2,2}, {1,2,4,3,1,1,4,1}, {2,4,4,1,1,1,2,2}},
  {{1,4,2,2,3,2,2,1}, {1,1,5,2,2,1,4,1}, {1,4,4,1,1,2,1,3}},
  {{1,3,3,1,4,2,2,1}, {1,1,5,3,1,1,3,2}, {2,4,4,1,1,2,2,1}},
  {{4,2,4,1,1,1,1,3}, {1,1,5,3,1,2,3,1}, {1,4,4,1,1,3,1,2}},
  {{4,2,4,1,1,2,1,2}, {1,1,5,4,1,1,3,1}, {1,4,4,1,1,4,1,1}},
  {{4,2,4,1,1,3,1,1}, {3,6,1,1,2,1,1,2}, {1,5,3,2,1,1,2,2}},
  {{3,3,3,2,1,1,1,3}, {3,6,1,1,2,2,1,1}, {1,4,4,1,2,1,2,2}},
  {{3,2,4,1,2,1,1,3}, {2,6,1,1,3,1,1,2}, {1,5,3,2,1,2,2,1}},
  {{4,2,4,1,2,1,2,1}, {2,6,1,1,3,2,1,1}, {1,4,4,1,2,2,2,1}},
  {{3,2,4,1,2,2,1,2}, {1,6,1,1,4,1,1,2}, {2,3,5,1,1,1,1,3}},
  {{3,3,3,2,1,3,1,1}, {1,6,1,1,4,2,1,1}, {3,3,5,1,1,1,2,1}},
  {{3,2,4,1,2,3,1,1}, {4,5,2,1,2,1,1,1}, {2,3,5,1,1,2,1,2}},
  {{2,4,2,3,1,1,1,3}, {3,6,1,2,2,1,1,1}, {2,3,5,1,1,3,1,1}},
  {{3,4,2,3,1,1,2,1}, {3,5,2,1,3,1,1,1}, {1,4,4,2,1,1,1,3}},
  {{2,3,3,2,2,1,1,3}, {2,6,1,2,3,1,1,1}, {2,4,4,2,1,1,2,1}},
  {{3,3,3,2,2,1,2,1}, {2,5,2,1,4,1,1,1}, {1,3,5,1,2,1,1,3}},
  {{2,2,4,1,3,1,1,3}, {1,6,1,2,4,1,1,1}, {2,3,5,1,2,1,2,1}},
  {{2,3,3,2,2,2,1,2}, {1,5,2,1,5,1,1,1}, {1,3,5,1,2,2,1,2}},
  {{2,4,2,3,1,3,1,1}, {1,4,3,1,1,1,5,1}, {1,4,4,2,1,3,1,1}},
  {{2,2,4,1,3,2,1,2}, {1,3,4,1,1,1,4,2}, {1,3,5,1,2,3,1,1}},
  {{2,3,3,2,2,3,1,1}, {1,3,4,1,1,2,4,1}, {1,5,3,3,1,1,2,1}},
  {{2,2,4,1,3,3,1,1}, {1,2,5,1,1,1,3,3}, {1,4,4,2,2,1,2,1}},
  {{1,5,1,4,1,1,1,3}, {2,2,5,1,1,1,4,1}, {1,3,5,1,3,1,2,1}},
  {{2,5,1,4,1,1,2,1}, {1,2,5,1,1,2,3,2}, {3,2,6,1,1,1,1,2}},
  {{1,4,2,3,2,1,1,3}, {1,2,5,1,1,3,3,1}, {3,2,6,1,1,2,1,1}},
  {{2,4,2,3,2,1,2,1}, {1,3,4,2,1,1,4,1}, {2,3,5,2,1,1,1,2}},
  {{1,3,3,2,3,1,1,3}, {1,2,5,1,2,1,4,1}, {2,2,6,1,2,1,1,2}},
  {{1,4,2,3,2,2,1,2}, {1,1,6,1,1,1,2,4}, {2,3,5,2,1,2,1,1}},
  {{1,5,1,4,1,3,1,1}, {2,1,6,1,1,1,3,2}, {2,2,6,1,2,2,1,1}},
  {{1,2,4,1,4,1,1,3}, {1,1,6,1,1,2,2,3}, {1,4,4,3,1,1,1,2}},
  {{1,3,3,2,3,2,1,2}, {2,1,6,1,1,2,3,1}, {1,3,5,2,2,1,1,2}},
  {{1,4,2,3,2,3,1,1}, {1,1,6,1,1,3,2,2}, {1,4,4,3,1,2,1,1}},
  {{1,2,4,1,4,2,1,2}, {1,1,6,1,1,4,2,1}, {1,2,6,1,3,1,1,2}},
  {{1,3,3,2,3,3,1,1}, {1,2,5,2,1,1,3,2}, {1,3,5,2,2,2,1,1}},
  {{1,5,1,4,2,1,2,1}, {1,1,6,1,2,1,3,2}, {1,2,6,1,3,2,1,1}},
  {{1,4,2,3,3,1,2,1}, {1,2,5,2,1,2,3,1}, {3,2,6,2,1,1,1,1}},
  {{1,3,3,2,4,1,2,1}, {1,1,6,1,2,2,3,1}, {2,3,5,3,1,1,1,1}},
  {{1,2,4,1,5,1,2,1}, {1,1,6,2,1,1,2,3}, {2,2,6,2,2,1,1,1}},
  {{5,1,5,1,1,1,1,2}, {2,1,6,2,1,1,3,1}, {1,4,4,4,1,1,1,1}},
  {{5,1,5,1,1,2,1,1}, {1,1,6,2,1,2,2,2}, {1,3,5,3,2,1,1,1}},
  {{4,2,4,2,1,1,1,2}, {1,1,6,2,1,3,2,1}, {1,2,6,2,3,1,1,1}},
  {{4,1,5,1,2,1,1,2}, {1,2,5,3,1,1,3,1}, {1,6,3,1,1,1,2,2}},
  {{4,2,4,2,1,2,1,1}, {1,1,6,2,2,1,3,1}, {1,6,3,1,1,2,2,1}},
  {{4,1,5,1,2,2,1,1}, {1,1,6,3,1,1,2,2}, {1,5,4,1,1,1,1,3}},
  {{3,3,3,3,1,1,1,2}, {1,1,6,3,1,2,2,1}, {2,5,4,1,1,1,2,1}},
  {{3,2,4,2,2,1,1,2}, {1,4,4,1,1,1,4,1}, {1,5,4,1,1,2,1,2}},
  {{3,3,3,3,1,2,1,1}, {1,3,5,1,1,1,3,2}, {1,5,4,1,1,3,1,1}},
  {{3,1,5,1,3,1,1,2}, {1,3,5,1,1,2,3,1}, {1,6,3,2,1,1,2,1}},
  {{3,2,4,2,2,2,1,1}, {1,2,6,1,1,1,2,3}, {1,5,4,1,2,1,2,1}},
  {{3,1,5,1,3,2,1,1}, {2,2,6,1,1,1,3,1}, {2,4,5,1,1,1,1,2}},
  {{2,4,2,4,1,1,1,2}, {1,2,6,1,1,2,2,2}, {2,4,5,1,1,2,1,1}},
  {{2,3,3,3,2,1,1,2}, {1,2,6,1,1,3,2,1}, {1,5,4,2,1,1,1,2}},
  {{2,4,2,4,1,2,1,1}, {1,3,5,2,1,1,3,1}, {1,4,5,1,2,1,1,2}},
  {{2,2,4,2,3,1,1,2}, {1,2,6,1,2,1,3,1}, {1,5,4,2,1,2,1,1}},
  {{2,3,3,3,2,2,1,1}, {1,2,6,2,1,1,2,2}, {1,4,5,1,2,2,1,1}},
  {{2,1,5,1,4,1,1,2}, {1,2,6,2,1,2,2,1}, {3,3,6,1,1,1,1,1}}
};

//------------------------------------------------------------------------

static int makeDataCodewords(GString *value, int *codewords);
static int countNumeric(GString *value, int start);
static int countText(GString *value, int start);
static int countBinary(GString *value, int start);
static GBool makeNumericCodewords(GString *value, int start, int n,
				  int *codewords, int &codewordIdx);
static int divMod900(int *d, int n);
static GBool makeByteCodewords(GString *value, int start, int n,
			       int *codewords, int &codewordIdx);
static GBool makeTextCodewords(GString *value, int start, int n,
			       int *codewords, int &codewordIdx);
static GBool appendTextHalfSymbol(int *codewords, int &codewordIdx,
				  int halfSymbol, int &prevHalfSymbol);
static GBool appendDataCodeword(int *codewords, int &codewordIdx,
				int codeword);
static void makeErrorCorrectionCodewords(int errorCorrectionLevel,
					 int *codewords, int length);
static void drawBarcode(int *codewords, int totalLength,
			int nRows, int nCols,
			int errorCorrectionLevel,
			double moduleWidth, double moduleHeight,
			double fieldWidth, double fieldHeight,
			GString *appearBuf);
static void drawPattern(int &x, int y, char *pattern, int length,
			GString *appearBuf);

//------------------------------------------------------------------------

GBool drawPDF417Barcode(double fieldWidth, double fieldHeight,
			double moduleWidth, double moduleHeight,
			int errorCorrectionLevel, GString *value,
			GString *appearBuf) {
  int codewords[maxDataCodewords + maxErrorCorrectionCodewords];

  if (moduleWidth <= 0 || moduleHeight <= 0) {
    error(errSyntaxError, -1, "Invalid module size in PDF417 barcode");
    return gFalse;
  }

  // each row contains:
  // - left quiet zone (2 modules, minimum)
  // - start code (17 modules)
  // - left row indicator (17 modules)
  // - nCols codewords (17 modules each)
  // - right row indicator (17 modules)
  // - stop code (18 modules)
  // - right quiet zone (2 modules, minimum)
  // total = 73 + nCols * 17
  int nCols = (int)floor((fieldWidth / moduleWidth - 73) / 17);
  if (nCols < 1) {
    error(errSyntaxError, -1, "Invalid field size for PDF417 barcode");
    return gFalse;
  }
  if (nCols > 30) {
    nCols = 30;
  }

  int nDataCodewords = makeDataCodewords(value, codewords);
  if (nDataCodewords == maxDataCodewords) {
    error(errSyntaxError, -1, "Too many codewords in PDF417 barcode");
    return gFalse;
  }

  int nErrorCorrectionCodewords = 1 << (errorCorrectionLevel + 1);

  // total codewords = 1 + nDataCodewords + nPadCodewords
  //                   + nErrorCorrectionCodewords
  int nRows = (1 + nDataCodewords + nErrorCorrectionCodewords + (nCols - 1))
              / nCols;
  if (nRows < 3) {
    nRows = 3;
  }
  if (nRows > 90) {
    error(errSyntaxError, -1, "Too many rows in PDF417 barcode");
    return gFalse;
  }
  int nPadCodewords = nRows * nCols
                      - (1 + nDataCodewords + nErrorCorrectionCodewords);
  int length = 1 + nDataCodewords + nPadCodewords;
  if (length > maxDataCodewords) {
    error(errSyntaxError, -1, "Too many codewords in PDF417 barcode");
    return gFalse;
  }
  codewords[0] = length;
  for (int i = 1 + nDataCodewords; i < length; ++i) {
    codewords[i] = 900;
  }

  makeErrorCorrectionCodewords(errorCorrectionLevel, codewords, length);

  // we need at least 3 modules of vertical quiet space above and below
  // (the spec says 2, but Adobe seems to use 3)
  if (nRows * moduleHeight + 6 * moduleWidth > fieldHeight) {
    error(errSyntaxError, -1, "Too many rows in PDF417 barcode");
    return gFalse;
  }

  // Adobe adjusts the module height to fill the available space
  double moduleHeight2 = (fieldHeight - 6 * moduleWidth) / nRows;

  drawBarcode(codewords, length + nErrorCorrectionCodewords,
	      nRows, nCols, errorCorrectionLevel,
	      moduleWidth, moduleHeight2, fieldWidth, fieldHeight,
	      appearBuf);

  return gTrue;
}

// Generate up to maxDataCodewords-1 codewords in codewords[1 .. m],
// leaving codewords[0] unused.  Returns the number of generated
// codewords (m), or maxDataCodewords if there are too many to fit.
static int makeDataCodewords(GString *value, int *codewords) {
  GBool textMode = gTrue;
  int codewordIdx = 1;
  int valueIdx = 0;
  while (valueIdx < value->getLength()) {

    int nNumeric = countNumeric(value, valueIdx);
    if (nNumeric >= 13) {
      if (!appendDataCodeword(codewords, codewordIdx, numericModeLatch) ||
	  !makeNumericCodewords(value, valueIdx, nNumeric,
				codewords, codewordIdx)) {
	return maxDataCodewords;
      }
      valueIdx += nNumeric;
      textMode = gFalse;

    } else {
      int nText = countText(value, valueIdx);
      if (nText >= 5) {
	if (!textMode) {
	  if (!appendDataCodeword(codewords, codewordIdx, textModeLatch)) {
	    return maxDataCodewords;
	  }
	}
	if (!makeTextCodewords(value, valueIdx, nText,
			       codewords, codewordIdx)) {
	  return maxDataCodewords;
	}
	valueIdx += nText;
	textMode = gTrue;

      } else {
	int nBinary = countBinary(value, valueIdx);
	//~ this doesn't currently use byte mode shift when nBinary==1
	//~   because we don't track the text submode (which is needed
	//~   because we'll return to that submode after the byte shift)
	int code = (nBinary % 6 == 0) ? byteModeLatchMult6
	                              : byteModeLatchNonMult6;
	if (!appendDataCodeword(codewords, codewordIdx, code) ||
	    !makeByteCodewords(value, valueIdx, nBinary,
			       codewords, codewordIdx)) {
	  return maxDataCodewords;
	}
	valueIdx += nBinary;
	textMode = gFalse;
      }
    }
  }

  return codewordIdx - 1;
}

static int countNumeric(GString *value, int start) {
  int n = 0;
  for (int i = start; i < value->getLength(); ++i) {
    char c = value->getChar(i);
    if (!(c >= '0' && c <= '9')) {
      break;
    }
    ++n;
  }
  return n;
}

static int countText(GString *value, int start) {
  int n = 0;
  for (int i = start; i < value->getLength(); ++i) {
    char c = value->getChar(i);
    if ((c & 0x80) ||
	(textCompaction[(int)c][0] == 30 &&
	 textCompaction[(int)c][1] == 30 &&
	 textCompaction[(int)c][2] == 30 &&
	 textCompaction[(int)c][3] == 30) ||
	countNumeric(value, i) >= 13) {
      break;
    }
    ++n;
  }
  return n;
}

static int countBinary(GString *value, int start) {
  int n = 0;
  for (int i = start; i < value->getLength(); ++i) {
    if (countNumeric(value, i) >= 13 ||
	countText(value, i) >= 5) {
      break;
    }
    ++n;
  }
  return n;
}

static GBool makeNumericCodewords(GString *value, int start, int n,
				  int *codewords, int &codewordIdx) {
  for (int i = 0; i < n; i += 44) {
    int nInDigits = n - i;
    if (nInDigits > 44) {
      nInDigits = 44;
    }
    int inDigits[45];
    inDigits[0] = 1;
    for (int j = 0; j < nInDigits; ++j) {
      inDigits[j+1] = value->getChar(start + i + j) - '0';
    }
    int nOutDigits = 1 + nInDigits / 3;
    int outDigits[15];
    for (int j = 0; j < nOutDigits; ++j) {
      outDigits[nOutDigits - 1 - j] = divMod900(inDigits, nInDigits + 1);
    }
    for (int j = 0; j < nOutDigits; ++j) {
      if (!appendDataCodeword(codewords, codewordIdx, outDigits[j])) {
	return gFalse;
      }
    }
  }
  return gTrue;
}

// Given x = sum(i = 0..n-1, d[i] * 10^(n-1-i))
// Compute x' = x / 900 and return x % 900.
static int divMod900(int *d, int n) {
  int r = 0;
  for (int i = 0; i < n; ++i) {
    int dd = r * 10 + d[i];
    d[i] = dd / 900;
    r = dd % 900;
  }
  return r;
}

static GBool makeByteCodewords(GString *value, int start, int n,
			       int *codewords, int &codewordIdx) {
  int i;
  for (i = 0; i <= n - 6; i += 6) {
    long long in = 0;
    for (int j = 0; j < 6; ++j) {
      in = (in << 8) | (value->getChar(start + i + j) & 0xff);
    }
    int outCodes[5];
    for (int j = 4; j >= 0; --j) {
      outCodes[j] = (int)(in % 900);
      in /= 900;
    }
    for (int j = 0; j < 4; ++j) {
      if (!appendDataCodeword(codewords, codewordIdx, outCodes[j])) {
	return gFalse;
      }
    }
  }
  for (; i < n; ++i) {
    if (!appendDataCodeword(codewords, codewordIdx,
			    value->getChar(start + i) & 0xff)) {
      return gFalse;
    }
  }
  return gTrue;
}

static GBool makeTextCodewords(GString *value, int start, int n,
			       int *codewords, int &codewordIdx) {
  int submode = textSubmodeAlpha;
  int prevSubmode = textSubmodeAlpha;
  int nextSubmode;
  int prevHalfSymbol = 30;
  int valueIdx = start;
  while (valueIdx < start + n) {
    int c = value->getChar(valueIdx) & 0x7f;
    nextSubmode = submode;
    switch (submode) {
    case textSubmodeAlpha:
      if (textCompaction[c][textSubmodeAlpha] < 30) {
	if (!appendTextHalfSymbol(codewords, codewordIdx,
				  textCompaction[c][textSubmodeAlpha],
				  prevHalfSymbol)) {
	  return gFalse;
	}
	++valueIdx;
      } else if (textCompaction[c][textSubmodeLower] < 30) {
	if (!appendTextHalfSymbol(codewords, codewordIdx,
				  textAlphaLowerLatch, prevHalfSymbol)) {
	  return gFalse;
	}
	nextSubmode = textSubmodeLower;
      } else if (textCompaction[c][textSubmodeMixed] < 30) {
	if (!appendTextHalfSymbol(codewords, codewordIdx,
				  textAlphaMixedLatch, prevHalfSymbol)) {
	  return gFalse;
	}
	nextSubmode = textSubmodeMixed;
      } else if (textCompaction[c][textSubmodePunc] < 30) {
	if (valueIdx < start + n - 2 &&
	    textCompaction[value->getChar(valueIdx + 1) & 0x7f]
	                  [textSubmodePunc] < 30 &&
	    textCompaction[value->getChar(valueIdx + 2) & 0x7f]
	                  [textSubmodePunc] < 30) {
	  if (!appendTextHalfSymbol(codewords, codewordIdx,
				    textAlphaMixedLatch, prevHalfSymbol) ||
	      !appendTextHalfSymbol(codewords, codewordIdx,
				    textMixedPuncLatch, prevHalfSymbol)) {
	    return gFalse;
	  }
	  nextSubmode = textSubmodePunc;
	} else {
	  if (!appendTextHalfSymbol(codewords, codewordIdx,
				    textAlphaPuncShift, prevHalfSymbol)) {
	    return gFalse;
	  }
	  nextSubmode = textSubmodePuncShift;
	}
      } else {
	// this shouldn't happen
	++valueIdx;
      }
      break;
    case textSubmodeAlphaShift:
      if (textCompaction[c][textSubmodeAlpha] < 30) {
	if (!appendTextHalfSymbol(codewords, codewordIdx,
				  textCompaction[c][textSubmodeAlpha],
				  prevHalfSymbol)) {
	  return gFalse;
	}
	++valueIdx;
      } else {
	// this shouldn't happen
	if (!appendTextHalfSymbol(codewords, codewordIdx,
				  0, prevHalfSymbol)) {
	  return gFalse;
	}
      }
      nextSubmode = prevSubmode;
      break;
    case textSubmodeLower:
      if (textCompaction[c][textSubmodeAlpha] < 30) {
	if (valueIdx < start + n - 2 &&
	    textCompaction[value->getChar(valueIdx + 1) & 0x7f]
	                  [textSubmodeAlpha] < 30 &&
	    textCompaction[value->getChar(valueIdx + 2) & 0x7f]
	                  [textSubmodeAlpha] < 30) {
	  if (!appendTextHalfSymbol(codewords, codewordIdx,
				    textLowerMixedLatch, prevHalfSymbol) ||
	      !appendTextHalfSymbol(codewords, codewordIdx,
				    textMixedAlphaLatch, prevHalfSymbol)) {
	    return gFalse;
	  }
	  nextSubmode = textSubmodeAlpha;
	} else {
	  if (!appendTextHalfSymbol(codewords, codewordIdx,
				    textLowerAlphaShift, prevHalfSymbol)) {
	    return gFalse;
	  }
	  nextSubmode = textSubmodeAlphaShift;
	}
      } else if (textCompaction[c][textSubmodeLower] < 30) {
	if (!appendTextHalfSymbol(codewords, codewordIdx,
				  textCompaction[c][textSubmodeLower],
				  prevHalfSymbol)) {
	  return gFalse;
	}
	++valueIdx;
      } else if (textCompaction[c][textSubmodeMixed] < 30) {
	if (!appendTextHalfSymbol(codewords, codewordIdx,
				  textLowerMixedLatch, prevHalfSymbol)) {
	  return gFalse;
	}
	nextSubmode = textSubmodeMixed;
      } else if (textCompaction[c][textSubmodePunc] < 30) {
	if (valueIdx < start + n - 2 &&
	    textCompaction[value->getChar(valueIdx + 1) & 0x7f]
	                  [textSubmodePunc] < 30 &&
	    textCompaction[value->getChar(valueIdx + 2) & 0x7f]
	                  [textSubmodePunc] < 30) {
	  if (!appendTextHalfSymbol(codewords, codewordIdx,
				    textLowerMixedLatch, prevHalfSymbol) ||
	      !appendTextHalfSymbol(codewords, codewordIdx,
				    textMixedPuncLatch, prevHalfSymbol)) {
	    return gFalse;
	  }
	  nextSubmode = textSubmodePunc;
	} else {
	  if (!appendTextHalfSymbol(codewords, codewordIdx,
				    textLowerPuncShift, prevHalfSymbol)) {
	    return gFalse;
	  }
	  nextSubmode = textSubmodePuncShift;
	}
      } else {
	// this shouldn't happen
	++valueIdx;
      }
      break;
    case textSubmodeMixed:
      if (textCompaction[c][textSubmodeAlpha] < 30) {
	if (!appendTextHalfSymbol(codewords, codewordIdx,
				  textMixedAlphaLatch, prevHalfSymbol)) {
	  return gFalse;
	}
	nextSubmode = textSubmodeAlpha;
      } else if (textCompaction[c][textSubmodeLower] < 30) {
	if (!appendTextHalfSymbol(codewords, codewordIdx,
				  textMixedLowerLatch, prevHalfSymbol)) {
	  return gFalse;
	}
	nextSubmode = textSubmodeLower;
      } else if (textCompaction[c][textSubmodeMixed] < 30) {
	if (!appendTextHalfSymbol(codewords, codewordIdx,
				  textCompaction[c][textSubmodeMixed],
				  prevHalfSymbol)) {
	  return gFalse;
	}
	++valueIdx;
      } else if (textCompaction[c][textSubmodePunc] < 30) {
	if (valueIdx < start + n - 1 &&
	    textCompaction[value->getChar(valueIdx + 1) & 0x7f]
	                  [textSubmodePunc] < 30) {
	  if (!appendTextHalfSymbol(codewords, codewordIdx,
				    textMixedPuncLatch, prevHalfSymbol)) {
	    return gFalse;
	  }
	  nextSubmode = textSubmodePunc;
	} else {
	  if (!appendTextHalfSymbol(codewords, codewordIdx,
				    textMixedPuncShift, prevHalfSymbol)) {
	    return gFalse;
	  }
	  nextSubmode = textSubmodePuncShift;
	}
      } else {
	// this shouldn't happen
	++valueIdx;
      }
      break;
    case textSubmodePunc:
      if (textCompaction[c][textSubmodeAlpha] < 30) {
	if (!appendTextHalfSymbol(codewords, codewordIdx,
				  textPuncAlphaLatch, prevHalfSymbol)) {
	  return gFalse;
	}
	nextSubmode = textSubmodeAlpha;
      } else if (textCompaction[c][textSubmodeLower] < 30) {
	if (!appendTextHalfSymbol(codewords, codewordIdx,
				  textPuncAlphaLatch, prevHalfSymbol) ||
	    !appendTextHalfSymbol(codewords, codewordIdx,
				  textAlphaLowerLatch, prevHalfSymbol)) {
	  return gFalse;
	}
	nextSubmode = textSubmodeLower;
      } else if (textCompaction[c][textSubmodeMixed] < 30) {
	if (!appendTextHalfSymbol(codewords, codewordIdx,
				  textPuncAlphaLatch, prevHalfSymbol) ||
	    !appendTextHalfSymbol(codewords, codewordIdx,
				  textAlphaMixedLatch, prevHalfSymbol)) {
	  return gFalse;
	}
	nextSubmode = textSubmodeMixed;
      } else if (textCompaction[c][textSubmodePunc] < 30) {
	if (!appendTextHalfSymbol(codewords, codewordIdx,
				  textCompaction[c][textSubmodePunc],
				  prevHalfSymbol)) {
	  return gFalse;
	}
	++valueIdx;
      } else {
	// this shouldn't happen
	++valueIdx;
      }
      break;
    case textSubmodePuncShift:
      if (textCompaction[c][textSubmodePunc] < 30) {
	if (!appendTextHalfSymbol(codewords, codewordIdx,
				  textCompaction[c][textSubmodePunc],
				  prevHalfSymbol)) {
	  return gFalse;
	}
	++valueIdx;
      } else {
	// this shouldn't happen
	if (!appendTextHalfSymbol(codewords, codewordIdx,
				  0, prevHalfSymbol)) {
	  return gFalse;
	}
      }
      nextSubmode = prevSubmode;
      break;
    }
    prevSubmode = submode;
    submode = nextSubmode;
  }

  // padding
  if (prevHalfSymbol < 30) {
    appendTextHalfSymbol(codewords, codewordIdx, 29, prevHalfSymbol);
  }

  return gTrue;
}

static GBool appendTextHalfSymbol(int *codewords, int &codewordIdx,
				  int halfSymbol, int &prevHalfSymbol) {
  if (prevHalfSymbol == 30) {
    prevHalfSymbol = halfSymbol;
  } else {
    int codeword = prevHalfSymbol * 30 + halfSymbol;
    if (!appendDataCodeword(codewords, codewordIdx, codeword)) {
      return gFalse;
    }
    prevHalfSymbol = 30;
  }
  return gTrue;
}

static GBool appendDataCodeword(int *codewords, int &codewordIdx,
				int codeword) {
  if (codewordIdx >= maxDataCodewords) {
    return gFalse;
  }
  codewords[codewordIdx++] = codeword;
  return gTrue;
}

// Compute the error correction codewords and add to codewords[].
static void makeErrorCorrectionCodewords(int errorCorrectionLevel,
					 int *codewords, int length) {
  int e[1 << (maxErrorCorrectionLevel + 1)];
  int k = 1 << (errorCorrectionLevel + 1);
  for (int i = 0; i < k; ++i) {
    e[i] = 0;
  }
  for (int i = 0; i < length; ++i) {
    int t1 = (codewords[i] + e[k-1]) % 929;
    int t2, t3;
    for (int j = k-1; j >= 1; --j) {
      t2 = (t1 * errorCorrectionCoeff[errorCorrectionLevel][j]) % 929;
      t3 = 929 - t2;
      e[j] = (e[j-1] + t3) % 929;
    }
    t2 = (t1 * errorCorrectionCoeff[errorCorrectionLevel][0]) % 929;
    t3 = 929 - t2;
    e[0] = t3 % 929;
  }
  int j = length;
  for (int i = k-1; i >= 0; --i) {
    int ee = e[i] ? (929 - e[i]) : 0;
    codewords[j++] = ee;
  }
}

// Draw the barcode in the rectangle (0, 0, fieldWidth, fieldHeight).
static void drawBarcode(int *codewords, int totalLength,
			int nRows, int nCols,
			int errorCorrectionLevel,
			double moduleWidth, double moduleHeight,
			double fieldWidth, double fieldHeight,
			GString *appearBuf) {
  double w = (73 + nCols * 17) * moduleWidth;
  double h = nRows * moduleHeight;
  double x0 = 0.5 * (fieldWidth - w);
  double y0 = 0.5 * (fieldHeight - h);

  appearBuf->append("0 g q\n");
  appearBuf->appendf("{0:.4f} 0 0 {1:.4f} {2:.4f} {3:.4f} cm\n",
		     moduleWidth, moduleHeight, x0, y0);

  int i = 0;
  for (int row = 0; row < nRows; ++row) {
    int cluster = row % 3;
    int y = nRows - 1 - row;
    int left = 0;
    int right = 0;
    switch (cluster) {
    case 0:
      left = 30 * (row / 3) + (nRows - 1) / 3;
      right = 30 * (row / 3) + (nCols - 1);
      break;
    case 1:
      left = 30 * (row / 3) + (errorCorrectionLevel * 3) + (nRows - 1) % 3;
      right = 30 * (row / 3) + (nRows - 1) / 3;
      break;
    case 2:
      left = 30 * (row / 3) + (nCols - 1);
      right = 30 * (row / 3) + (errorCorrectionLevel * 3) + (nRows - 1) % 3;
      break;
    }
    int x = 0;
    drawPattern(x, y, startPattern, startPatternLength, appearBuf);
    drawPattern(x, y, patterns[left][cluster], patternLength, appearBuf);
    for (int col = 0; col < nCols; ++col) {
      drawPattern(x, y, patterns[codewords[i]][cluster], patternLength,
		  appearBuf);
      ++i;
    }
    drawPattern(x, y, patterns[right][cluster], patternLength, appearBuf);
    drawPattern(x, y, stopPattern, stopPatternLength, appearBuf);
  }

  appearBuf->append("Q\n");
}

static void drawPattern(int &x, int y, char *pattern, int length,
			GString *appearBuf) {
  for (int i = 0; i < length; ++i) {
    if (!(i & 1)) {
      appearBuf->appendf("{0:d} {1:d} {2:d} 1 re f\n", x, y, pattern[i]);
    }
    x += pattern[i];
  }
}
