/*
    SPDX-FileCopyrightText: 2008 Evgeniy Ivanov <powerfox@kde.ru>

    SPDX-License-Identifier: GPL-2.0-only OR GPL-3.0-only OR LicenseRef-KDE-Accepted-GPL
*/

#ifndef KDEVPLATFORM_DVCS_PLUGIN_H
#define KDEVPLATFORM_DVCS_PLUGIN_H

#include <QObject>
#include <QUrl>

#include <interfaces/iplugin.h>

#include "dvcsevent.h"
#include <vcs/vcsexport.h>
#include <vcs/interfaces/idistributedversioncontrol.h>
#include <vcs/interfaces/ibranchingversioncontrol.h>

class QMenu;

namespace KDevelop
{
class DVcsJob;
class DistributedVersionControlPluginPrivate;

/**
 * DistributedVersionControlPlugin is a base class for git/hg/bzr plugins. This class implements
 * KDevelop::IBasicVersionControl, KDevelop::IDistributedVersionControl and KDevelop::IPlugin (contextMenuExtension).
 * DistributedVersionControlPlugin class uses IDVCSexecutor to get all jobs
 * from real DVCS plugins like Git. It is based on KDevelop's CVS plugin (also looks like svn plugin is it's relative too).
 * @note Create only special items in contextMenuExtension, all standard menu items are created in vcscommon plugin!
 */
class KDEVPLATFORMVCS_EXPORT DistributedVersionControlPlugin : public IPlugin, public IDistributedVersionControl, public IBranchingVersionControl
{
    Q_OBJECT
    Q_INTERFACES(KDevelop::IBasicVersionControl KDevelop::IDistributedVersionControl KDevelop::IBranchingVersionControl)
public:
    DistributedVersionControlPlugin(const QString& componentName, QObject* parent, const KPluginMetaData& metaData);
    ~DistributedVersionControlPlugin() override;

    // Begin: KDevelop::IBasicVersionControl

    /** Used in KDevelop's appwizardplugin (creates import widget) */
    VcsImportMetadataWidget* createImportMetadataWidget(QWidget* parent) override;

    // From KDevelop::IPlugin
    /** Creates context menu
     * @note Create only special items here (like checkout), all standard menu items are created in vcscommon plugin!
     */
    ContextMenuExtension contextMenuExtension(Context* context, QWidget* parent) override;

    /**
      * Parses the output generated by a <tt>dvcs log</tt> command and
      * fills the given QList with all commits infos found in the given output.
      * @param job The finished job of a <tt>dvcs log</tt> call
      * @param revisions Will be filled with all revision infos found in @p jobOutput
      * TODO: Change it to pass the results in @c job->getResults()
      */
    virtual void parseLogOutput(const DVcsJob * job,
                                QVector<KDevelop::DVcsEvent>& revisions) const = 0;
    
    /** Returns the list of all commits (in all branches).
     * @see CommitView and CommitViewDelegate to see how this list is used.
     */
    virtual QVector<KDevelop::DVcsEvent> allCommits(const QString& repo) = 0;

    /**
     * When a plugin wants to add elements to the vcs menu, this method can be
     * overridden.
     */
    virtual void additionalMenuEntries(QMenu* menu, const QList<QUrl>& urls);
public Q_SLOTS:
    //slots for context menu
    void ctxBranchManager();

protected:
    /** Checks if dirPath is located in DVCS repository */
    virtual bool isValidDirectory(const QUrl &dirPath) = 0;

private:
    const QScopedPointer<class DistributedVersionControlPluginPrivate> d_ptr;
    Q_DECLARE_PRIVATE(DistributedVersionControlPlugin)
};

}

#endif
