// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

import {IMAGE_FILE} from './image_processor_test_support.js';

/**
 * Bytes for a sample transparent PNG.
 *
 * https://screenshot.googleplex.com/3ZTEbK2D3CCaDCb.png
 */
export const TRANSPARENT_PNG_BYTES = new Uint8Array([
  137, 80,  78,  71,  13,  10,  26,  10,  0,   0,   0,   13,  73,  72,  68,
  82,  0,   0,   0,   194, 0,   0,   0,   68,  8,   6,   0,   0,   0,   211,
  8,   234, 244, 0,   0,   0,   1,   115, 82,  71,  66,  0,   174, 206, 28,
  233, 0,   0,   0,   4,   103, 65,  77,  65,  0,   0,   177, 143, 11,  252,
  97,  5,   0,   0,   0,   32,  99,  72,  82,  77,  0,   0,   122, 38,  0,
  0,   128, 132, 0,   0,   250, 0,   0,   0,   128, 232, 0,   0,   117, 48,
  0,   0,   234, 96,  0,   0,   58,  152, 0,   0,   23,  112, 156, 186, 81,
  60,  0,   0,   9,   158, 73,  68,  65,  84,  120, 94,  237, 157, 65,  168,
  77,  93,  20,  199, 191, 175, 20,  69,  81,  36,  73,  146, 40,  138, 162,
  12,  12,  12,  20,  69,  81,  36,  73,  146, 40,  138, 162, 12,  12,  20,
  165, 40,  138, 34,  73,  50,  16,  51,  138, 50,  48,  48,  80,  20,  69,
  145, 36,  73,  162, 40,  138, 50,  120, 3,   3,   95,  249, 106, 127, 126,
  219, 251, 159, 111, 191, 253, 206, 57,  251, 220, 123, 223, 117, 207, 189,
  103, 157, 122, 189, 207, 61,  103, 239, 179, 214, 218, 235, 191, 254, 107,
  173, 189, 223, 253, 254, 118, 191, 174, 191, 236, 50,  11,  52,  221, 2,
  0,   193, 46,  179, 64,  211, 45,  240, 151, 12,  240, 207, 191, 206, 217,
  143, 217, 160, 105, 62,  32,  255, 207, 128, 48,  52,  52,  228, 138, 126,
  82,  209, 162, 108, 44,  247, 82,  151, 141, 47,  182, 189, 217, 175, 251,
  254, 227, 203, 3,   57,  169, 1,   193, 2,   65,  83,  3,   161, 1,   97,
  56,  10,  24,  35,  53,  155, 145, 12,  8,   6,   4,   111, 129, 166, 7,
  2,   3,   130, 1,   193, 128, 48,  236, 3,   86,  35,  88,  68,  52,  70,
  176, 98,  249, 119, 56,  104, 122, 106, 208, 116, 253, 45,  53,  178, 212,
  200, 2,   65,  156, 26,  149, 245, 250, 155, 30,  49,  76,  255, 193, 237,
  42,  141, 218, 80,  51,  32,  20,  91,  192, 128, 96,  64,  48,  234, 180,
  26,  98,  160, 107, 40,  99,  132, 224, 72,  73,  25,  27,  90,  49,  61,
  216, 205, 4,   3,   130, 1,   33,  139, 244, 77,  14,  4,   6,   4,   3,
  130, 1,   33,  56,  16,  154, 109, 168, 89,  177, 108, 197, 114, 145, 5,
  6,   185, 89,  96,  140, 96,  140, 96,  140, 144, 199, 8,   118, 12,  219,
  142, 97,  219, 49,  236, 68,  139, 176, 201, 197, 148, 253, 97,  204, 80,
  106, 249, 251, 190, 189, 106, 71,  44,  236, 136, 133, 237, 19,  197, 71,
  44,  44,  53,  178, 212, 200, 82,  163, 26,  165, 70,  183, 110, 221, 114,
  219, 182, 109, 115, 11,  23,  46,  116, 227, 199, 143, 231, 171, 102, 220,
  132, 9,   19,  220, 130, 5,   11,  220, 230, 205, 155, 221, 165, 75,  151,
  220, 167, 79,  159, 50,  186, 254, 240, 225, 131, 127, 158, 231, 154, 186,
  144, 237, 118, 117, 176, 221, 209, 163, 71,  221, 210, 165, 75,  221, 196,
  137, 19,  189, 13,  199, 141, 27,  231, 230, 207, 159, 239, 182, 108, 217,
  226, 110, 220, 184, 225, 109, 154, 186, 218, 125, 191, 230, 237, 245, 248,
  90,  165, 70,  247, 239, 223, 247, 11,  178, 120, 241, 98,  119, 249, 242,
  101, 247, 246, 237, 91,  191, 8,   31,  63,  126, 116, 247, 238, 221, 115,
  27,  55,  110, 244, 11,  197, 207, 146, 37,  75,  188, 13,  119, 238, 220,
  153, 125, 102, 64,  168, 206, 104, 63,  127, 254, 244, 0,   192, 233, 183,
  110, 221, 234, 238, 222, 189, 235, 62,  127, 254, 236, 237, 253, 242, 229,
  75,  119, 254, 252, 121, 31,  136, 176, 233, 178, 101, 203, 220, 187, 119,
  239, 74,  177, 208, 107, 71,  238, 244, 253, 181, 1,   194, 197, 139, 23,
  253, 162, 28,  60,  120, 208, 125, 251, 246, 109, 84,  100, 215, 42,  92,
  189, 122, 213, 47,  206, 137, 19,  39,  178, 133, 121, 245, 234, 85,  6,
  134, 186, 50,  194, 185, 115, 231, 188, 140, 117, 232, 211, 3,   2,   152,
  21,  123, 95,  191, 126, 189, 176, 70,  96,  29,  246, 236, 217, 227, 229,
  158, 49,  99,  134, 123, 243, 230, 77,  45,  228, 207, 19,  98,  32,  128,
  112, 231, 206, 29,  191, 40,  80,  113, 202, 145, 191, 124, 249, 226, 23,
  230, 193, 131, 7,   153, 61,  190, 127, 255, 94,  123, 32,  136, 201, 234,
  0,   132, 195, 135, 15,  123, 123, 157, 58,  117, 42,  19,  167, 200, 238,
  128, 97,  197, 138, 21,  254, 249, 69,  139, 22,  57,  64,  212, 13,  71,
  236, 212, 145, 59,  29,  63,  130, 17,  202, 184, 175, 211, 23,  21,  141,
  39,  237, 153, 60,  121, 178, 175, 1,   112, 242, 42,  142, 194, 162, 48,
  46,  156, 179, 78,  142, 150, 167, 3,   58,  214, 33,  117, 123, 254, 252,
  185, 15,  58,  83,  167, 78,  117, 95,  191, 126, 173, 180, 161, 246, 236,
  217, 179, 44,  208, 144, 178, 230, 173, 101, 63,  215, 16,  146, 189, 167,
  71,  44,  142, 29,  59,  230, 141, 188, 118, 237, 218, 142, 114, 208, 186,
  3,   97,  246, 236, 217, 181, 0,   194, 161, 67,  135, 188, 28,  49,  251,
  166, 28,  121, 238, 220, 185, 126, 220, 154, 53,  107, 12,  8,   221, 104,
  175, 82,  136, 97,  96,  10,  183, 78,  24,  41,  4,   2,   157, 16,  114,
  224, 73,  147, 38,  249, 232, 199, 59,  168, 65,  242, 228, 127, 248, 240,
  161, 219, 183, 111, 159, 167, 125, 158, 101, 158, 89,  179, 102, 57,  28,
  38,  76,  3,   72,  17,  40,  216, 119, 236, 216, 145, 57,  52,  197, 252,
  134, 13,  27,  178, 110, 11,  227, 246, 238, 221, 235, 107, 156, 248, 170,
  2,   132, 71,  143, 30,  185, 245, 235, 215, 187, 41,  83,  166, 248, 119,
  76,  155, 54,  205, 237, 222, 189, 219, 51,  165, 100, 39,  138, 211, 85,
  195, 33,  137, 234, 124, 142, 110, 60,  203, 79,  138, 81,  105, 70,  48,
  247, 241, 227, 199, 71,  216, 35,  5,   4,   213, 10,  211, 167, 79,  207,
  198, 81,  84,  171, 91,  199, 120, 10,  106, 228, 15,  237, 78,  77,  199,
  149, 103, 123, 10,  116, 244, 16,  91,  134, 250, 198, 242, 116, 43,  35,
  97,  222, 90,  48,  130, 90,  118, 50,  88,  106, 33,  139, 12,  34,  32,
  220, 190, 125, 219, 59,  4,   64,  160, 163, 132, 131, 235, 30,  236, 19,
  143, 231, 30,  206, 1,   120, 112, 124, 156, 157, 54,  109, 92,  144, 107,
  193, 53,  215, 211, 167, 79,  29,  206, 189, 105, 211, 38,  15,  26,  229,
  209, 220, 159, 55,  111, 222, 136, 52,  79,  99,  226, 223, 161, 44,  180,
  132, 1,   34,  157, 49,  90,  195, 63,  126, 252, 240, 109, 98,  62,  163,
  80,  125, 255, 254, 189, 151, 61,  150, 67,  245, 85,  85,  70,  148, 211,
  197, 129, 33,  5,   132, 147, 39,  79,  142, 168, 195, 98,  57,  110, 222,
  188, 233, 3,   8,   29,  168, 216, 238, 212, 34,  177, 221, 207, 158, 61,
  235, 117, 91,  183, 110, 157, 123, 253, 250, 245, 40,  125, 227, 96,  50,
  240, 64,  208, 2,   142, 21,  16,  112, 72,  156, 58,  188, 136, 210, 188,
  7,   122, 143, 13,  202, 62,  5,   172, 16,  94,  20,  226, 98,  6,   125,
  206, 56,  22,  76,  242, 210, 90,  196, 9,   53,  31,  207, 209, 129, 17,
  171, 224, 16,  225, 165, 113, 69,  145, 145, 113, 121, 197, 168, 100, 39,
  34,  107, 44,  145, 88,  243, 241, 57,  109, 79,  28,  145, 207, 82,  129,
  68,  227, 90,  5,   2,   207, 199, 58,  132, 114, 96,  247, 112, 111, 7,
  57,  36,  59,  129, 33,  212, 27,  38,  64,  95,  108, 168, 14,  161, 228,
  214, 152, 253, 251, 247, 143, 80,  101, 224, 129, 32,  70,  8,   59,  24,
  121, 139, 153, 50,  132, 22,  41,  47,  45,  81,  167, 169, 106, 177, 10,
  51,  196, 17,  86,  239, 215, 231, 128, 34,  148, 73,  50,  111, 223, 190,
  221, 143, 37,  189, 169, 10,  4,   242, 117, 198, 156, 57,  115, 102, 148,
  234, 48,  92,  30,  136, 37,  199, 227, 199, 143, 189, 28,  169, 75,  178,
  202, 222, 49,  59,  166, 198, 11,  8,   48,  74,  168, 119, 8,   142, 120,
  142, 34,  187, 75,  223, 48,  61,  211, 88,  233, 11,  120, 194, 43,  181,
  254, 41,  249, 203, 198, 107, 108, 79,  139, 101, 229, 172, 113, 4,   141,
  21,  75,  25,  34,  149, 26,  148, 69,  100, 24,  1,   39,  36,  157, 98,
  163, 142, 28,  55,  5,   132, 88,  158, 120, 33,  227, 232, 92,  246, 126,
  242, 110, 238, 195, 40,  180, 130, 243, 216, 41,  6,   113, 60,  95,  85,
  71,  144, 189, 169, 117, 242, 128, 92,  52,  143, 138, 108, 54,  59,  243,
  128, 80,  52,  46,  79,  111, 233, 123, 229, 202, 149, 108, 19,  79,  227,
  197, 198, 177, 253, 82,  235, 95,  85,  255, 188, 121, 106, 1,   4,   25,
  24,  231, 43,  139, 108, 41,  67,  180, 3,   4,   82,  138, 85,  171, 86,
  121, 154, 166, 40,  37,  26,  177, 105, 68,  126, 222, 46,  16,  180, 144,
  228, 245, 85,  25,  33,  174, 29,  194, 127, 99,  151, 57,  115, 230, 248,
  92,  186, 40,  18,  183, 194, 8,   7,   14,  28,  240, 186, 145, 150, 180,
  2,   4,   138, 90,  198, 237, 218, 181, 171, 99,  32,  84,  209, 151, 90,
  169, 103, 140, 80,  133, 62,  138, 144, 151, 114, 212, 162, 113, 208, 39,
  123, 8,   24,  7,   80,  20,  205, 147, 66,  124, 89,  196, 101, 206, 60,
  160, 144, 135, 242, 57,  69,  105, 252, 222, 162, 136, 91,  4,   56,  141,
  167, 207, 206, 51,  20,  147, 85,  35,  167, 24,  136, 110, 80,  85,  253,
  139, 128, 154, 26,  255, 226, 197, 139, 204, 22,  252, 183, 174, 178, 245,
  99,  207, 70,  107, 68,  103, 171, 42,  192, 139, 236, 46,  125, 169, 21,
  170, 188, 191, 21,  160, 167, 244, 47,  243, 163, 158, 127, 247, 41,  245,
  1,   11,  75,  100, 214, 33,  175, 88,  161, 110, 0,   65,  109, 74,  24,
  96,  172, 128, 64,  59,  21,  61,  232, 42,  85,  5,   130, 58,  78,  71,
  142, 28,  233, 58,  16,  176, 35,  105, 40,  246, 94,  185, 114, 101, 37,
  71,  20,  107, 175, 94,  189, 122, 212, 50,  180, 19,  128, 164, 111, 120,
  76,  166, 221, 64,  58,  86,  64,  98,  158, 158, 3,   1,   33,  232, 229,
  11,  12,  167, 79,  159, 30,  117, 222, 40,  94,  1,   156, 55,  52,  130,
  22,  68,  7,   199, 138, 28,  59,  220, 27,  208, 24,  186, 76,  122, 158,
  98,  27,  250, 79,  49,  2,   231, 155, 98,  234, 86,  161, 151, 231, 208,
  154, 79,  114, 135, 157, 45,  157, 159, 98,  95,  64,  7,   13,  203, 2,
  65,  24,  105, 85,  83,  180, 226, 72,  232, 168, 22,  49,  103, 187, 184,
  138, 198, 95,  187, 118, 205, 3,   155, 231, 227, 157, 255, 80,  142, 86,
  236, 46,  125, 105, 115, 107, 206, 86,  228, 207, 11,  138, 157, 142, 175,
  13,  16,  80,  132, 19,  143, 234, 115, 211, 234, 196, 161, 112, 46,  218,
  116, 56,  14,  91,  253, 20,  148, 228, 143, 97,  116, 130, 174, 229, 104,
  208, 109, 158, 81,  116, 255, 201, 147, 39,  153, 29,  181, 153, 71,  132,
  100, 33,  121, 23,  197, 36,  206, 161, 231, 57,  146, 16,  82,  179, 62,
  39,  85,  224, 57,  222, 199, 15,  64,  150, 204, 121, 239, 39,  207, 103,
  44,  69,  57,  41,  73,  88,  67,  0,   78,  162, 51,  247, 41,  36,  233,
  177, 11,  16,  252, 166, 187, 18,  22,  143, 177, 190, 101, 142, 44,  89,
  98,  231, 193, 1,   245, 78,  54,  193, 72,  203, 180, 87,  193, 166, 29,
  58,  105, 175, 128, 250, 36,  239, 248, 75,  187, 118, 71,  95,  214, 15,
  157, 176, 3,   169, 105, 158, 190, 161, 29,  83,  25,  193, 64,  1,   1,
  101, 200, 71,  1,   4,   197, 217, 204, 153, 51,  51,  135, 36,  141, 193,
  113, 113, 62,  54,  189, 116, 197, 199, 176, 243, 242, 243, 48,  114, 113,
  159, 49,  92,  204, 67,  155, 78,  206, 141, 19,  2,   6,   158, 215, 126,
  0,   247, 104, 29,  234, 210, 179, 23,  46,  92,  240, 69,  44,  255, 230,
  55,  27,  78,  121, 167, 102, 181, 64,  225, 187, 216, 47,  8,   1,   201,
  220, 56,  31,  45,  77,  116, 84,  48,  16,  48,  104, 55,  42,  159, 207,
  211, 151, 231, 218, 117, 4,   246, 66,  104, 249, 106, 231, 155, 185, 96,
  166, 229, 203, 151, 123, 91,  23,  181, 103, 59,  177, 59,  250, 194, 140,
  164, 196, 188, 71,  41,  106, 168, 47,  157, 188, 70,  3,   161, 104, 65,
  187, 29,  17,  170, 58,  82,  170, 88,  174, 187, 252, 69,  118, 172, 170,
  255, 160, 142, 175, 85,  106, 84,  182, 24,  6,   132, 114, 11,  52,  221,
  145, 59,  213, 223, 128, 48,  236, 95,  85,  13,  105, 140, 144, 15,  200,
  170, 246, 171, 43,  163, 140, 0,   66,  89,  204, 233, 119, 69,  199, 74,
  254, 162, 118, 97,  93,  24,  171, 174, 142, 54,  86,  246, 239, 70,  234,
  153, 213, 127, 169, 69,  108, 167, 43,  17,  207, 89,  103, 67,  132, 93,
  161, 50,  71,  162, 171, 34,  32,  196, 127, 24,  148, 178, 225, 32,  232,
  223, 15,  169, 107, 59,  129, 192, 128, 208, 226, 87,  62,  210, 82,  20,
  16,  212, 93,  42,  106, 79,  14,  98,  32,  48,  32,  252, 90,  213, 166,
  71,  180, 248, 252, 125, 220, 166, 53,  70,  232, 223, 98,  222, 24,  161,
  69,  70,  24,  212, 136, 88,  53,  53,  28,  84,  253, 13,  8,   6,   132,
  140, 233, 155, 204, 104, 6,   4,   3,   130, 1,   161, 46,  127, 179, 108,
  197, 230, 239, 111, 167, 75,  93,  77,  175, 209, 186, 169, 255, 40,  70,
  24,  212, 28,  208, 28,  205, 128, 150, 10,  52,  182, 179, 220, 226, 206,
  114, 59,  125, 106, 3,   98,  253, 129, 104, 64,  48,  32,  120, 11,  116,
  51,  245, 232, 135, 64,  96,  64,  48,  32,  24,  16,  134, 125, 160, 22,
  127, 161, 214, 244, 136, 100, 250, 23,  127, 165, 253, 159, 96,  20,  99,
  4,   99,  4,   99,  4,   99,  132, 255, 75,  95,  139, 200, 189, 141, 200,
  189, 182, 191, 49,  130, 49,  130, 49,  66,  204, 8,   101, 189, 214, 94,
  35,  214, 222, 223, 236, 136, 221, 205, 245, 183, 35,  22,  118, 196, 194,
  142, 88,  216, 17,  139, 145, 17,  54,  181, 243, 216, 205, 136, 244, 39,
  186, 34,  38,  127, 250, 127, 180, 216, 211, 47,  1,   182, 179, 70,  118,
  214, 168, 215, 129, 192, 82,  35,  75,  141, 44,  53,  178, 212, 200, 82,
  163, 48,  93,  106, 114, 106, 104, 140, 96,  140, 96,  140, 208, 42,  35,
  164, 34,  134, 221, 55,  11,  244, 187, 5,   42,  21,  203, 253, 174, 164,
  201, 111, 22,  72,  89,  224, 63,  178, 188, 124, 76,  14,  198, 80,  73,
  0,   0,   0,   0,   73,  69,  78,  68,  174, 66,  96,  130,
]);

/** File for transparent sample PNG. */
export const TRANSPARENT_PNG_FILE = new File(
    [TRANSPARENT_PNG_BYTES], 'valid-transparent.png', {type: 'image/png'});

/**
 * Re-exported, renamed sample image file (to clarify its opaque nature).
 */
export const OPAQUE_PNG_FILE = IMAGE_FILE;

/**
 * Bytes for a sample transparent WebP.
 *
 * https://www.gstatic.com/webp/gallery3/2_webp_a.webp
 */
export const TRANSPARENT_WEBP_BYTES = new Uint8Array([
  82,  73,  70,  70,  250, 54,  0,   0,   87,  69,  66,  80,  86,  80,  56,
  88,  10,  0,   0,   0,   16,  0,   0,   0,   129, 1,   0,   138, 1,   0,
  65,  76,  80,  72,  29,  14,  0,   0,   1,   9,   71,  110, 219, 72,  18,
  156, 120, 118, 253, 255, 195, 213, 93,  149, 202, 116, 223, 35,  250, 63,
  1,   186, 101, 152, 113, 7,   152, 147, 189, 45,  199, 41,  217, 123, 242,
  120, 78,  118, 59,  217, 27,  178, 72,  142, 178, 40,  211, 134, 243, 53,
  71,  21,  117, 170, 226, 230, 226, 250, 116, 87,  92,  77,  180, 83,  5,
  187, 28,  75,  57,  110, 145, 189, 144, 13,  100, 79,  101, 11,  217, 19,
  217, 68,  246, 81,  78,  110, 147, 61,  100, 39,  217, 210, 167, 151, 108,
  229, 8,   191, 6,   191, 65,  118, 102, 2,   196, 124, 77,  132, 100, 75,
  97,  14,  155, 72,  2,   31,  55,  224, 72,  250, 31,  95,  15,  5,   254,
  3,   239, 33,  179, 194, 151, 66,  141, 200, 13,  82,  65,  166, 54,  219,
  237, 100, 47,  208, 64,  142, 118, 90,  136, 136, 94,  191, 71,  9,   177,
  194, 35,  38,  30,  96,  152, 1,   16,  79,  152, 129, 15,  143, 56,  123,
  127, 95,  160, 53,  86,  248, 124, 62,  103, 118, 149, 166, 33,  51,  7,
  6,   228, 72,  146, 195, 54,  123, 157, 127, 214, 0,   120, 7,   200, 254,
  71,  196, 4,   228, 111, 132, 192, 35,  149, 143, 120, 162, 137, 91,  116,
  18,  110, 101, 144, 27,  100, 148, 231, 194, 24,  121, 46,  227, 12,  213,
  123, 78,  122, 25,  203, 197, 95,  240, 183, 24,  201, 157, 225, 25,  255,
  65,  238, 24,  240, 11,  62,  105, 133, 22,  89,  37,  145, 89,  26,  210,
  215, 234, 154, 56,  154, 225, 131, 111, 210, 245, 37,  119, 37,  181, 9,
  11,  121, 217, 146, 187, 90,  178, 194, 216, 87,  146, 4,   65,  242, 68,
  144, 88,  115, 201, 70,  146, 64,  246, 15,  68,  154, 87,  246, 210, 210,
  49,  160, 147, 86,  159, 244, 124, 242, 68,  46,  145, 9,   7,   145, 129,
  140, 114, 1,   19,  54,  242, 217, 21,  164, 41,  77,  118, 14,  200, 117,
  155, 179, 108, 197, 107, 71,  67,  110, 60,  88,  85,  173, 170, 170, 86,
  85,  99,  61,  213, 199, 83,  237, 199, 106, 34,  173, 108, 50,  144, 70,
  54,  220, 212, 214, 30,  41,  201, 176, 239, 208, 81,  226, 4,   11,  140,
  2,   130, 130, 9,   10,  176, 128, 132, 72,  9,   146, 104, 215, 110, 253,
  231, 125, 28,  42,  136, 152, 0,   54,  108, 35,  41,  202, 221, 51,  102,
  159, 161, 186, 146, 178, 4,   218, 166, 109, 103, 141, 253, 46,  158, 117,
  238, 123, 177, 205, 158, 109, 219, 249, 149, 124, 1,   154, 233, 189, 74,
  203, 86,  215, 182, 109, 219, 118, 238, 93,  163, 234, 238, 49,  198, 78,
  99,  237, 212, 107, 70,  4,   36,  72,  114, 219, 70,  112, 98,  22,  177,
  88,  2,   184, 28,  158, 148, 179, 253, 175, 227, 42,  255, 14,  100, 42,
  176, 168, 192, 162, 130, 136, 10,  98,  42,  136, 75,  48,  21,  196, 29,
  28,  83,  1,   166, 2,   156, 10,  208, 44,  217, 105, 42,  64,  179, 226,
  178, 146, 43,  224, 54,  87,  207, 153, 255, 239, 39,  39,  231, 121, 34,
  194, 21,  36,  171, 109, 243, 80,  140, 133, 101, 82,  221, 80,  67,  206,
  159, 235, 252, 143, 79,  77,  37,  214, 94,  64,  74,  146, 14,  20,  195,
  56,  87,  121, 241, 215, 191, 172, 28,  115, 76,  41,  46,  172, 162, 118,
  254, 63,  201, 74,  47,  49,  165, 16,  35,  155, 27,  98,  227, 189, 175,
  86,  127, 173, 255, 45,  10,  123, 22,  183, 57,  223, 158, 91,  118, 135,
  69,  97,  94,  248, 211, 219, 185, 187, 191, 244, 48,  207, 7,   226, 214,
  15,  2,   138, 64,  156, 56,  19,  76,  172, 68,  51,  40,  141, 121, 254,
  137, 45,  44,  5,   37,  158, 74,  158, 70,  166, 146, 175, 69,  84,  159,
  48,  190, 145, 164, 59,  138, 200, 82,  105, 160, 104, 110, 131, 46,  54,
  190, 26,  126, 82,  240, 83,  213, 247, 134, 154, 185, 254, 77,  62,  137,
  150, 142, 21,  84,  182, 148, 28,  83,  79,  201, 121, 111, 162, 230, 217,
  55,  116, 92,  118, 239, 205, 232, 169, 250, 206, 80,  129, 157, 146, 166,
  109, 231, 137, 48,  251, 37,  178, 194, 85,  95,  243, 32,  21,  78,  174,
  101, 66,  170, 219, 179, 197, 175, 178, 114, 215, 87,  186, 39,  120, 149,
  198, 87,  159, 64,  190, 130, 30,  172, 211, 226, 253, 251, 26,  54,  156,
  213, 228, 39,  8,   26,  186, 236, 116, 42,  200, 193, 168, 243, 89,  91,
  3,   184, 184, 163, 66,  251, 98,  184, 143, 160, 209, 103, 81,  89,  172,
  25,  208, 147, 136, 74,  65,  74,  128, 170, 18,  128, 149, 141, 140, 28,
  255, 136, 242, 62,  74,  121, 227, 218, 61,  196, 172, 167, 176, 0,   126,
  162, 9,   85,  72,  73,  213, 21,  22,  72,  88,  148, 253, 61,  167, 18,
  184, 0,   9,   37,  112, 5,   136, 231, 249, 26,  133, 143, 169, 9,   205,
  66,  19,  84,  222, 91,  96,  81,  176, 156, 236, 74,  152, 79,  51,  139,
  176, 31,  66,  86,  251, 172, 37,  1,   145, 41,  254, 147, 93,  96,  200,
  84,  255, 200, 80,  60,  84,  55,  238, 164, 8,   1,   7,   229, 143, 163,
  211, 9,   168, 255, 236, 26,  4,   26,  147, 246, 24,  249, 231, 226, 95,
  244, 175, 160, 12,  224, 167, 59,  24,  254, 119, 30,  182, 215, 190, 141,
  4,   8,   181, 84,  181, 175, 236, 60,  96,  40,  196, 58,  232, 206, 65,
  84,  143, 236, 106, 253, 57,  130, 85,  116, 231, 244, 135, 65,  96,  96,
  58,  104, 206, 217, 103, 131, 0,   193, 168, 52,  171, 61,  148, 114, 220,
  131, 222, 156, 252, 168, 105, 193, 20,  182, 213, 249, 250, 169, 66,  211,
  6,   165, 202, 102, 28,  1,   131, 94,  229, 60,  45,  52,  104, 13,  253,
  92,  164, 83,  248, 183, 202, 63,  206, 66,  84,  3,   190, 214, 22,  119,
  202, 127, 68,  126, 98,  66,  106, 13,  78,  219, 248, 93,  66,  127, 20,
  84,  35,  198, 237, 198, 130, 6,   85,  179, 160, 70,  93,  78,  126, 98,
  191, 93,  161, 133, 108, 65,  214, 167, 61,  96,  250, 185, 136, 223, 82,
  224, 248, 111, 159, 178, 22,  254, 39,  14,  90,  75,  233, 5,   0,   171,
  40,  232, 95,  96,  161, 97,  51,  137, 212, 194, 255, 196, 130, 235, 62,
  184, 0,   154, 213, 183, 5,   208, 133, 73,  85,  0,   17,  167, 37,  10,
  128, 10,  64,  108, 139, 0,   239, 222, 100, 139, 0,   239, 165, 100, 138,
  160, 191, 203, 143, 95,  244, 177, 252, 20,  211, 109, 96,  158, 46,  188,
  11,  68,  42,  21,  35,  5,   0,   91,  0,   17,  203, 255, 17,  18,  159,
  43,  90,  190, 4,   184, 41,  128, 11,  65,  87,  0,   19,  87,  0,   19,
  187, 129, 68,  106, 17,  250, 87,  124, 46,  128, 243, 252, 87,  225, 159,
  253, 82,  0,   86,  217, 191, 249, 159, 52,  194, 255, 196, 131, 116, 9,
  187, 0,   208, 22,  0,   60,  253, 75,  68,  156, 81,  174, 141, 95,  144,
  76,  57,  176, 191, 192, 124, 13,  115, 194, 63,  23,  219, 240, 15,  233,
  84,  211, 202, 126, 244, 25,  105, 145, 201, 154, 254, 184, 72,  91,  0,
  232, 244, 138, 67,  141, 170, 197, 161, 70,  11,  0,   81,  167, 120, 1,
  148, 203, 237, 208, 162, 215, 40,  195, 234, 43,  180, 208, 104, 28,  61,
  3,   222, 138, 146, 163, 62,  94,  228, 11,  94,  234, 180, 49,  11,  26,
  200, 190, 73,  107, 195, 94,  33,  143, 186, 206, 235, 114, 214, 89,  127,
  7,   218, 23,  232, 160, 139, 167, 104, 160, 105, 90,  52,  121, 10,  212,
  200, 147, 82,  110, 63,  130, 181, 137, 28,  212, 41,  41,  207, 187, 194,
  230, 104, 29,  216, 162, 160, 133, 12,  58,  5,   200, 50,  66,  140, 16,
  129, 128, 145, 74,  0,   26,  161, 0,   93,  94,  161, 231, 68,  193, 120,
  61,  22,  165, 199, 82,  192, 163, 119, 99,  105, 67,  28,  91,  78,  126,
  96,  237, 219, 34,  197, 34,  16,  253, 8,   73,  136, 7,   157, 106, 63,
  66,  244, 201, 137, 31,  104, 220, 70,  218, 253, 204, 194, 4,   89,  137,
  166, 53,  166, 190, 168, 249, 235, 187, 137, 84,  21,  253, 103, 170, 158,
  254, 209, 239, 26,  75,  216, 243, 42,  167, 159, 177, 134, 8,   237, 121,
  29,  156, 171, 119, 21,  251, 215, 229, 198, 49,  103, 163, 245, 159, 159,
  14,  210, 85,  56,  181, 31,  64,  110, 21,  194, 167, 138, 253, 220, 162,
  218, 11,  133, 228, 13,  143, 201, 198, 104, 63,  127, 54,  25,  231, 42,
  12,  114, 169, 26,  214, 60,  187, 69,  151, 160, 49,  37,  144, 105, 186,
  250, 245, 247, 23,  0,   148, 182, 112, 148, 192, 150, 181, 148, 78,  117,
  207, 37,  202, 86,  78,  161, 4,   42,  164, 103, 14,  234, 163, 235, 212,
  28,  84,  204, 244, 95,  107, 164, 76,  255, 243, 45,  137, 127, 5,   97,
  147, 111, 135, 39,  222, 89,  195, 239, 191, 6,   25,  201, 64,  95,  57,
  95,  173, 157, 102, 127, 62,  145, 113, 191, 56,  133, 191, 68,  158, 232,
  216, 153, 237, 68,  154, 174, 181, 119, 187, 93,  11,  137, 233, 30,  51,
  206, 196, 236, 208, 246, 223, 31,  9,   238, 18,  50,  195, 126, 211, 201,
  40,  158, 153, 17,  136, 60,  119, 237, 122, 219, 235, 169, 232, 200, 236,
  104, 202, 227, 200, 146, 129, 163, 30,  218, 202, 170, 177, 178, 44,  13,
  129, 28,  189, 247, 185, 221, 135, 121, 60,  194, 65,  102, 212, 182, 19,
  126, 232, 46,  51,  243, 178, 90,  63,  138, 121, 58,  186, 227, 156, 148,
  206, 12,  43,  241, 60,  78,  66,  26,  213, 67,  183, 194, 41,  213, 58,
  189, 233, 78,  35,  115, 87,  243, 205, 120, 251, 86,  101, 228, 246, 75,
  234, 112, 253, 206, 113, 97,  41,  211, 121, 72,  43,  244, 168, 158, 110,
  61,  240, 31,  248, 179, 239, 237, 166, 124, 43,  204, 141, 199, 167, 89,
  40,  164, 27,  173, 23,  254, 182, 234, 230, 115, 127, 107, 62,  55,  81,
  97,  153, 194, 125, 170, 51,  164, 142, 177, 98,  23,  119, 89,  233, 6,
  158, 135, 183, 53,  74,  60,  113, 69,  243, 120, 253, 215, 119, 109, 127,
  130, 227, 72,  87,  230, 82,  170, 241, 169, 100, 191, 191, 234, 240, 52,
  13,  172, 150, 179, 126, 26,  252, 117, 70,  193, 103, 123, 183, 185, 64,
  71,  33,  212, 154, 90,  242, 120, 185, 77,  200, 60,  15,  194, 25,  237,
  45,  203, 40,  248, 188, 154, 220, 30,  31,  72,  206, 108, 69,  243, 188,
  234, 6,   181, 163, 173, 178, 135, 233, 58,  119, 217, 208, 180, 239, 251,
  255, 183, 184, 16,  87,  223, 226, 137, 171, 110, 99,  188, 59,  218, 207,
  157, 223, 120, 127, 160, 162, 164, 141, 166, 95,  65,  44,  128, 42,  103,
  169, 4,   38,  27,  23,  180, 60,  23,  90,  18,  21,  165, 127, 233, 103,
  153, 48,  5,   48,  9,   27,  98,  68,  249, 58,  188, 68,  34,  150, 208,
  31,  141, 5,   128, 192, 63,  242, 65,  232, 95,  16,  68,  232, 103, 243,
  131, 163, 0,   38,  111, 37,  80,  37,  252, 103, 10,  56,  120, 63,  26,
  41,  240, 94,  171, 102, 66,  43,  241, 54,  41,  5,   42,  22,  76,  5,
  128, 25,  127, 92,  129, 2,   115, 135, 72,  63,  23,  25,  9,   88,  155,
  252, 168, 18,  255, 3,   204, 50,  19,  16,  39,  127, 229, 81,  7,   188,
  37,  240, 150, 16,  250, 217, 196, 63,  242, 155, 22,  56,  204, 244, 115,
  145, 17,  123, 106, 242, 69,  170, 236, 115, 94,  44,  112, 78,  190, 40,
  136, 42,  120, 173, 48,  167, 214, 166, 53,  61,  200, 166, 168, 29,  222,
  155, 209, 71,  207, 88,  31,  76,  166, 159, 139, 140, 160, 57,  247, 139,
  7,   157, 186, 94,  0,   139, 23,  17,  246, 89,  20,  161, 159, 105, 245,
  51,  239, 176, 94,  134, 43,  133, 103, 139, 148, 195, 254, 60,  24,  178,
  65,  55,  30,  102, 150, 220, 81,  27,  128, 220, 47,  86,  173, 17,  70,
  3,   50,  68,  221, 218, 86,  108, 43,  136, 232, 0,   127, 42,  108, 89,
  222, 106, 7,   208, 22,  123, 148, 25,  15,  79,  68,  223, 222, 165, 133,
  40,  211, 79,  132, 80,  6,   79,  68,  225, 25,  142, 135, 41,  83,  237,
  122, 16,  88,  136, 122, 118, 74,  233, 96,  137, 242, 145, 201, 147, 123,
  81,  47,  2,   58,  176, 86,  46,  218, 56,  72,  137, 180, 45,  153, 110,
  17,  37,  210, 215, 212, 22,  0,   201, 134, 131, 166, 168, 251, 22,  148,
  214, 105, 62,  32,  250, 75,  48,  245, 221, 163, 104, 216, 77,  125, 211,
  127, 171, 77,  222, 52,  220, 63,  40,  243, 105, 240, 24,  75,  104, 76,
  59,  61,  231, 144, 169, 30,  118, 154, 205, 225, 156, 169, 53,  250, 75,
  24,  91,  1,   75,  127, 255, 104, 180, 49,  215, 183, 2,   152, 218, 78,
  149, 181, 80,  119, 228, 94,  181, 122, 172, 212, 2,   247, 5,   215, 181,
  74,  232, 244, 224, 221, 250, 215, 64,  165, 179, 172, 121, 114, 78,  229,
  176, 36,  172, 35,  193, 248, 141, 251, 195, 55,  255, 61,  216, 15,  149,
  108, 254, 180, 196, 237, 209, 99,  219, 135, 238, 253, 137, 246, 110, 73,
  27,  70,  92,  217, 139, 143, 246, 132, 191, 81,  226, 192, 213, 242, 88,
  248, 148, 57,  113, 141, 232, 177, 204, 224, 223, 144, 226, 82,  227, 67,
  133, 137, 22,  133, 7,   245, 186, 159, 163, 178, 130, 237, 64,  204, 197,
  200, 142, 126, 46,  50,  208, 207, 69,  230, 135, 96,  159, 145, 107, 195,
  212, 85,  239, 233, 10,  0,   45,  57,  158, 77,  242, 31,  23,  169, 81,
  113, 54,  234, 78,  186, 2,   128, 167, 103, 228, 89,  15,  209, 29,  59,
  255, 19,  43,  244, 144, 41,  128, 146, 49,  238, 17,  82,  126, 41,  64,
  129, 160, 145, 141, 8,   136, 17,  228, 101, 172, 134, 24,  253, 70,  217,
  153, 255, 163, 195, 154, 167, 55,  119, 228, 177, 191, 107, 219, 210, 77,
  246, 218, 255, 181, 223, 54,  111, 228, 171, 15,  246, 216, 122, 217, 124,
  195, 165, 185, 157, 38,  83,  252, 237, 206, 205, 27,  185, 246, 135, 169,
  53,  219, 108, 187, 180, 209, 1,   88,  183, 213, 74,  148, 82,  120, 199,
  255, 193, 203, 27,  79,  112, 233, 86,  241, 191, 21,  139, 69,  249, 0,
  56,  109, 171, 241, 203, 251, 239, 149, 130, 110, 217, 220, 22,  75,  61,
  32,  43,  54,  44,  239, 138, 128, 253, 235, 166, 241, 203, 159, 87,  254,
  72,  82,  75,  44,  208, 194, 46,  165, 188, 112, 225, 248, 101, 211, 113,
  68,  143, 194, 204, 218, 185, 46,  26,  75,  214, 44,  11,  223, 204, 164,
  220, 123, 223, 232, 229, 234, 215, 88,  22,  239, 241, 73,  106, 252, 234,
  205, 19,  198, 46,  55,  236, 179, 127, 41,  61,  32,  5,   77,  51,  83,
  5,   20,  179, 171, 58,  153, 75,  202, 184, 171, 62,  28,  185, 220, 115,
  119, 21,  4,   24,  5,   253, 33,  228, 125, 147, 3,   180, 185, 255, 171,
  19,  196, 58,  69,  134, 182, 77,  181, 128, 88,  182, 84,  116, 145, 134,
  90,  43,  127, 189, 116, 220, 247, 103, 110, 32,  41,  146, 162, 163, 77,
  226, 205, 187, 73,  243, 192, 43,  39,  23,  151, 222, 55,  120, 3,   29,
  24,  102, 166, 68,  27,  200, 29,  253, 22,  126, 191, 252, 146, 17,  203,
  29,  191, 95,  226, 150, 206, 79,  182, 139, 27,  247, 223, 118, 196, 95,
  192, 246, 108, 1,   64,  227, 96,  111, 168, 110, 118, 138, 12,  166, 150,
  244, 208, 210, 123, 128, 90,  240, 251, 21,  111, 140, 85,  190, 191, 110,
  223, 163, 196, 202, 194, 65,  208, 31,  96,  232, 218, 107, 122, 156, 242,
  251, 173, 207, 12,  248, 129, 30,  151, 210, 77,  181, 49,  1,   83,  78,
  167, 82,  172, 40,  39,  241, 225, 71,  198, 40,  127, 60,  248, 210, 121,
  187, 25,  9,   37,  224, 32,  60,  249, 14,  80,  126, 158, 27,  101, 227,
  195, 215, 124, 97,  8,   80,  198, 65,  145, 233, 166, 61,  196, 220, 93,
  35,  26,  208, 4,   37,  247, 252, 238, 246, 187, 255, 24,  153, 124, 250,
  212, 203, 7,   29,  179, 52,  173, 141, 248, 148, 72,  79,  206, 127, 250,
  231, 100, 249, 184, 228, 187, 71,  30,  218, 243, 31,  106, 122, 212, 131,
  160, 135, 67,  246, 208, 53,  32,  158, 192, 109, 5,   170, 211, 9,   88,
  247, 133, 235, 46,  187, 113, 195, 118, 240, 248, 219, 168, 80,  231, 253,
  249, 233, 247, 239, 44,  59,  232, 224, 165, 96,  129, 176, 91,  123, 65,
  91,  213, 209, 134, 224, 255, 199, 59,  47,  239, 62,  217, 189, 160, 52,
  141, 135, 183, 191, 253, 237, 155, 11,  135, 29,  49,  71,  137, 56,  161,
  24,  60,  23,  1,   198, 53,  116, 105, 125, 209, 52,  131, 31,  127, 255,
  57,  208, 9,   65,  6,   151, 243, 106, 47,  148, 103, 134, 113, 117, 177,
  125, 233, 182, 235, 119, 155, 136, 191, 115, 36,  132, 3,   230, 152, 165,
  195, 80,  167, 214, 222, 253, 225, 93,  40,  28,  129, 152, 148, 128, 208,
  199, 213, 30,  249, 90,  118, 157, 219, 125, 14,  165, 84,  123, 211, 168,
  135, 69,  233, 233, 135, 132, 64,  182, 6,   108, 25,  186, 167, 225, 64,
  208, 107, 74,  248, 31,  184, 23,  162, 78,  128, 97,  176, 39,  36,  63,
  12,  46,  194, 61,  70,  147, 128, 4,   215, 107, 222, 129, 149, 97,  83,
  176, 186, 120, 45,  2,   6,   35,  170, 88,  33,  120, 29,  202, 173, 202,
  206, 192, 174, 70,  130, 163, 21,  151, 87,  146, 207, 245, 6,   66,  120,
  29,  26,  134, 4,   148, 58,  85,  96,  252, 164, 184, 103, 76,  64,  50,
  30,  110, 7,   57,  104, 250, 32,  211, 65,  167, 21,  8,   189, 1,   100,
  34,  40,  142, 168, 40,  20,  253, 245, 30,  246, 211, 138, 195, 103, 104,
  15,  54,  12,  44,  112, 172, 203, 192, 52,  119, 204, 134, 58,  2,   95,
  212, 93,  182, 6,   136, 65,  158, 64,  239, 33,  112, 95,  76,  231, 191,
  152, 2,   225, 26,  247, 15,  146, 14,  96,  87,  61,  20,  217, 233, 64,
  105, 136, 227, 12,  23,  176, 72,  19,  121, 174, 101, 2,   201, 27,  154,
  80,  13,  192, 176, 67,  223, 178, 96,  195, 10,  8,   181, 45,  169, 100,
  17,  71,  80,  17,  205, 195, 178, 32,  52,  11,  131, 239, 113, 70,  101,
  24,  12,  233, 158, 122, 254, 64,  83,  169, 87,  142, 55,  48,  184, 245,
  141, 62,  234, 149, 168, 220, 84,  118, 50,  236, 76,  144, 138, 38,  97,
  83,  128, 12,  184, 4,   81,  176, 35,  59,  61,  238, 29,  101, 84,  98,
  90,  43,  186, 34,  145, 32,  235, 175, 167, 157, 48,  231, 2,   85,  19,
  173, 165, 153, 204, 122, 128, 121, 90,  73,  203, 66,  90,  220, 211, 13,
  198, 39,  166, 139, 98,  178, 159, 243, 222, 208, 234, 150, 141, 93,  253,
  195, 93,  134, 210, 162, 149, 15,  17,  85,  2,   218, 18,  99,  128, 130,
  89,  139, 195, 172, 245, 182, 196, 244, 203, 89,  13,  144, 243, 66,  56,
  7,   44,  7,   79,  113, 89,  175, 193, 212, 208, 24,  32,  193, 180, 157,
  2,   0,   57,  222, 174, 115, 95,  12,  84,  107, 231, 99,  116, 11,  117,
  200, 41,  33,  80,  80,  201, 22,  165, 193, 105, 33,  185, 133, 176, 61,
  31,  37,  118, 254, 118, 136, 54,  39,  35,  75,  138, 136, 236, 251, 91,
  4,   196, 67,  93,  167, 198, 176, 59,  164, 98,  113, 186, 182, 244, 65,
  210, 186, 80,  172, 232, 12,  115, 187, 64,  163, 64,  211, 2,   245, 244,
  128, 70,  156, 97,  159, 28,  36,  111, 137, 69,  50,  34,  61,  160, 141,
  74,  162, 123, 117, 157, 253, 201, 1,   99,  127, 96,  90,  186, 255, 27,
  52,  139, 175, 197, 189, 0,   152, 131, 29,  154, 25,  153, 60,  153, 132,
  21,  34,  228, 58,  122, 31,  144, 144, 129, 42,  87,  8,   179, 135, 46,
  203, 38,  1,   113, 248, 17,  210, 19,  104, 80,  52,  135, 53,  207, 230,
  222, 255, 242, 111, 141, 232, 177, 48,  228, 166, 166, 191, 20,  218, 155,
  102, 172, 166, 251, 187, 120, 198, 192, 48,  174, 127, 6,   211, 91,  238,
  193, 23,  194, 156, 2,   43,  32,  105, 12,  75,  29,  72,  245, 16,  169,
  113, 51,  80,  226, 204, 178, 47,  24,  143, 67,  186, 25,  32,  182, 74,
  67,  145, 150, 161, 18,  84,  52,  243, 14,  99,  109, 10,  15,  236, 73,
  214, 54,  60,  120, 1,   113, 76,  244, 173, 126, 105, 11,  26,  90,  245,
  234, 2,   251, 187, 216, 252, 223, 243, 202, 107, 59,  21,  1,   107, 90,
  79,  75,  110, 126, 184, 195, 146, 13,  164, 203, 216, 27,  124, 233, 110,
  103, 206, 97,  168, 218, 227, 144, 3,   18,  62,  79,  196, 46,  225, 138,
  157, 157, 216, 254, 42,  38,  0,   86,  80,  56,  32,  182, 40,  0,   0,
  112, 216, 0,   157, 1,   42,  130, 1,   139, 1,   62,  49,  20,  135, 66,
  162, 33,  11,  167, 182, 38,  16,  1,   130, 89,  91,  188, 99,  255, 120,
  56,  151, 36,  164, 87,  13,  227, 88,  81,  154, 213, 202, 243, 201, 250,
  215, 106, 182, 59,  233, 159, 144, 191, 149, 191, 47,  149, 207, 236, 63,
  127, 127, 120, 191, 204, 251, 170, 246, 85,  215, 30,  89,  158, 95,  250,
  199, 248, 255, 238, 95,  185, 31,  226, 255, 255, 255, 255, 250, 145, 254,
  203, 218, 15,  234, 95,  247, 222, 224, 31,  162, 63,  223, 255, 182, 127,
  125, 255, 107, 254, 75,  255, 255, 255, 79,  189, 222, 162, 63,  168, 255,
  192, 245, 17,  252, 163, 250, 175, 251, 95,  240, 127, 189, 63,  60,  31,
  201, 127, 170, 251, 28,  255, 29,  251, 85,  238, 9,   253, 103, 251, 191,
  254, 79,  91,  175, 96,  143, 64,  15,  218, 15,  86,  239, 245, 159, 248,
  191, 206, 255, 220, 255, 255, 244, 89,  251, 61,  255, 131, 252, 167, 239,
  255, 208, 175, 243, 127, 235, 255, 243, 191, 62,  190, 64,  61,  0,   61,
  0,   61,  61,  251, 77,  253, 239, 211, 255, 141, 191, 135, 252, 192, 254,
  203, 228, 83,  237, 127, 189, 254, 94,  127, 123, 209, 71,  249, 95,  219,
  207, 213, 127, 124, 253, 205, 252, 195, 246, 189, 241, 159, 128, 47,  226,
  31,  203, 127, 203, 126, 86,  255, 128, 253, 181, 221, 85,  232, 59,  239,
  103, 216, 191, 215, 253, 210, 124, 76,  125, 183, 251, 95,  76,  254, 206,
  127, 205, 247, 0,   253, 111, 255, 105, 198, 139, 233, 30,  192, 191, 204,
  255, 188, 127, 201, 254, 253, 248, 205, 244, 207, 253, 23,  253, 207, 242,
  31,  153, 94,  212, 255, 58,  255, 39,  255, 131, 253, 23,  250, 95,  144,
  191, 230, 95,  214, 63,  224, 127, 137, 252, 169, 249, 182, 246, 87,  251,
  85,  236, 75,  250, 253, 255, 43,  243, 252, 123, 225, 107, 65,  121, 220,
  59,  235, 64,  76,  53,  31,  99,  3,   255, 110, 222, 19,  98,  122, 18,
  120, 247, 194, 214, 130, 243, 189, 192, 114, 224, 203, 251, 51,  158, 206,
  240, 86,  63,  182, 90,  222, 62,  40,  141, 11,  231, 70,  122, 81,  143,
  140, 186, 154, 163, 171, 27,  162, 200, 89,  49,  220, 39,  174, 1,   127,
  205, 229, 6,   248, 40,  26,  27,  171, 130, 12,  31,  75,  129, 89,  238,
  217, 230, 143, 188, 70,  12,  205, 173, 251, 122, 63,  72,  206, 204, 82,
  65,  222, 111, 255, 98,  243, 240, 14,  2,   76,  227, 68,  139, 29,  179,
  241, 193, 218, 239, 45,  67,  221, 176, 204, 7,   62,  176, 57,  111, 248,
  203, 128, 96,  63,  23,  18,  34,  35,  76,  36,  254, 81,  254, 173, 237,
  243, 51,  182, 203, 251, 205, 175, 255, 162, 103, 30,  37,  146, 216, 18,
  19,  121, 221, 81,  120, 46,  207, 188, 146, 50,  80,  38,  157, 46,  96,
  74,  235, 5,   161, 254, 60,  232, 1,   182, 147, 44,  133, 146, 233, 140,
  218, 223, 155, 134, 12,  46,  87,  88,  129, 103, 124, 137, 221, 31,  179,
  59,  152, 227, 56,  29,  103, 114, 44,  193, 255, 95,  222, 117, 23,  246,
  233, 19,  132, 245, 16,  70,  243, 14,  169, 229, 7,   22,  134, 201, 222,
  88,  140, 95,  185, 117, 212, 116, 242, 186, 10,  203, 102, 8,   223, 164,
  109, 175, 66,  76,  119, 122, 103, 16,  67,  49,  70,  6,   142, 11,  68,
  195, 108, 123, 2,   144, 67,  126, 123, 75,  105, 145, 236, 27,  3,   248,
  130, 101, 61,  205, 240, 125, 43,  169, 58,  107, 23,  220, 92,  111, 153,
  162, 59,  244, 132, 34,  224, 138, 76,  3,   82,  88,  231, 158, 2,   23,
  81,  182, 200, 216, 23,  81,  181, 112, 140, 157, 44,  225, 134, 42,  219,
  126, 133, 80,  215, 117, 137, 117, 230, 48,  198, 181, 64,  220, 148, 8,
  252, 173, 187, 86,  100, 237, 68,  184, 212, 151, 6,   10,  118, 93,  102,
  128, 37,  192, 48,  145, 160, 73,  127, 153, 92,  158, 72,  167, 143, 47,
  3,   85,  36,  199, 111, 54,  31,  75,  171, 75,  211, 60,  251, 153, 105,
  43,  128, 80,  128, 103, 226, 160, 140, 170, 84,  243, 126, 251, 181, 58,
  148, 65,  91,  94,  8,   130, 200, 192, 90,  223, 3,   140, 20,  112, 169,
  211, 158, 186, 91,  92,  3,   2,   103, 81,  182, 186, 150, 73,  114, 12,
  139, 119, 110, 37,  163, 193, 216, 191, 238, 33,  15,  35,  147, 239, 140,
  149, 155, 27,  182, 51,  59,  103, 22,  184, 53,  40,  85,  121, 252, 219,
  251, 122, 182, 98,  234, 171, 231, 187, 178, 6,   161, 215, 196, 178, 22,
  58,  188, 177, 44,  162, 75,  206, 161, 88,  162, 70,  163, 30,  109, 227,
  237, 120, 116, 173, 232, 116, 110, 49,  187, 228, 86,  205, 129, 207, 114,
  160, 137, 240, 100, 166, 37,  125, 173, 135, 245, 16,  142, 151, 71,  151,
  240, 224, 0,   120, 177, 25,  150, 27,  180, 126, 35,  10,  188, 2,   119,
  122, 122, 92,  159, 248, 86,  183, 252, 8,   226, 185, 39,  114, 215, 185,
  151, 102, 64,  129, 151, 8,   44,  233, 183, 217, 139, 10,  63,  212, 61,
  37,  19,  228, 236, 125, 153, 170, 241, 102, 21,  76,  215, 189, 94,  9,
  88,  243, 28,  15,  191, 143, 253, 119, 163, 96,  189, 1,   241, 254, 17,
  108, 41,  199, 215, 99,  152, 87,  117, 66,  165, 38,  173, 214, 117, 2,
  199, 86,  84,  2,   20,  73,  5,   92,  37,  88,  225, 74,  183, 237, 17,
  34,  147, 143, 170, 153, 57,  86,  214, 79,  43,  247, 13,  192, 118, 251,
  106, 90,  78,  32,  157, 144, 177, 207, 14,  85,  82,  104, 49,  193, 65,
  95,  192, 84,  38,  132, 129, 142, 217, 237, 234, 209, 132, 235, 24,  8,
  196, 16,  153, 116, 142, 237, 193, 76,  71,  176, 243, 246, 224, 39,  12,
  170, 26,  0,   187, 144, 95,  122, 206, 203, 107, 15,  20,  90,  151, 25,
  61,  107, 157, 43,  69,  215, 57,  94,  26,  40,  84,  22,  131, 75,  155,
  196, 55,  3,   11,  230, 220, 216, 15,  80,  30,  153, 151, 5,   149, 166,
  91,  193, 166, 234, 102, 181, 243, 109, 136, 149, 164, 235, 186, 65,  145,
  64,  172, 92,  108, 35,  18,  213, 183, 164, 37,  149, 224, 164, 174, 110,
  201, 170, 37,  6,   51,  30,  31,  109, 27,  214, 246, 40,  30,  64,  179,
  122, 234, 63,  39,  145, 41,  240, 66,  180, 13,  247, 160, 146, 148, 56,
  120, 171, 85,  106, 22,  191, 175, 56,  141, 0,   200, 5,   50,  164, 60,
  70,  151, 179, 163, 198, 243, 86,  25,  74,  159, 206, 193, 83,  122, 224,
  79,  140, 189, 243, 119, 242, 29,  30,  23,  231, 255, 48,  14,  225, 18,
  170, 121, 169, 206, 63,  28,  179, 224, 164, 200, 143, 127, 110, 147, 0,
  253, 143, 79,  153, 7,   126, 54,  28,  174, 2,   231, 9,   219, 27,  161,
  88,  89,  246, 156, 167, 140, 52,  229, 123, 214, 243, 143, 72,  111, 209,
  98,  109, 84,  51,  26,  72,  35,  181, 61,  127, 28,  171, 124, 244, 225,
  226, 79,  35,  181, 61,  127, 28,  171, 124, 243, 34,  89,  57,  232, 144,
  147, 40,  33,  191, 50,  15,  127, 229, 169, 112, 89,  109, 41,  6,   122,
  63,  99,  211, 230, 230, 65,  6,   201, 109, 178, 4,   250, 209, 218, 15,
  232, 246, 168, 192, 121, 14,  109, 179, 126, 38,  156, 125, 250, 173, 103,
  163, 249, 196, 66,  36,  141, 150, 55,  38,  33,  2,   239, 230, 103, 19,
  0,   56,  89,  233, 206, 238, 176, 48,  144, 105, 20,  163, 107, 47,  22,
  104, 25,  156, 112, 26,  95,  127, 123, 231, 45,  163, 64,  144, 141, 196,
  93,  131, 211, 139, 211, 26,  129, 174, 90,  182, 102, 215, 76,  186, 253,
  61,  60,  22,  140, 134, 201, 127, 170, 126, 81,  211, 160, 41,  133, 79,
  221, 99,  104, 12,  57,  163, 154, 22,  110, 33,  132, 159, 190, 120, 13,
  164, 237, 48,  33,  70,  179, 149, 87,  124, 230, 171, 233, 196, 227, 68,
  6,   32,  94,  154, 178, 199, 220, 51,  25,  202, 4,   117, 119, 239, 243,
  60,  214, 193, 195, 219, 67,  82,  211, 71,  245, 128, 194, 48,  10,  214,
  105, 223, 216, 229, 5,   234, 65,  98,  159, 169, 54,  174, 73,  221, 254,
  227, 68,  62,  171, 28,  92,  8,   231, 140, 129, 242, 78,  129, 254, 148,
  43,  243, 149, 106, 87,  185, 243, 240, 123, 57,  203, 28,  94,  6,   213,
  149, 254, 4,   176, 208, 101, 233, 178, 84,  220, 51,  148, 103, 79,  91,
  210, 76,  244, 58,  39,  48,  177, 247, 13,  14,  17,  239, 59,  156, 148,
  183, 24,  253, 255, 167, 168, 85,  254, 190, 31,  64,  28,  201, 121, 13,
  231, 149, 188, 113, 183, 102, 115, 43,  232, 92,  214, 214, 65,  104, 4,
  186, 31,  18,  14,  223, 18,  107, 233, 248, 144, 222, 172, 144, 248, 246,
  53,  145, 168, 51,  252, 166, 254, 181, 68,  36,  76,  210, 129, 70,  89,
  230, 219, 206, 46,  173, 238, 83,  58,  17,  245, 104, 38,  245, 181, 213,
  121, 241, 163, 177, 160, 41,  42,  132, 47,  23,  26,  13,  231, 195, 37,
  26,  153, 71,  1,   246, 159, 93,  206, 201, 118, 40,  29,  239, 29,  64,
  170, 223, 113, 244, 56,  232, 28,  197, 75,  81,  30,  47,  180, 193, 196,
  155, 250, 8,   73,  155, 215, 225, 227, 114, 36,  230, 251, 183, 103, 206,
  113, 175, 152, 127, 111, 145, 75,  200, 87,  185, 135, 20,  88,  48,  38,
  92,  3,   198, 78,  12,  222, 135, 247, 250, 13,  7,   55,  121, 138, 109,
  23,  29,  149, 2,   205, 82,  236, 134, 208, 32,  176, 224, 138, 184, 182,
  115, 103, 142, 243, 185, 119, 137, 110, 172, 254, 251, 171, 31,  18,  177,
  53,  71,  230, 243, 250, 171, 128, 185, 89,  124, 17,  85,  250, 193, 85,
  12,  228, 110, 56,  190, 110, 73,  11,  238, 135, 215, 216, 81,  186, 99,
  55,  116, 3,   9,   117, 250, 218, 211, 89,  49,  222, 36,  97,  80,  0,
  0,   254, 252, 34,  170, 148, 224, 150, 240, 250, 0,   139, 57,  140, 9,
  118, 14,  2,   172, 96,  233, 16,  211, 213, 124, 55,  164, 39,  160, 50,
  63,  134, 55,  165, 122, 45,  20,  190, 228, 233, 104, 146, 113, 142, 91,
  51,  0,   175, 51,  111, 224, 166, 22,  35,  108, 35,  28,  17,  100, 37,
  251, 151, 204, 102, 245, 14,  84,  177, 211, 17,  243, 104, 182, 18,  230,
  165, 127, 254, 33,  228, 232, 210, 52,  115, 129, 17,  145, 21,  196, 239,
  164, 94,  24,  115, 16,  195, 181, 113, 176, 56,  4,   104, 252, 48,  37,
  76,  90,  199, 149, 59,  189, 241, 254, 85,  253, 59,  214, 208, 73,  9,
  205, 31,  95,  104, 134, 65,  68,  101, 70,  229, 39,  245, 253, 254, 103,
  31,  233, 29,  49,  116, 47,  109, 122, 54,  234, 198, 83,  97,  179, 145,
  8,   156, 233, 145, 229, 235, 42,  163, 92,  252, 176, 224, 96,  184, 220,
  55,  118, 166, 12,  157, 113, 233, 101, 222, 253, 69,  173, 52,  30,  50,
  102, 56,  247, 111, 230, 175, 25,  184, 213, 15,  170, 187, 175, 169, 210,
  235, 92,  76,  0,   175, 152, 204, 185, 141, 229, 227, 109, 37,  126, 66,
  255, 240, 27,  22,  10,  93,  138, 7,   115, 136, 213, 43,  75,  5,   0,
  57,  239, 113, 182, 193, 255, 73,  191, 5,   125, 2,   215, 224, 61,  137,
  103, 206, 40,  200, 30,  243, 6,   38,  152, 210, 159, 14,  113, 205, 143,
  33,  231, 45,  67,  246, 236, 29,  154, 128, 56,  193, 64,  133, 123, 161,
  244, 207, 119, 198, 28,  81,  22,  132, 108, 60,  86,  220, 240, 163, 88,
  219, 149, 221, 181, 175, 22,  133, 192, 178, 61,  209, 45,  34,  163, 180,
  243, 95,  158, 207, 151, 213, 132, 202, 214, 254, 220, 172, 44,  247, 154,
  52,  137, 80,  90,  228, 166, 109, 48,  76,  147, 106, 45,  112, 26,  71,
  155, 89,  44,  1,   112, 99,  149, 221, 133, 254, 104, 35,  151, 22,  145,
  205, 45,  82,  143, 49,  54,  174, 157, 23,  106, 208, 224, 111, 229, 175,
  153, 119, 76,  38,  11,  214, 253, 213, 138, 128, 158, 130, 177, 175, 29,
  104, 251, 8,   113, 129, 29,  238, 213, 26,  49,  91,  186, 46,  239, 223,
  98,  234, 135, 180, 181, 77,  138, 153, 168, 163, 40,  2,   202, 148, 193,
  204, 187, 75,  195, 230, 176, 145, 29,  148, 183, 188, 28,  121, 73,  162,
  150, 18,  83,  119, 113, 9,   93,  92,  106, 30,  66,  206, 207, 185, 17,
  225, 61,  117, 189, 141, 116, 122, 144, 136, 250, 51,  38,  163, 165, 206,
  1,   140, 23,  170, 162, 9,   170, 140, 153, 97,  223, 151, 16,  135, 222,
  161, 125, 104, 60,  100, 13,  137, 222, 157, 163, 232, 15,  20,  167, 210,
  223, 103, 156, 84,  162, 183, 132, 36,  37,  211, 5,   56,  103, 17,  178,
  93,  140, 32,  158, 199, 229, 119, 86,  152, 108, 137, 231, 124, 194, 67,
  45,  119, 133, 18,  97,  168, 73,  22,  48,  99,  47,  188, 177, 77,  247,
  169, 243, 159, 113, 142, 225, 182, 24,  94,  156, 35,  228, 158, 151, 45,
  192, 215, 117, 205, 228, 31,  125, 100, 1,   110, 244, 122, 144, 183, 74,
  113, 47,  100, 234, 230, 136, 211, 118, 160, 11,  53,  10,  46,  76,  145,
  39,  196, 76,  34,  18,  48,  58,  69,  63,  159, 116, 26,  40,  86,  65,
  67,  12,  123, 255, 183, 26,  181, 127, 1,   210, 109, 11,  69,  24,  170,
  168, 149, 9,   155, 100, 46,  94,  32,  78,  160, 131, 173, 6,   52,  87,
  185, 101, 109, 232, 97,  245, 119, 216, 64,  212, 90,  255, 182, 39,  244,
  40,  221, 188, 158, 177, 7,   8,   26,  156, 82,  199, 192, 191, 83,  85,
  226, 83,  3,   119, 252, 207, 171, 226, 184, 212, 60,  103, 238, 186, 207,
  160, 132, 105, 97,  247, 103, 107, 222, 6,   205, 196, 65,  249, 143, 41,
  42,  13,  109, 250, 254, 147, 183, 253, 129, 94,  41,  228, 188, 250, 177,
  53,  127, 251, 83,  244, 145, 70,  84,  111, 34,  67,  172, 167, 214, 183,
  226, 133, 190, 173, 84,  217, 86,  43,  69,  193, 239, 169, 189, 131, 87,
  113, 118, 236, 97,  70,  55,  76,  111, 178, 253, 151, 242, 248, 54,  29,
  74,  205, 76,  87,  10,  134, 86,  244, 84,  182, 203, 4,   217, 88,  37,
  30,  27,  250, 174, 130, 105, 180, 236, 174, 175, 91,  76,  250, 149, 131,
  112, 46,  21,  186, 149, 9,   185, 31,  162, 11,  15,  87,  105, 216, 162,
  163, 32,  188, 117, 144, 34,  32,  201, 204, 226, 191, 182, 245, 97,  102,
  49,  105, 40,  77,  119, 151, 133, 122, 173, 115, 102, 57,  80,  250, 10,
  64,  1,   28,  140, 24,  20,  245, 38,  134, 186, 113, 4,   78,  120, 175,
  191, 199, 133, 223, 31,  56,  141, 23,  12,  77,  118, 26,  250, 130, 12,
  69,  162, 164, 233, 19,  29,  31,  86,  181, 101, 107, 153, 233, 102, 254,
  159, 179, 194, 51,  200, 113, 243, 146, 250, 248, 120, 19,  153, 116, 154,
  19,  8,   98,  211, 239, 131, 148, 68,  105, 89,  248, 110, 217, 243, 110,
  28,  88,  226, 15,  7,   11,  254, 62,  53,  26,  32,  93,  168, 247, 234,
  118, 239, 225, 165, 34,  3,   167, 168, 40,  86,  137, 71,  189, 3,   143,
  69,  63,  4,   71,  21,  186, 142, 88,  250, 121, 179, 28,  63,  55,  206,
  134, 243, 68,  60,  62,  12,  239, 86,  97,  61,  30,  37,  51,  145, 13,
  162, 117, 122, 101, 136, 161, 107, 227, 30,  168, 154, 204, 94,  62,  93,
  136, 111, 166, 120, 117, 167, 239, 238, 39,  127, 241, 119, 216, 104, 208,
  242, 105, 60,  81,  228, 173, 99,  185, 102, 219, 41,  56,  44,  119, 117,
  160, 170, 108, 253, 166, 159, 158, 174, 251, 135, 220, 191, 139, 155, 201,
  69,  29,  136, 249, 229, 27,  168, 42,  2,   210, 150, 162, 148, 216, 113,
  141, 123, 87,  155, 217, 121, 220, 240, 39,  21,  72,  0,   133, 137, 151,
  52,  149, 114, 131, 14,  184, 122, 93,  177, 33,  66,  209, 225, 193, 27,
  239, 4,   147, 175, 241, 49,  195, 80,  110, 100, 50,  233, 55,  103, 33,
  246, 111, 106, 11,  106, 69,  182, 150, 208, 122, 164, 174, 181, 57,  112,
  154, 7,   156, 75,  143, 207, 16,  235, 128, 121, 96,  128, 56,  229, 53,
  222, 191, 163, 86,  218, 219, 105, 144, 242, 137, 144, 100, 113, 69,  177,
  242, 65,  49,  246, 82,  141, 189, 154, 121, 67,  69,  61,  204, 182, 22,
  194, 236, 203, 211, 127, 196, 250, 91,  148, 116, 91,  1,   159, 226, 236,
  81,  186, 7,   236, 138, 242, 17,  127, 13,  255, 218, 82,  26,  166, 156,
  246, 103, 141, 62,  181, 214, 146, 14,  15,  209, 89,  80,  245, 243, 233,
  182, 177, 121, 172, 133, 123, 71,  31,  2,   162, 166, 106, 224, 96,  158,
  105, 211, 49,  69,  244, 180, 3,   170, 145, 206, 125, 129, 80,  51,  215,
  236, 20,  109, 118, 213, 252, 43,  32,  41,  69,  88,  251, 234, 236, 99,
  155, 168, 237, 183, 70,  247, 50,  44,  234, 6,   8,   60,  243, 191, 45,
  126, 63,  108, 177, 188, 41,  72,  146, 92,  135, 209, 122, 182, 213, 84,
  14,  72,  49,  2,   136, 96,  215, 10,  5,   117, 184, 233, 44,  188, 133,
  1,   125, 73,  254, 202, 17,  103, 150, 221, 20,  172, 95,  20,  190, 52,
  23,  38,  3,   80,  75,  78,  8,   26,  244, 24,  236, 138, 180, 44,  238,
  125, 75,  41,  188, 237, 29,  246, 232, 217, 74,  169, 152, 220, 10,  164,
  149, 39,  32,  230, 122, 252, 70,  222, 114, 192, 251, 55,  13,  59,  38,
  141, 14,  236, 51,  7,   250, 125, 90,  133, 114, 99,  44,  99,  167, 17,
  165, 40,  147, 232, 175, 45,  122, 8,   65,  50,  70,  141, 251, 115, 239,
  252, 219, 245, 243, 203, 216, 0,   5,   197, 105, 6,   64,  210, 101, 214,
  53,  15,  166, 188, 28,  226, 29,  206, 206, 224, 170, 4,   203, 206, 210,
  138, 10,  220, 186, 140, 151, 187, 214, 203, 113, 231, 116, 141, 185, 16,
  176, 96,  108, 5,   163, 62,  55,  197, 132, 210, 146, 186, 155, 20,  153,
  32,  64,  136, 28,  179, 22,  70,  218, 104, 185, 39,  137, 35,  63,  136,
  11,  250, 189, 149, 48,  7,   59,  178, 44,  215, 75,  114, 3,   101, 133,
  233, 29,  155, 64,  133, 65,  33,  223, 156, 30,  92,  43,  77,  237, 154,
  104, 86,  32,  32,  169, 42,  154, 181, 165, 193, 240, 51,  149, 148, 53,
  252, 240, 175, 20,  34,  196, 17,  215, 224, 22,  220, 66,  212, 6,   80,
  223, 55,  209, 45,  246, 173, 125, 125, 227, 19,  108, 231, 210, 48,  13,
  35,  196, 147, 159, 252, 192, 54,  215, 117, 210, 232, 124, 132, 198, 108,
  230, 188, 12,  213, 114, 189, 4,   151, 121, 213, 226, 18,  105, 42,  153,
  67,  1,   131, 200, 123, 253, 98,  163, 3,   206, 189, 245, 224, 136, 126,
  207, 239, 36,  23,  234, 52,  78,  159, 237, 220, 152, 13,  14,  152, 90,
  164, 182, 10,  6,   52,  179, 44,  197, 3,   196, 149, 173, 108, 126, 44,
  126, 229, 188, 136, 69,  231, 137, 226, 177, 125, 138, 217, 145, 216, 29,
  202, 146, 100, 75,  252, 87,  60,  65,  168, 141, 136, 56,  153, 27,  202,
  219, 193, 123, 29,  244, 1,   125, 122, 186, 162, 216, 190, 113, 54,  139,
  199, 190, 97,  61,  50,  59,  220, 14,  195, 19,  208, 2,   255, 248, 47,
  51,  160, 203, 217, 196, 186, 2,   196, 117, 43,  248, 12,  7,   110, 42,
  44,  220, 200, 2,   18,  64,  33,  3,   50,  161, 156, 173, 102, 67,  124,
  220, 36,  15,  1,   83,  176, 66,  223, 101, 239, 103, 34,  162, 237, 217,
  33,  7,   91,  203, 111, 69,  18,  190, 43,  167, 178, 123, 240, 188, 194,
  181, 209, 223, 24,  105, 221, 205, 194, 135, 223, 58,  199, 226, 152, 149,
  188, 140, 148, 217, 36,  214, 185, 27,  222, 12,  193, 139, 149, 105, 254,
  92,  76,  136, 112, 28,  207, 228, 25,  93,  228, 92,  39,  141, 32,  127,
  69,  86,  18,  105, 26,  11,  240, 228, 59,  92,  231, 231, 160, 112, 243,
  220, 39,  151, 213, 147, 205, 68,  106, 195, 196, 216, 247, 173, 244, 150,
  71,  228, 0,   228, 32,  125, 216, 133, 38,  42,  192, 77,  85,  123, 134,
  136, 0,   172, 53,  170, 71,  231, 223, 37,  97,  85,  113, 183, 200, 138,
  206, 47,  77,  56,  86,  203, 222, 223, 25,  100, 113, 89,  239, 230, 45,
  83,  248, 126, 161, 182, 246, 143, 44,  150, 254, 63,  87,  252, 182, 54,
  246, 219, 145, 122, 105, 252, 125, 136, 114, 58,  2,   41,  75,  26,  88,
  243, 20,  231, 16,  211, 173, 49,  188, 30,  76,  78,  225, 220, 88,  224,
  233, 133, 248, 106, 120, 35,  95,  129, 188, 63,  18,  186, 16,  243, 119,
  237, 163, 201, 171, 144, 140, 74,  121, 129, 35,  40,  249, 39,  247, 6,
  26,  156, 88,  215, 254, 165, 187, 107, 78,  222, 245, 29,  234, 79,  231,
  191, 248, 47,  143, 134, 83,  193, 252, 44,  194, 234, 138, 231, 223, 255,
  70,  147, 102, 13,  155, 9,   171, 41,  222, 90,  238, 77,  131, 174, 93,
  225, 27,  85,  133, 43,  115, 243, 144, 182, 210, 51,  119, 130, 127, 9,
  241, 23,  97,  88,  46,  212, 39,  30,  251, 178, 136, 148, 77,  84,  192,
  149, 61,  107, 232, 133, 60,  224, 228, 124, 14,  96,  113, 5,   8,   243,
  81,  128, 115, 198, 149, 1,   11,  212, 49,  48,  126, 59,  77,  183, 236,
  159, 44,  22,  191, 114, 64,  198, 123, 59,  250, 228, 193, 221, 10,  162,
  40,  82,  129, 44,  3,   228, 27,  252, 151, 226, 21,  209, 233, 252, 140,
  133, 21,  111, 159, 157, 107, 211, 238, 71,  236, 125, 126, 253, 2,   207,
  169, 64,  48,  243, 0,   40,  178, 115, 59,  212, 5,   126, 236, 102, 127,
  144, 134, 93,  71,  172, 53,  134, 133, 247, 22,  209, 129, 127, 216, 159,
  225, 184, 112, 97,  111, 170, 70,  199, 171, 138, 23,  226, 234, 221, 255,
  1,   123, 16,  10,  9,   25,  229, 105, 110, 33,  28,  226, 37,  89,  66,
  137, 195, 87,  251, 203, 225, 252, 73,  144, 217, 140, 42,  250, 14,  53,
  125, 127, 187, 150, 43,  153, 236, 61,  184, 172, 103, 254, 109, 69,  107,
  185, 213, 5,   52,  181, 151, 209, 140, 250, 7,   251, 244, 179, 245, 81,
  4,   233, 157, 225, 135, 157, 222, 234, 15,  111, 253, 65,  7,   106, 247,
  225, 120, 246, 101, 138, 184, 215, 250, 109, 74,  65,  105, 224, 244, 122,
  238, 89,  255, 244, 129, 32,  227, 74,  21,  255, 49,  164, 192, 143, 97,
  185, 191, 36,  242, 212, 248, 83,  250, 51,  139, 127, 215, 238, 17,  236,
  116, 39,  246, 142, 23,  243, 60,  172, 197, 128, 243, 225, 207, 216, 174,
  121, 235, 154, 197, 59,  162, 179, 51,  170, 89,  71,  205, 231, 246, 70,
  216, 157, 204, 137, 114, 2,   70,  165, 137, 13,  172, 209, 62,  235, 198,
  121, 189, 111, 150, 219, 60,  147, 98,  166, 60,  137, 56,  21,  83,  20,
  145, 133, 88,  136, 220, 230, 38,  172, 210, 37,  6,   173, 81,  3,   112,
  251, 90,  112, 59,  193, 145, 252, 149, 78,  32,  129, 131, 57,  190, 108,
  203, 78,  94,  107, 100, 105, 230, 110, 227, 63,  174, 154, 90,  212, 79,
  111, 27,  159, 100, 26,  121, 83,  193, 190, 91,  136, 57,  194, 71,  187,
  35,  234, 226, 144, 212, 39,  243, 117, 233, 44,  89,  36,  221, 127, 213,
  40,  22,  196, 36,  186, 221, 240, 133, 27,  113, 19,  139, 226, 206, 70,
  230, 29,  121, 18,  96,  195, 137, 40,  1,   131, 158, 108, 11,  127, 36,
  0,   152, 110, 39,  184, 185, 27,  242, 158, 241, 195, 181, 210, 207, 235,
  228, 30,  5,   70,  89,  244, 240, 157, 145, 64,  198, 54,  61,  243, 101,
  111, 247, 122, 146, 202, 118, 238, 236, 37,  57,  77,  172, 185, 206, 159,
  157, 195, 223, 202, 171, 145, 22,  202, 29,  33,  161, 197, 75,  238, 27,
  90,  112, 75,  109, 64,  248, 207, 89,  170, 184, 147, 6,   116, 191, 64,
  134, 65,  189, 18,  79,  214, 210, 249, 169, 119, 223, 62,  41,  208, 244,
  44,  187, 240, 124, 220, 58,  117, 67,  66,  198, 136, 29,  165, 33,  217,
  2,   76,  31,  115, 128, 131, 86,  204, 195, 0,   174, 100, 97,  47,  192,
  78,  88,  135, 122, 53,  101, 105, 54,  101, 243, 6,   196, 236, 182, 85,
  208, 93,  195, 40,  68,  115, 119, 253, 19,  52,  17,  182, 114, 66,  165,
  202, 66,  212, 168, 41,  210, 40,  96,  251, 241, 231, 156, 83,  146, 167,
  67,  60,  156, 225, 24,  249, 138, 79,  185, 79,  184, 55,  184, 199, 157,
  180, 49,  41,  79,  17,  186, 26,  137, 178, 56,  182, 210, 246, 16,  177,
  179, 70,  10,  70,  123, 255, 33,  111, 223, 242, 50,  62,  252, 108, 107,
  141, 15,  114, 238, 191, 38,  73,  83,  106, 79,  65,  36,  91,  76,  82,
  204, 208, 6,   42,  149, 0,   57,  212, 57,  140, 103, 53,  70,  239, 121,
  85,  240, 78,  134, 232, 53,  243, 31,  243, 195, 174, 193, 91,  81,  144,
  245, 60,  215, 241, 188, 16,  5,   212, 211, 85,  31,  119, 92,  35,  143,
  48,  103, 25,  226, 102, 223, 167, 69,  234, 234, 196, 43,  126, 214, 2,
  85,  16,  85,  139, 7,   129, 58,  23,  78,  73,  118, 135, 195, 119, 241,
  205, 22,  36,  134, 252, 219, 99,  151, 213, 167, 189, 144, 90,  24,  186,
  193, 9,   191, 187, 246, 131, 183, 62,  199, 161, 126, 126, 250, 63,  159,
  28,  113, 143, 45,  183, 92,  169, 69,  190, 13,  119, 156, 118, 203, 51,
  119, 38,  100, 223, 255, 107, 210, 17,  229, 152, 39,  107, 192, 32,  83,
  171, 27,  182, 199, 82,  46,  246, 242, 110, 116, 191, 184, 157, 202, 16,
  156, 21,  108, 110, 158, 108, 227, 188, 149, 114, 97,  227, 166, 233, 11,
  66,  127, 97,  187, 83,  47,  120, 49,  162, 147, 180, 134, 101, 53,  153,
  223, 175, 73,  141, 187, 133, 181, 149, 15,  180, 132, 33,  69,  240, 87,
  225, 81,  63,  230, 50,  143, 219, 60,  202, 55,  46,  177, 219, 72,  93,
  176, 81,  142, 146, 140, 82,  79,  126, 124, 87,  222, 37,  211, 228, 67,
  126, 126, 39,  170, 233, 63,  204, 90,  137, 56,  241, 47,  204, 145, 100,
  40,  9,   51,  140, 118, 16,  93,  72,  207, 231, 226, 233, 14,  133, 63,
  206, 235, 143, 101, 216, 248, 62,  82,  248, 181, 192, 142, 231, 141, 109,
  48,  142, 151, 56,  200, 108, 21,  79,  55,  131, 219, 125, 191, 197, 121,
  129, 43,  31,  64,  207, 124, 136, 79,  199, 99,  222, 175, 149, 213, 75,
  110, 59,  206, 80,  39,  189, 144, 148, 71,  219, 92,  124, 190, 211, 79,
  228, 145, 90,  199, 151, 196, 74,  72,  237, 141, 152, 144, 25,  94,  133,
  196, 212, 44,  138, 34,  17,  76,  166, 216, 250, 231, 149, 59,  244, 82,
  141, 206, 41,  59,  201, 210, 200, 82,  74,  54,  229, 48,  70,  19,  33,
  169, 200, 116, 2,   107, 201, 242, 182, 168, 242, 240, 37,  172, 230, 145,
  44,  196, 158, 18,  39,  126, 102, 98,  17,  68,  80,  200, 176, 142, 212,
  244, 149, 248, 192, 2,   170, 231, 18,  65,  191, 103, 105, 124, 148, 54,
  54,  139, 158, 171, 198, 129, 228, 123, 166, 56,  171, 179, 0,   107, 49,
  147, 193, 204, 3,   241, 82,  141, 151, 181, 35,  80,  39,  244, 152, 127,
  124, 244, 123, 123, 208, 210, 21,  85,  203, 221, 113, 90,  49,  182, 179,
  58,  68,  12,  126, 28,  86,  90,  248, 223, 82,  38,  205, 229, 130, 164,
  201, 125, 152, 246, 201, 163, 214, 104, 64,  97,  92,  1,   92,  72,  115,
  200, 240, 72,  164, 232, 0,   42,  148, 1,   197, 92,  123, 14,  235, 222,
  6,   213, 182, 41,  252, 3,   93,  109, 93,  223, 245, 107, 85,  231, 244,
  175, 107, 142, 197, 16,  37,  129, 1,   95,  205, 198, 4,   14,  206, 254,
  6,   165, 87,  20,  221, 93,  161, 104, 51,  85,  118, 144, 175, 92,  212,
  167, 47,  211, 108, 253, 251, 121, 182, 153, 160, 128, 234, 84,  73,  96,
  73,  71,  237, 87,  184, 18,  205, 228, 41,  79,  245, 134, 84,  66,  175,
  116, 103, 2,   203, 238, 10,  7,   129, 176, 144, 226, 250, 70,  25,  65,
  113, 162, 105, 204, 78,  55,  158, 74,  108, 86,  31,  215, 2,   121, 164,
  56,  84,  205, 207, 161, 153, 94,  233, 22,  126, 126, 49,  123, 23,  148,
  163, 126, 190, 217, 176, 35,  146, 212, 179, 52,  134, 22,  151, 238, 207,
  67,  47,  134, 124, 44,  57,  182, 224, 115, 70,  7,   97,  17,  213, 213,
  85,  63,  150, 171, 71,  109, 69,  23,  96,  62,  142, 36,  154, 32,  22,
  119, 96,  222, 183, 208, 150, 59,  157, 246, 224, 30,  77,  194, 156, 224,
  0,   81,  114, 176, 209, 187, 250, 101, 55,  151, 232, 171, 79,  243, 196,
  224, 251, 54,  50,  108, 89,  170, 24,  6,   181, 123, 151, 113, 125, 190,
  85,  195, 181, 49,  54,  221, 118, 26,  244, 28,  52,  103, 183, 111, 80,
  160, 104, 113, 126, 121, 89,  9,   236, 46,  239, 55,  162, 17,  30,  104,
  57,  92,  45,  125, 121, 250, 113, 65,  141, 25,  132, 175, 31,  227, 172,
  48,  54,  81,  72,  103, 245, 251, 145, 223, 206, 179, 239, 214, 24,  140,
  20,  191, 69,  118, 190, 169, 188, 208, 174, 229, 120, 53,  20,  94,  104,
  101, 161, 201, 89,  75,  201, 250, 142, 250, 79,  235, 127, 184, 35,  11,
  77,  77,  47,  164, 73,  246, 92,  163, 68,  238, 139, 210, 2,   97,  215,
  190, 1,   186, 179, 29,  55,  131, 195, 116, 145, 244, 218, 151, 31,  141,
  190, 65,  159, 88,  143, 243, 215, 205, 150, 230, 18,  41,  93,  17,  232,
  131, 175, 83,  106, 249, 179, 136, 207, 0,   139, 25,  47,  12,  126, 198,
  57,  219, 48,  231, 8,   187, 119, 248, 86,  14,  179, 70,  158, 162, 43,
  61,  46,  69,  18,  160, 224, 249, 234, 105, 52,  211, 54,  132, 224, 155,
  72,  131, 192, 165, 236, 103, 182, 183, 127, 228, 76,  86,  43,  181, 242,
  158, 127, 225, 64,  103, 65,  28,  13,  88,  255, 218, 57,  5,   218, 207,
  162, 195, 144, 3,   102, 90,  111, 60,  38,  71,  49,  98,  166, 60,  161,
  245, 71,  239, 123, 144, 62,  30,  225, 108, 41,  249, 196, 95,  149, 239,
  184, 202, 12,  40,  34,  107, 81,  238, 39,  167, 177, 52,  145, 164, 109,
  210, 38,  81,  8,   242, 227, 21,  180, 242, 14,  161, 81,  42,  231, 226,
  19,  4,   139, 215, 90,  24,  223, 153, 110, 24,  8,   127, 34,  193, 218,
  62,  5,   218, 164, 216, 56,  66,  55,  9,   65,  197, 159, 29,  170, 81,
  83,  32,  134, 229, 179, 7,   85,  17,  134, 8,   206, 6,   45,  46,  229,
  143, 197, 102, 59,  83,  231, 230, 41,  138, 49,  41,  201, 160, 84,  33,
  120, 147, 166, 32,  101, 191, 144, 171, 253, 190, 91,  54,  120, 73,  72,
  230, 179, 248, 188, 31,  13,  113, 109, 40,  1,   170, 86,  213, 68,  219,
  210, 14,  111, 39,  182, 3,   74,  122, 52,  34,  229, 118, 95,  4,   199,
  141, 155, 228, 99,  176, 95,  107, 207, 97,  246, 16,  240, 19,  96,  172,
  86,  33,  223, 20,  131, 57,  49,  163, 149, 111, 123, 170, 231, 9,   170,
  59,  78,  236, 238, 18,  68,  227, 85,  222, 11,  144, 126, 71,  126, 161,
  139, 93,  80,  120, 7,   250, 74,  132, 152, 196, 192, 85,  214, 158, 117,
  156, 227, 140, 37,  223, 82,  167, 232, 81,  242, 71,  63,  9,   83,  231,
  49,  94,  79,  195, 157, 37,  178, 79,  25,  255, 179, 174, 107, 139, 242,
  37,  184, 247, 192, 126, 140, 23,  144, 252, 104, 26,  100, 19,  184, 237,
  147, 26,  128, 63,  135, 232, 206, 171, 196, 3,   138, 233, 194, 102, 42,
  152, 128, 22,  16,  29,  137, 156, 38,  214, 71,  215, 112, 105, 227, 19,
  151, 39,  247, 220, 131, 144, 104, 216, 39,  160, 92,  184, 115, 211, 230,
  13,  45,  95,  104, 231, 82,  0,   82,  57,  252, 18,  30,  2,   29,  142,
  45,  44,  228, 60,  85,  241, 154, 18,  54,  152, 229, 39,  12,  108, 47,
  242, 99,  161, 36,  174, 119, 131, 94,  98,  100, 228, 36,  29,  112, 97,
  113, 158, 54,  183, 101, 178, 181, 193, 29,  171, 135, 151, 90,  37,  119,
  71,  252, 175, 189, 24,  246, 87,  53,  204, 242, 74,  133, 135, 107, 69,
  112, 107, 236, 168, 2,   109, 104, 229, 141, 150, 177, 220, 249, 27,  182,
  3,   6,   183, 46,  137, 117, 48,  0,   191, 245, 158, 1,   211, 11,  178,
  251, 182, 253, 246, 163, 146, 103, 252, 244, 207, 114, 12,  200, 86,  100,
  105, 245, 95,  250, 4,   150, 62,  246, 70,  182, 226, 193, 211, 170, 161,
  160, 119, 55,  55,  163, 18,  135, 229, 69,  217, 30,  215, 218, 223, 110,
  149, 130, 164, 218, 48,  26,  125, 229, 38,  111, 77,  223, 229, 108, 117,
  49,  163, 150, 226, 225, 121, 236, 78,  214, 43,  200, 156, 62,  190, 237,
  14,  189, 15,  60,  35,  60,  47,  52,  71,  163, 125, 157, 180, 137, 190,
  36,  179, 54,  66,  230, 72,  188, 113, 23,  104, 119, 233, 40,  56,  197,
  251, 0,   216, 159, 195, 119, 243, 150, 33,  32,  54,  54,  94,  56,  170,
  163, 169, 32,  60,  212, 81,  140, 110, 157, 174, 205, 201, 42,  119, 76,
  142, 120, 60,  168, 135, 108, 140, 102, 198, 133, 234, 247, 34,  95,  166,
  37,  219, 66,  110, 87,  16,  167, 140, 246, 100, 92,  223, 193, 161, 149,
  89,  40,  128, 42,  12,  187, 161, 127, 118, 153, 16,  90,  81,  177, 139,
  177, 226, 54,  184, 107, 84,  226, 121, 225, 110, 199, 211, 251, 215, 217,
  44,  124, 18,  141, 62,  43,  253, 79,  32,  16,  197, 201, 63,  85,  163,
  116, 7,   51,  207, 26,  4,   49,  138, 242, 166, 199, 105, 109, 76,  226,
  100, 96,  44,  128, 74,  156, 145, 124, 208, 153, 139, 176, 236, 103, 3,
  116, 140, 210, 175, 137, 12,  181, 252, 115, 151, 235, 75,  247, 107, 210,
  142, 206, 127, 199, 246, 228, 24,  165, 245, 3,   246, 232, 98,  202, 44,
  128, 138, 173, 227, 218, 49,  10,  14,  130, 255, 246, 81,  151, 59,  207,
  210, 197, 42,  184, 255, 115, 68,  242, 82,  151, 155, 99,  177, 72,  123,
  227, 125, 241, 191, 60,  251, 95,  211, 129, 226, 183, 193, 29,  49,  156,
  74,  209, 246, 44,  106, 216, 57,  133, 41,  137, 122, 198, 67,  153, 17,
  188, 49,  152, 56,  213, 210, 123, 201, 119, 218, 0,   192, 89,  203, 234,
  49,  176, 152, 70,  89,  184, 6,   104, 221, 72,  143, 187, 199, 237, 24,
  94,  52,  87,  54,  88,  176, 221, 97,  112, 242, 154, 246, 124, 62,  23,
  69,  215, 75,  48,  200, 245, 238, 74,  93,  150, 246, 3,   35,  183, 160,
  145, 49,  144, 87,  129, 220, 202, 126, 252, 123, 113, 51,  77,  145, 165,
  151, 120, 169, 51,  131, 121, 84,  120, 104, 234, 183, 177, 141, 155, 243,
  218, 36,  223, 31,  206, 199, 227, 147, 14,  29,  4,   248, 73,  112, 129,
  85,  14,  176, 211, 172, 91,  233, 131, 218, 54,  255, 165, 211, 85,  252,
  249, 44,  196, 138, 35,  183, 49,  124, 90,  143, 164, 244, 150, 210, 104,
  78,  68,  157, 235, 49,  23,  241, 163, 27,  228, 251, 72,  75,  134, 235,
  48,  64,  31,  185, 91,  186, 106, 11,  38,  132, 153, 82,  144, 167, 151,
  255, 130, 1,   115, 203, 202, 159, 158, 106, 38,  139, 238, 104, 186, 150,
  65,  240, 30,  168, 198, 17,  140, 150, 75,  252, 111, 44,  158, 169, 217,
  190, 149, 229, 151, 80,  212, 252, 142, 78,  134, 82,  144, 237, 110, 90,
  91,  64,  52,  239, 57,  245, 183, 154, 14,  220, 26,  238, 170, 107, 180,
  235, 213, 62,  154, 169, 210, 153, 6,   132, 146, 31,  76,  89,  90,  64,
  255, 93,  80,  31,  144, 153, 253, 132, 95,  141, 73,  3,   58,  124, 112,
  129, 170, 159, 101, 193, 203, 131, 38,  246, 118, 92,  68,  1,   58,  177,
  182, 113, 100, 148, 3,   206, 213, 166, 66,  148, 203, 42,  122, 153, 19,
  179, 149, 206, 42,  68,  94,  95,  198, 54,  33,  193, 70,  118, 70,  43,
  219, 137, 251, 245, 70,  214, 126, 64,  46,  224, 81,  135, 191, 175, 58,
  178, 101, 232, 176, 82,  103, 69,  46,  77,  146, 116, 0,   47,  105, 236,
  25,  161, 41,  133, 153, 146, 224, 91,  188, 156, 134, 175, 251, 5,   157,
  117, 133, 197, 51,  245, 197, 110, 73,  243, 4,   247, 192, 98,  106, 125,
  141, 119, 231, 11,  26,  82,  31,  122, 8,   241, 250, 180, 5,   129, 165,
  53,  162, 100, 142, 18,  43,  118, 73,  68,  2,   165, 10,  215, 166, 124,
  212, 61,  148, 240, 113, 246, 77,  185, 191, 193, 245, 90,  240, 81,  57,
  126, 227, 58,  36,  190, 84,  157, 144, 124, 113, 62,  145, 49,  184, 167,
  218, 100, 134, 63,  218, 82,  234, 158, 211, 3,   108, 86,  120, 152, 182,
  26,  60,  46,  74,  198, 255, 42,  175, 97,  122, 224, 239, 27,  168, 33,
  131, 173, 168, 147, 227, 240, 16,  32,  203, 59,  64,  132, 222, 41,  43,
  137, 47,  20,  145, 83,  66,  35,  255, 33,  111, 84,  241, 9,   7,   85,
  20,  12,  247, 79,  120, 238, 169, 127, 116, 184, 14,  38,  26,  101, 126,
  108, 237, 45,  179, 79,  253, 11,  134, 4,   26,  167, 59,  7,   180, 120,
  215, 205, 99,  71,  87,  121, 102, 85,  95,  33,  130, 28,  188, 21,  146,
  5,   115, 183, 117, 13,  229, 130, 47,  232, 160, 86,  207, 210, 56,  149,
  186, 89,  73,  212, 20,  125, 197, 128, 154, 192, 82,  119, 173, 243, 220,
  94,  21,  128, 141, 130, 74,  237, 182, 11,  87,  244, 143, 74,  203, 144,
  105, 186, 178, 193, 230, 89,  196, 159, 156, 39,  243, 243, 232, 66,  8,
  77,  196, 53,  246, 95,  43,  205, 110, 62,  206, 80,  231, 221, 146, 171,
  214, 227, 67,  112, 187, 183, 1,   4,   251, 253, 196, 78,  40,  31,  42,
  4,   19,  60,  136, 43,  195, 57,  136, 112, 140, 81,  199, 7,   175, 76,
  108, 74,  215, 2,   118, 85,  183, 188, 20,  246, 62,  61,  38,  136, 44,
  64,  53,  15,  170, 156, 0,   241, 105, 170, 149, 179, 249, 223, 95,  34,
  191, 130, 186, 63,  229, 11,  238, 229, 32,  110, 150, 3,   178, 202, 235,
  213, 91,  29,  146, 23,  115, 53,  98,  175, 188, 255, 78,  52,  62,  181,
  49,  19,  183, 168, 165, 47,  235, 203, 51,  64,  136, 207, 151, 186, 178,
  160, 239, 48,  49,  191, 111, 206, 12,  140, 10,  157, 97,  136, 26,  224,
  252, 250, 158, 116, 58,  110, 187, 212, 186, 194, 114, 221, 77,  232, 148,
  224, 74,  64,  196, 106, 27,  163, 6,   238, 137, 200, 125, 85,  72,  107,
  184, 147, 115, 126, 99,  13,  29,  136, 181, 255, 165, 192, 59,  163, 125,
  64,  8,   54,  250, 183, 242, 3,   46,  132, 174, 91,  187, 142, 16,  139,
  102, 212, 227, 34,  47,  230, 13,  16,  145, 222, 112, 65,  131, 197, 217,
  142, 178, 226, 186, 225, 157, 0,   161, 199, 64,  46,  34,  55,  136, 118,
  136, 108, 166, 218, 72,  89,  188, 147, 39,  232, 203, 231, 231, 76,  230,
  195, 255, 123, 8,   201, 61,  95,  109, 211, 194, 13,  129, 247, 77,  26,
  195, 193, 251, 235, 155, 158, 160, 243, 131, 166, 219, 183, 6,   206, 14,
  171, 22,  143, 215, 175, 156, 57,  239, 159, 195, 250, 152, 238, 156, 110,
  203, 169, 186, 215, 41,  203, 10,  8,   29,  100, 49,  240, 110, 21,  198,
  73,  85,  248, 61,  50,  30,  18,  54,  87,  162, 224, 60,  222, 16,  64,
  58,  51,  97,  75,  78,  194, 182, 47,  138, 29,  171, 165, 189, 254, 66,
  231, 23,  13,  171, 227, 156, 135, 74,  216, 204, 162, 213, 209, 102, 228,
  210, 221, 50,  33,  249, 110, 133, 48,  117, 51,  71,  169, 91,  34,  208,
  1,   246, 62,  134, 234, 17,  184, 88,  8,   22,  223, 254, 123, 133, 184,
  169, 223, 45,  214, 9,   242, 221, 200, 241, 194, 129, 169, 255, 130, 172,
  55,  148, 144, 233, 117, 110, 13,  177, 123, 101, 169, 178, 157, 123, 62,
  181, 70,  205, 10,  237, 147, 126, 33,  88,  179, 229, 188, 150, 180, 206,
  141, 182, 22,  69,  223, 195, 40,  188, 177, 36,  190, 131, 248, 115, 6,
  174, 124, 70,  136, 129, 129, 181, 177, 13,  218, 62,  143, 57,  67,  178,
  121, 129, 58,  79,  177, 194, 163, 100, 149, 22,  216, 240, 246, 255, 31,
  60,  63,  150, 19,  187, 232, 184, 138, 191, 181, 198, 114, 9,   5,   205,
  81,  242, 223, 217, 3,   121, 190, 131, 10,  216, 37,  121, 123, 157, 147,
  137, 202, 76,  113, 90,  64,  115, 126, 13,  15,  162, 97,  176, 254, 82,
  251, 58,  29,  76,  115, 7,   114, 104, 156, 33,  180, 179, 16,  63,  114,
  59,  156, 49,  146, 145, 131, 148, 124, 206, 114, 83,  51,  87,  105, 148,
  218, 58,  58,  204, 253, 129, 183, 50,  180, 169, 45,  13,  78,  56,  248,
  5,   5,   131, 96,  215, 195, 10,  242, 249, 70,  77,  0,   243, 128, 41,
  29,  3,   200, 72,  62,  133, 222, 152, 24,  50,  16,  54,  128, 78,  24,
  65,  111, 255, 242, 226, 42,  161, 109, 247, 255, 138, 13,  218, 176, 106,
  199, 250, 170, 192, 237, 128, 204, 120, 14,  43,  223, 214, 214, 195, 174,
  41,  61,  180, 230, 158, 173, 228, 53,  236, 66,  201, 209, 71,  110, 145,
  175, 44,  45,  95,  254, 54,  238, 96,  108, 223, 219, 79,  172, 2,   28,
  42,  209, 172, 11,  114, 211, 37,  176, 44,  172, 230, 189, 21,  120, 91,
  245, 102, 214, 177, 250, 248, 96,  235, 212, 255, 98,  70,  129, 196, 144,
  167, 182, 11,  5,   135, 107, 245, 128, 145, 99,  127, 207, 97,  94,  210,
  45,  200, 162, 215, 155, 214, 231, 240, 145, 235, 191, 128, 154, 86,  133,
  145, 206, 116, 84,  170, 140, 17,  217, 169, 66,  144, 200, 32,  153, 4,
  28,  43,  6,   135, 243, 0,   107, 10,  131, 35,  67,  249, 128, 53,  133,
  65,  232, 178, 64,  182, 5,   69,  168, 136, 218, 183, 208, 47,  127, 0,
  2,   166, 111, 198, 142, 237, 190, 39,  252, 78,  201, 161, 103, 46,  26,
  203, 196, 16,  48,  4,   68,  223, 151, 243, 159, 235, 33,  245, 226, 101,
  72,  154, 0,   237, 142, 121, 143, 245, 178, 236, 74,  116, 64,  253, 35,
  221, 76,  204, 161, 128, 44,  113, 6,   255, 110, 105, 229, 93,  141, 86,
  106, 231, 131, 23,  227, 208, 219, 83,  149, 181, 41,  10,  165, 246, 125,
  187, 52,  127, 70,  105, 134, 100, 105, 55,  47,  203, 180, 184, 122, 34,
  163, 245, 174, 68,  218, 219, 38,  238, 223, 64,  145, 178, 219, 103, 51,
  31,  118, 14,  85,  242, 156, 232, 19,  189, 160, 54,  191, 210, 76,  92,
  83,  226, 218, 236, 151, 103, 210, 157, 72,  116, 172, 22,  215, 76,  0,
  118, 163, 10,  203, 173, 176, 22,  137, 8,   120, 170, 112, 8,   212, 180,
  172, 22,  212, 228, 56,  147, 39,  202, 167, 19,  0,   0,   0,   7,   254,
  101, 171, 224, 8,   88,  186, 41,  9,   223, 83,  185, 40,  57,  104, 11,
  192, 19,  2,   98,  118, 182, 110, 128, 130, 218, 198, 68,  226, 205, 16,
  77,  200, 8,   188, 253, 177, 254, 206, 3,   55,  30,  197, 74,  135, 19,
  22,  36,  93,  69,  223, 46,  252, 194, 72,  173, 77,  192, 169, 156, 80,
  169, 76,  176, 64,  246, 26,  210, 16,  189, 89,  24,  245, 85,  198, 164,
  235, 221, 6,   158, 222, 46,  217, 13,  178, 20,  64,  163, 153, 71,  208,
  207, 42,  149, 187, 193, 44,  198, 109, 243, 89,  80,  187, 87,  17,  242,
  157, 78,  171, 142, 5,   183, 114, 94,  237, 205, 211, 200, 242, 155, 45,
  236, 171, 11,  144, 91,  212, 30,  252, 15,  206, 63,  82,  37,  214, 119,
  223, 113, 171, 170, 237, 31,  199, 71,  89,  240, 197, 249, 114, 254, 85,
  56,  6,   115, 82,  1,   112, 95,  66,  95,  177, 45,  77,  230, 110, 238,
  118, 212, 121, 99,  248, 217, 160, 143, 0,   132, 40,  37,  67,  77,  119,
  125, 4,   20,  222, 175, 68,  73,  248, 65,  76,  56,  190, 108, 49,  181,
  224, 187, 132, 163, 235, 106, 85,  68,  154, 163, 123, 129, 49,  19,  27,
  245, 93,  1,   28,  189, 2,   241, 93,  139, 58,  35,  209, 161, 134, 229,
  223, 230, 54,  52,  45,  96,  97,  106, 191, 116, 34,  74,  169, 40,  141,
  210, 101, 120, 109, 190, 60,  16,  170, 136, 238, 56,  180, 83,  28,  229,
  55,  94,  246, 202, 229, 14,  15,  162, 127, 22,  113, 122, 232, 247, 245,
  93,  129, 131, 83,  158, 186, 216, 247, 229, 20,  192, 184, 65,  123, 125,
  121, 242, 55,  198, 57,  36,  164, 81,  158, 129, 51,  3,   35,  233, 0,
  124, 218, 253, 174, 236, 208, 76,  5,   228, 166, 216, 84,  9,   81,  128,
  5,   2,   101, 28,  88,  143, 63,  173, 69,  68,  20,  50,  167, 184, 68,
  217, 222, 46,  217, 132, 235, 50,  47,  236, 97,  239, 243, 17,  137, 72,
  74,  162, 40,  140, 247, 171, 206, 178, 75,  202, 71,  101, 107, 132, 22,
  104, 166, 114, 125, 184, 226, 237, 215, 189, 241, 198, 227, 148, 2,   15,
  255, 224, 17,  226, 46,  40,  36,  5,   108, 82,  212, 224, 160, 222, 248,
  231, 2,   17,  13,  67,  226, 169, 224, 96,  172, 106, 245, 159, 63,  97,
  230, 54,  105, 122, 79,  222, 146, 246, 111, 177, 152, 170, 164, 232, 211,
  222, 165, 34,  57,  119, 155, 97,  46,  249, 250, 48,  156, 238, 4,   217,
  236, 36,  56,  241, 79,  111, 7,   167, 179, 214, 188, 225, 238, 81,  10,
  39,  43,  149, 31,  218, 21,  215, 67,  228, 228, 160, 224, 188, 251, 41,
  144, 243, 28,  192, 216, 247, 93,  52,  214, 195, 220, 167, 88,  92,  125,
  149, 19,  69,  12,  148, 12,  135, 236, 95,  204, 89,  180, 69,  56,  97,
  148, 176, 168, 26,  177, 21,  75,  27,  54,  239, 54,  33,  55,  14,  215,
  197, 77,  241, 51,  224, 167, 53,  124, 226, 160, 98,  144, 69,  204, 225,
  43,  241, 198, 94,  32,  237, 199, 221, 54,  210, 116, 18,  184, 128, 105,
  127, 168, 173, 161, 23,  198, 207, 159, 75,  48,  167, 34,  26,  114, 125,
  170, 62,  83,  145, 101, 76,  178, 233, 219, 74,  238, 114, 68,  66,  40,
  1,   52,  179, 204, 70,  166, 213, 11,  223, 59,  31,  221, 189, 197, 26,
  126, 218, 116, 140, 123, 144, 4,   254, 82,  69,  122, 174, 78,  189, 134,
  105, 213, 113, 231, 230, 123, 16,  192, 68,  244, 204, 165, 185, 239, 161,
  8,   138, 243, 139, 124, 220, 142, 131, 102, 108, 57,  8,   76,  109, 76,
  170, 151, 132, 50,  148, 100, 92,  122, 213, 64,  177, 31,  66,  114, 129,
  182, 198, 207, 138, 184, 77,  69,  4,   77,  192, 41,  238, 203, 226, 89,
  230, 100, 133, 217, 188, 1,   202, 58,  49,  255, 131, 228, 211, 193, 21,
  203, 56,  136, 120, 126, 164, 38,  247, 5,   216, 86,  40,  238, 65,  226,
  65,  21,  250, 222, 64,  131, 176, 53,  168, 179, 235, 120, 29,  84,  69,
  79,  57,  237, 236, 69,  69,  246, 179, 31,  139, 237, 27,  109, 244, 52,
  59,  114, 198, 80,  145, 174, 138, 24,  240, 11,  236, 115, 53,  96,  215,
  155, 136, 11,  15,  255, 14,  150, 157, 60,  229, 137, 56,  4,   2,   162,
  203, 160, 78,  255, 142, 150, 77,  212, 47,  240, 67,  255, 248, 78,  166,
  174, 250, 120, 205, 253, 214, 148, 47,  50,  125, 235, 233, 238, 214, 7,
  12,  46,  152, 187, 209, 40,  2,   106, 107, 96,  51,  77,  224, 185, 177,
  99,  178, 151, 251, 136, 134, 231, 194, 219, 10,  125, 91,  118, 248, 14,
  39,  66,  67,  204, 19,  190, 225, 98,  24,  102, 195, 8,   64,  255, 123,
  41,  221, 28,  232, 19,  118, 105, 107, 4,   13,  110, 190, 209, 249, 74,
  100, 189, 164, 27,  119, 54,  239, 254, 52,  57,  159, 102, 41,  214, 143,
  245, 220, 93,  33,  232, 26,  101, 19,  197, 171, 63,  94,  84,  59,  81,
  134, 199, 135, 248, 149, 208, 142, 166, 46,  136, 8,   12,  120, 150, 214,
  98,  37,  27,  113, 15,  119, 53,  255, 245, 105, 143, 182, 138, 142, 99,
  85,  48,  114, 16,  25,  145, 203, 16,  168, 151, 17,  102, 185, 128, 181,
  228, 62,  54,  52,  193, 94,  235, 191, 7,   6,   225, 137, 93,  2,   240,
  94,  186, 222, 127, 49,  67,  105, 118, 220, 253, 166, 6,   51,  169, 213,
  156, 26,  226, 16,  174, 141, 163, 162, 163, 151, 110, 167, 81,  90,  62,
  22,  1,   242, 210, 229, 35,  193, 39,  60,  178, 74,  3,   4,   197, 165,
  200, 231, 163, 166, 110, 198, 178, 204, 97,  189, 114, 114, 5,   173, 45,
  36,  151, 49,  237, 5,   140, 205, 104, 147, 17,  222, 235, 209, 15,  101,
  18,  201, 4,   232, 92,  117, 254, 95,  11,  207, 245, 93,  195, 161, 83,
  206, 64,  128, 246, 213, 156, 235, 188, 230, 131, 203, 165, 91,  157, 104,
  69,  26,  159, 235, 191, 221, 197, 142, 11,  212, 152, 188, 141, 185, 54,
  193, 116, 41,  26,  240, 217, 0,   90,  61,  108, 109, 158, 242, 58,  100,
  64,  31,  37,  172, 148, 64,  157, 220, 215, 152, 105, 236, 180, 159, 73,
  44,  77,  41,  123, 119, 81,  222, 206, 12,  188, 155, 92,  2,   14,  43,
  80,  126, 251, 95,  227, 95,  12,  49,  249, 174, 252, 238, 111, 61,  195,
  26,  56,  4,   229, 8,   140, 30,  13,  93,  28,  81,  95,  150, 244, 240,
  146, 218, 152, 184, 16,  138, 181, 111, 127, 4,   248, 161, 103, 50,  26,
  197, 165, 68,  54,  108, 192, 110, 149, 16,  167, 182, 80,  170, 170, 193,
  102, 132, 164, 204, 5,   215, 48,  48,  34,  143, 41,  78,  59,  236, 167,
  118, 85,  236, 129, 184, 215, 33,  72,  63,  234, 176, 235, 53,  216, 38,
  177, 141, 74,  167, 189, 41,  20,  217, 25,  127, 28,  188, 255, 221, 198,
  170, 45,  42,  252, 94,  53,  97,  12,  122, 97,  203, 69,  134, 143, 240,
  8,   76,  96,  82,  44,  114, 183, 153, 58,  70,  73,  145, 8,   87,  20,
  40,  112, 225, 170, 17,  217, 43,  59,  87,  243, 89,  229, 175, 150, 137,
  180, 62,  74,  64,  61,  103, 25,  27,  238, 185, 114, 113, 110, 104, 190,
  248, 23,  92,  214, 176, 17,  98,  36,  72,  20,  128, 25,  78,  49,  30,
  99,  4,   114, 22,  202, 219, 251, 218, 64,  59,  94,  9,   226, 82,  200,
  172, 216, 67,  61,  13,  206, 2,   220, 72,  18,  10,  167, 165, 158, 42,
  90,  149, 156, 134, 236, 186, 140, 49,  18,  25,  178, 91,  249, 121, 121,
  51,  126, 10,  149, 98,  51,  170, 8,   207, 118, 251, 244, 250, 206, 6,
  77,  173, 214, 25,  19,  54,  215, 195, 190, 140, 38,  50,  32,  89,  128,
  34,  82,  61,  156, 127, 37,  192, 36,  150, 107, 136, 24,  79,  222, 84,
  140, 92,  54,  237, 23,  201, 9,   53,  76,  133, 51,  78,  189, 126, 163,
  1,   59,  198, 4,   251, 88,  158, 202, 40,  55,  252, 117, 252, 11,  134,
  167, 172, 247, 10,  48,  241, 163, 187, 23,  123, 88,  92,  76,  10,  7,
  139, 237, 94,  201, 56,  20,  158, 50,  94,  63,  216, 1,   71,  59,  132,
  251, 139, 12,  174, 178, 192, 206, 244, 88,  245, 144, 237, 233, 148, 239,
  182, 121, 249, 25,  136, 255, 12,  36,  125, 80,  138, 131, 96,  130, 56,
  142, 130, 67,  199, 163, 81,  122, 53,  54,  206, 124, 157, 34,  193, 158,
  250, 78,  170, 75,  221, 56,  55,  227, 111, 189, 49,  24,  208, 23,  168,
  53,  115, 69,  238, 139, 217, 215, 197, 90,  202, 109, 252, 65,  254, 203,
  112, 116, 128, 172, 253, 111, 200, 129, 233, 128, 224, 147, 217, 244, 111,
  28,  98,  167, 33,  120, 65,  17,  126, 98,  123, 233, 238, 132, 58,  103,
  6,   150, 132, 11,  251, 6,   94,  3,   31,  95,  21,  107, 168, 83,  69,
  138, 219, 165, 189, 42,  143, 45,  29,  178, 198, 56,  231, 99,  189, 80,
  48,  90,  243, 171, 248, 20,  164, 63,  242, 187, 156, 183, 238, 9,   108,
  204, 153, 99,  164, 163, 134, 27,  51,  190, 255, 197, 167, 244, 238, 65,
  139, 13,  249, 82,  217, 124, 30,  141, 19,  78,  49,  121, 198, 103, 255,
  59,  13,  125, 97,  253, 216, 84,  187, 111, 235, 112, 5,   12,  47,  184,
  241, 57,  111, 215, 27,  126, 44,  79,  82,  47,  181, 253, 57,  159, 29,
  119, 15,  167, 144, 165, 72,  219, 213, 3,   47,  124, 13,  233, 25,  168,
  127, 80,  136, 98,  69,  26,  41,  29,  119, 231, 242, 252, 88,  156, 123,
  130, 247, 9,   4,   76,  136, 237, 94,  57,  55,  226, 236, 4,   58,  117,
  46,  239, 98,  235, 181, 45,  35,  84,  92,  66,  141, 103, 105, 125, 5,
  136, 63,  243, 12,  130, 81,  3,   184, 190, 74,  248, 217, 203, 149, 174,
  32,  85,  242, 103, 107, 74,  141, 102, 10,  215, 105, 16,  137, 215, 26,
  215, 138, 249, 68,  184, 73,  142, 120, 27,  133, 179, 126, 233, 220, 250,
  99,  138, 217, 33,  111, 111, 137, 55,  33,  86,  219, 177, 135, 161, 156,
  35,  231, 21,  97,  77,  75,  106, 104, 254, 253, 130, 74,  122, 147, 73,
  253, 184, 248, 196, 185, 73,  144, 93,  14,  122, 182, 172, 182, 37,  50,
  110, 212, 217, 112, 52,  172, 227, 57,  129, 88,  109, 147, 122, 174, 113,
  25,  43,  148, 90,  5,   191, 152, 200, 36,  157, 138, 3,   233, 187, 157,
  177, 196, 190, 81,  176, 195, 160, 170, 59,  135, 111, 101, 165, 84,  77,
  116, 110, 167, 251, 138, 2,   238, 166, 101, 208, 237, 59,  177, 71,  210,
  176, 48,  25,  74,  68,  162, 47,  164, 18,  159, 111, 214, 103, 224, 134,
  92,  118, 29,  104, 61,  58,  109, 168, 219, 100, 222, 237, 182, 231, 99,
  63,  46,  139, 24,  112, 125, 189, 101, 242, 112, 13,  83,  24,  141, 184,
  39,  27,  223, 41,  15,  120, 136, 30,  12,  126, 186, 167, 103, 126, 188,
  87,  150, 166, 237, 210, 107, 208, 231, 36,  34,  81,  192, 65,  240, 129,
  232, 111, 148, 88,  71,  23,  7,   108, 240, 50,  248, 98,  8,   243, 98,
  80,  204, 132, 164, 214, 161, 190, 208, 15,  100, 243, 206, 179, 168, 132,
  14,  0,   90,  0,   229, 197, 156, 134, 88,  105, 52,  196, 14,  69,  52,
  103, 184, 103, 37,  238, 244, 241, 217, 175, 46,  103, 243, 191, 49,  51,
  150, 6,   53,  72,  139, 11,  194, 47,  12,  54,  177, 150, 75,  195, 222,
  52,  100, 211, 148, 56,  34,  248, 236, 118, 229, 211, 71,  202, 60,  180,
  118, 110, 153, 143, 80,  111, 172, 202, 189, 205, 96,  38,  58,  233, 224,
  69,  56,  234, 52,  97,  133, 224, 162, 181, 60,  85,  145, 254, 74,  227,
  8,   172, 100, 38,  140, 161, 190, 76,  74,  23,  52,  173, 120, 237, 242,
  93,  73,  120, 170, 254, 12,  62,  150, 67,  209, 85,  182, 152, 112, 49,
  150, 82,  178, 199, 178, 240, 189, 84,  62,  229, 220, 107, 222, 177, 130,
  37,  152, 150, 141, 109, 246, 144, 137, 75,  66,  202, 65,  71,  112, 65,
  87,  49,  106, 84,  4,   114, 82,  32,  222, 206, 32,  227, 86,  255, 237,
  90,  158, 53,  151, 119, 127, 202, 114, 188, 203, 47,  220, 211, 3,   180,
  207, 60,  176, 56,  169, 202, 174, 123, 154, 228, 237, 131, 107, 211, 80,
  219, 17,  70,  22,  139, 158, 111, 96,  196, 178, 56,  56,  58,  61,  204,
  89,  13,  137, 80,  126, 144, 186, 165, 161, 6,   252, 56,  117, 36,  90,
  126, 187, 34,  87,  238, 239, 100, 199, 161, 53,  150, 227, 128, 84,  125,
  132, 68,  58,  33,  243, 12,  102, 30,  201, 60,  243, 101, 159, 247, 144,
  220, 205, 80,  11,  36,  58,  8,   39,  58,  28,  115, 46,  86,  219, 128,
  180, 63,  186, 97,  163, 248, 248, 204, 219, 181, 184, 30,  5,   130, 11,
  148, 249, 138, 49,  222, 172, 236, 156, 235, 180, 253, 117, 225, 32,  79,
  6,   173, 109, 0,   170, 244, 81,  188, 244, 85,  40,  168, 73,  75,  248,
  113, 176, 159, 250, 29,  61,  247, 229, 34,  15,  178, 82,  85,  56,  181,
  189, 133, 70,  87,  193, 57,  254, 48,  163, 202, 174, 28,  247, 90,  184,
  246, 53,  104, 81,  201, 188, 65,  159, 153, 91,  115, 99,  177, 40,  228,
  8,   76,  96,  85,  37,  71,  19,  75,  176, 50,  160, 195, 25,  229, 218,
  149, 156, 233, 11,  44,  129, 181, 101, 38,  236, 112, 20,  71,  111, 57,
  255, 255, 55,  14,  236, 144, 180, 155, 29,  154, 171, 136, 5,   154, 89,
  151, 160, 71,  184, 159, 81,  69,  192, 3,   3,   255, 246, 188, 119, 219,
  107, 216, 160, 116, 28,  246, 205, 225, 234, 150, 218, 26,  51,  158, 2,
  216, 210, 174, 149, 138, 5,   152, 146, 160, 3,   214, 93,  114, 92,  226,
  98,  7,   15,  127, 75,  176, 162, 252, 37,  54,  226, 58,  224, 206, 89,
  123, 78,  67,  122, 86,  159, 85,  127, 198, 141, 77,  222, 219, 246, 185,
  116, 75,  236, 122, 208, 97,  99,  205, 24,  222, 195, 35,  18,  37,  160,
  41,  247, 2,   179, 194, 221, 112, 172, 59,  119, 75,  56,  251, 3,   4,
  179, 146, 232, 185, 41,  209, 31,  25,  35,  60,  178, 58,  164, 111, 182,
  212, 55,  22,  246, 97,  109, 235, 81,  251, 186, 118, 244, 223, 202, 90,
  53,  178, 167, 181, 138, 75,  56,  226, 92,  165, 190, 37,  49,  139, 249,
  216, 149, 190, 200, 123, 122, 48,  3,   236, 78,  246, 50,  235, 234, 74,
  169, 222, 208, 183, 242, 213, 155, 163, 161, 22,  3,   70,  30,  164, 221,
  37,  7,   254, 19,  206, 230, 163, 28,  212, 155, 37,  64,  13,  221, 83,
  166, 104, 243, 12,  155, 44,  146, 233, 205, 217, 148, 0,   228, 153, 172,
  151, 127, 231, 56,  33,  170, 217, 131, 31,  231, 223, 132, 212, 182, 168,
  190, 172, 90,  211, 129, 43,  238, 51,  129, 236, 224, 226, 105, 178, 248,
  115, 202, 180, 211, 55,  110, 106, 242, 113, 219, 3,   248, 200, 202, 200,
  189, 76,  112, 228, 89,  53,  246, 255, 117, 43,  211, 149, 32,  9,   9,
  75,  15,  204, 59,  186, 123, 157, 13,  124, 252, 150, 98,  75,  3,   229,
  111, 207, 117, 214, 143, 228, 191, 161, 221, 58,  158, 166, 212, 157, 36,
  125, 179, 61,  4,   179, 205, 133, 177, 3,   233, 79,  226, 194, 216, 126,
  49,  177, 1,   16,  146, 94,  8,   144, 196, 212, 81,  58,  255, 200, 133,
  174, 138, 13,  145, 146, 178, 247, 196, 163, 71,  228, 109, 67,  23,  213,
  121, 123, 200, 84,  201, 127, 101, 232, 197, 185, 154, 16,  65,  57,  5,
  81,  25,  154, 23,  182, 230, 209, 29,  73,  114, 202, 41,  253, 18,  137,
  97,  18,  90,  190, 16,  64,  0,   184, 104, 125, 71,  16,  202, 19,  228,
  34,  5,   36,  171, 43,  121, 136, 186, 182, 41,  64,  167, 56,  11,  242,
  162, 232, 24,  96,  251, 120, 3,   156, 117, 138, 148, 121, 171, 38,  151,
  156, 120, 65,  224, 23,  134, 115, 98,  238, 183, 223, 182, 148, 173, 179,
  134, 107, 139, 34,  106, 115, 118, 187, 249, 163, 48,  36,  219, 153, 51,
  181, 35,  236, 190, 120, 94,  237, 94,  133, 11,  19,  132, 44,  57,  18,
  156, 190, 66,  39,  114, 57,  64,  93,  239, 89,  82,  161, 40,  96,  224,
  99,  168, 84,  201, 31,  116, 167, 171, 222, 74,  28,  105, 30,  100, 96,
  54,  159, 170, 174, 98,  109, 52,  153, 45,  124, 7,   102, 108, 99,  170,
  133, 107, 70,  163, 255, 0,   16,  206, 92,  1,   234, 119, 187, 57,  165,
  18,  213, 221, 166, 73,  255, 38,  239, 236, 253, 112, 36,  86,  151, 168,
  30,  19,  204, 3,   147, 51,  169, 48,  235, 152, 0,   0,   0,   9,   23,
  0,   0,   140, 200, 0,   94,  120, 24,  69,  152, 0,   0,
]);

/** File for transparent sample WebP. */
export const TRANSPARENT_WEBP_FILE = new File(
    [TRANSPARENT_WEBP_BYTES], 'valid-transparent.webp', {type: 'image/webp'});

/**
 * Bytes for a sample opaque WebP.
 *
 * https://www.gstatic.com/webp/gallery/1.webp
 */
export const OPAQUE_WEBP_BYTES = new Uint8Array([
  82,  73,  70,  70,  190, 10,  0,   0,   87,  69,  66,  80,  86,  80,  56,
  32,  178, 10,  0,   0,   144, 45,  0,   157, 1,   42,  137, 0,   92,  0,
  62,  157, 66,  154, 73,  40,  39,  37,  36,  45,  22,  139, 192, 176, 19,
  137, 64,  24,  198, 250, 33,  190, 149, 150, 134, 125, 239, 127, 110, 126,
  167, 36,  235, 37,  125, 23,  176, 26,  251, 255, 67,  185, 131, 179, 185,
  136, 67,  239, 147, 71,  216, 140, 62,  30,  87,  189, 135, 238, 222, 161,
  125, 42,  253, 23,  63,  96,  16,  84,  22,  58,  26,  35,  125, 3,   188,
  128, 196, 24,  34,  210, 156, 3,   190, 234, 22,  250, 68,  58,  116, 129,
  34,  147, 230, 97,  109, 158, 233, 48,  134, 64,  220, 43,  210, 223, 211,
  31,  41,  154, 137, 245, 77,  152, 204, 228, 126, 71,  22,  104, 173, 217,
  230, 196, 109, 80,  185, 97,  215, 208, 169, 131, 212, 39,  214, 102, 157,
  3,   144, 70,  194, 145, 156, 204, 72,  183, 168, 102, 195, 143, 33,  170,
  231, 28,  163, 48,  58,  133, 137, 15,  192, 139, 227, 154, 200, 2,   176,
  241, 95,  180, 172, 68,  74,  203, 221, 102, 81,  105, 29,  53,  168, 131,
  84,  243, 215, 43,  56,  20,  94,  172, 245, 137, 133, 189, 25,  232, 243,
  172, 135, 146, 67,  34,  2,   127, 113, 73,  181, 251, 182, 96,  30,  165,
  172, 70,  135, 129, 191, 133, 81,  70,  243, 45,  1,   187, 91,  178, 140,
  21,  73,  10,  139, 139, 206, 54,  52,  241, 19,  22,  7,   90,  107, 226,
  61,  75,  210, 201, 221, 15,  143, 102, 51,  156, 61,  123, 222, 236, 79,
  241, 140, 3,   56,  71,  255, 43,  3,   33,  99,  150, 242, 172, 30,  110,
  42,  76,  134, 141, 143, 241, 251, 252, 103, 180, 207, 156, 177, 48,  76,
  66,  47,  52,  122, 255, 195, 36,  147, 125, 154, 115, 226, 212, 203, 213,
  237, 211, 99,  57,  137, 88,  108, 92,  221, 162, 76,  121, 149, 121, 120,
  232, 132, 233, 254, 27,  104, 124, 99,  250, 69,  125, 67,  24,  189, 194,
  231, 147, 184, 244, 20,  120, 246, 105, 232, 209, 142, 235, 142, 251, 169,
  225, 108, 220, 180, 44,  101, 67,  208, 160, 152, 20,  20,  134, 184, 66,
  232, 208, 0,   0,   254, 86,  148, 82,  163, 153, 147, 217, 149, 176, 203,
  37,  249, 182, 211, 255, 230, 104, 253, 241, 191, 150, 69,  37,  223, 254,
  96,  159, 225, 98,  246, 141, 108, 250, 215, 41,  162, 207, 47,  8,   66,
  22,  180, 143, 21,  74,  5,   157, 172, 160, 62,  137, 147, 21,  89,  87,
  242, 112, 172, 159, 132, 75,  94,  8,   113, 154, 151, 5,   84,  56,  21,
  9,   159, 108, 220, 186, 158, 103, 27,  185, 180, 97,  147, 250, 130, 148,
  5,   17,  193, 163, 212, 149, 29,  57,  128, 79,  231, 19,  151, 12,  215,
  21,  225, 248, 101, 84,  173, 33,  178, 180, 107, 116, 201, 127, 160, 133,
  58,  120, 219, 223, 131, 140, 57,  250, 218, 34,  212, 225, 101, 178, 146,
  224, 169, 151, 35,  79,  82,  172, 11,  112, 235, 24,  73,  87,  31,  240,
  48,  60,  236, 22,  189, 206, 11,  39,  2,   166, 17,  13,  32,  139, 175,
  243, 172, 109, 8,   0,   91,  17,  197, 76,  111, 115, 82,  127, 157, 217,
  234, 178, 176, 205, 21,  114, 44,  175, 53,  234, 244, 19,  124, 206, 112,
  138, 50,  94,  192, 26,  199, 175, 221, 200, 217, 72,  199, 5,   198, 140,
  224, 68,  180, 65,  254, 101, 197, 114, 225, 200, 227, 54,  61,  200, 48,
  248, 212, 245, 229, 8,   232, 183, 158, 43,  154, 227, 183, 34,  227, 204,
  29,  35,  223, 19,  47,  142, 240, 139, 246, 158, 163, 196, 181, 147, 186,
  240, 58,  88,  126, 244, 204, 160, 207, 223, 192, 191, 60,  96,  119, 58,
  217, 77,  146, 44,  93,  44,  224, 194, 123, 133, 63,  134, 35,  11,  31,
  255, 58,  7,   87,  87,  127, 46,  32,  41,  99,  66,  55,  4,   110, 225,
  21,  241, 64,  92,  96,  86,  4,   247, 106, 184, 239, 117, 106, 186, 84,
  21,  25,  229, 48,  183, 210, 200, 69,  33,  235, 41,  84,  224, 57,  122,
  214, 13,  132, 157, 108, 99,  59,  236, 25,  158, 194, 210, 137, 127, 151,
  196, 102, 163, 65,  203, 232, 26,  205, 203, 39,  122, 161, 93,  114, 186,
  1,   55,  35,  128, 212, 204, 35,  12,  199, 178, 89,  174, 28,  240, 72,
  197, 105, 7,   123, 202, 64,  228, 54,  74,  173, 43,  11,  145, 122, 104,
  150, 177, 90,  242, 73,  232, 33,  0,   181, 48,  188, 236, 12,  208, 6,
  177, 175, 6,   34,  212, 141, 1,   121, 40,  137, 79,  235, 202, 10,  57,
  164, 89,  81,  24,  184, 24,  218, 255, 98,  69,  113, 167, 249, 203, 110,
  220, 139, 140, 211, 4,   228, 131, 167, 241, 140, 108, 63,  3,   204, 105,
  4,   59,  164, 70,  46,  1,   97,  157, 155, 231, 168, 67,  68,  44,  140,
  114, 34,  169, 63,  155, 22,  207, 188, 139, 23,  237, 164, 48,  156, 149,
  154, 231, 147, 53,  210, 54,  95,  29,  196, 169, 252, 67,  203, 57,  101,
  200, 207, 80,  89,  33,  13,  62,  229, 202, 243, 233, 73,  120, 163, 110,
  29,  89,  63,  238, 53,  195, 86,  165, 246, 237, 206, 154, 25,  190, 190,
  159, 112, 194, 112, 42,  153, 50,  196, 145, 26,  8,   133, 202, 55,  188,
  166, 201, 13,  41,  8,   145, 147, 74,  159, 161, 118, 252, 57,  163, 90,
  50,  17,  66,  154, 84,  240, 14,  247, 57,  108, 208, 90,  25,  250, 143,
  27,  109, 166, 112, 71,  50,  201, 122, 191, 69,  104, 121, 199, 79,  87,
  71,  24,  132, 7,   38,  142, 143, 177, 251, 218, 243, 143, 60,  132, 2,
  27,  134, 249, 186, 111, 145, 84,  208, 50,  62,  86,  50,  122, 124, 234,
  74,  151, 84,  20,  209, 93,  140, 29,  33,  198, 195, 158, 29,  237, 190,
  109, 220, 205, 21,  61,  60,  21,  58,  184, 192, 184, 43,  202, 156, 127,
  1,   147, 37,  212, 114, 208, 231, 128, 233, 27,  25,  174, 32,  192, 228,
  174, 248, 193, 129, 130, 164, 82,  75,  134, 237, 138, 238, 97,  46,  186,
  174, 115, 210, 50,  29,  49,  173, 136, 248, 17,  213, 32,  79,  120, 235,
  247, 176, 241, 95,  121, 32,  49,  2,   97,  164, 5,   155, 194, 10,  2,
  189, 149, 112, 96,  251, 2,   232, 207, 132, 158, 96,  254, 15,  202, 143,
  6,   214, 91,  232, 175, 111, 28,  125, 246, 15,  74,  250, 114, 18,  105,
  100, 199, 52,  145, 38,  100, 236, 141, 252, 168, 7,   124, 181, 154, 2,
  153, 70,  248, 180, 72,  190, 187, 110, 185, 116, 179, 18,  71,  8,   66,
  120, 188, 165, 240, 56,  213, 12,  16,  225, 134, 53,  198, 103, 46,  161,
  247, 31,  10,  60,  114, 195, 195, 91,  15,  30,  194, 190, 128, 29,  68,
  164, 9,   44,  111, 62,  58,  162, 136, 38,  152, 63,  184, 113, 123, 192,
  36,  129, 155, 160, 172, 23,  63,  52,  7,   117, 176, 101, 171, 57,  22,
  188, 111, 168, 237, 135, 76,  173, 169, 139, 202, 51,  68,  20,  82,  69,
  160, 235, 95,  136, 185, 72,  188, 126, 5,   29,  220, 253, 224, 127, 230,
  58,  144, 49,  16,  222, 233, 39,  171, 127, 104, 235, 125, 101, 37,  3,
  29,  51,  158, 139, 140, 181, 139, 42,  241, 130, 114, 44,  183, 39,  254,
  192, 99,  65,  71,  59,  172, 158, 68,  175, 117, 131, 70,  131, 77,  223,
  183, 184, 34,  72,  117, 113, 130, 216, 218, 145, 198, 177, 194, 99,  45,
  139, 218, 147, 83,  214, 26,  24,  148, 190, 108, 144, 158, 79,  181, 190,
  36,  35,  131, 168, 5,   188, 88,  52,  252, 55,  190, 253, 127, 166, 1,
  125, 40,  210, 215, 239, 208, 206, 222, 47,  163, 191, 221, 194, 243, 248,
  52,  143, 165, 40,  46,  204, 160, 48,  93,  204, 201, 92,  123, 194, 17,
  80,  92,  141, 162, 148, 184, 229, 204, 193, 84,  102, 184, 201, 39,  221,
  17,  103, 174, 45,  220, 88,  0,   189, 210, 165, 188, 124, 190, 54,  149,
  80,  43,  138, 169, 243, 248, 73,  208, 100, 161, 2,   164, 85,  85,  196,
  108, 235, 52,  225, 8,   90,  249, 229, 186, 90,  214, 13,  102, 88,  141,
  81,  206, 169, 80,  90,  238, 15,  34,  234, 84,  148, 174, 68,  146, 24,
  113, 240, 242, 188, 175, 114, 84,  194, 117, 28,  230, 160, 221, 73,  122,
  102, 166, 78,  208, 201, 249, 220, 225, 72,  81,  136, 106, 134, 46,  103,
  107, 63,  206, 141, 130, 228, 131, 203, 44,  229, 228, 145, 246, 15,  89,
  150, 223, 238, 235, 52,  43,  112, 122, 36,  130, 167, 189, 207, 198, 93,
  236, 21,  69,  192, 70,  35,  88,  129, 253, 100, 92,  100, 203, 248, 132,
  220, 107, 184, 86,  94,  108, 136, 37,  163, 90,  229, 222, 239, 192, 80,
  255, 158, 37,  58,  101, 103, 137, 151, 189, 241, 142, 153, 117, 77,  99,
  78,  165, 65,  155, 31,  140, 32,  238, 127, 47,  28,  132, 136, 150, 252,
  44,  253, 14,  3,   252, 26,  190, 37,  222, 223, 79,  131, 29,  20,  231,
  229, 68,  59,  225, 98,  73,  31,  191, 229, 72,  189, 180, 97,  152, 187,
  116, 95,  37,  0,   123, 44,  104, 108, 33,  188, 159, 200, 89,  212, 70,
  7,   106, 77,  61,  134, 7,   54,  4,   193, 213, 14,  83,  153, 53,  140,
  143, 16,  18,  179, 30,  83,  64,  20,  213, 25,  226, 107, 210, 70,  172,
  134, 106, 219, 138, 58,  242, 203, 113, 70,  179, 79,  234, 100, 61,  132,
  89,  16,  52,  123, 159, 65,  121, 45,  156, 140, 186, 43,  218, 126, 194,
  71,  242, 196, 150, 229, 167, 186, 87,  68,  171, 49,  115, 89,  227, 22,
  155, 193, 144, 20,  88,  221, 101, 200, 139, 201, 209, 75,  221, 181, 209,
  74,  36,  154, 233, 241, 63,  127, 179, 244, 53,  25,  242, 170, 160, 208,
  220, 185, 82,  208, 15,  210, 232, 131, 107, 164, 93,  141, 237, 74,  118,
  144, 30,  186, 64,  114, 158, 203, 238, 26,  22,  71,  35,  251, 6,   108,
  53,  177, 42,  200, 88,  4,   157, 59,  235, 208, 10,  173, 58,  4,   214,
  168, 62,  163, 30,  163, 190, 224, 205, 42,  27,  70,  65,  145, 213, 248,
  73,  150, 74,  192, 157, 32,  63,  250, 8,   190, 31,  102, 188, 114, 47,
  111, 202, 41,  199, 77,  250, 64,  58,  196, 13,  27,  238, 103, 76,  82,
  103, 123, 44,  141, 214, 221, 135, 224, 198, 193, 190, 93,  55,  234, 236,
  176, 196, 43,  191, 33,  198, 199, 3,   132, 81,  125, 171, 27,  120, 51,
  127, 31,  196, 48,  4,   53,  177, 118, 166, 188, 45,  244, 139, 70,  95,
  18,  145, 182, 244, 80,  37,  159, 175, 35,  29,  247, 21,  19,  176, 130,
  89,  246, 254, 124, 77,  145, 95,  16,  174, 96,  213, 218, 121, 183, 98,
  201, 71,  97,  130, 90,  12,  135, 123, 2,   25,  91,  83,  199, 83,  102,
  243, 148, 176, 106, 78,  89,  150, 29,  10,  106, 75,  130, 38,  53,  192,
  94,  234, 222, 173, 79,  110, 119, 153, 74,  170, 100, 157, 170, 146, 81,
  34,  240, 91,  211, 82,  162, 62,  48,  17,  8,   7,   38,  99,  163, 27,
  71,  217, 83,  11,  251, 83,  86,  67,  142, 108, 222, 21,  18,  6,   245,
  48,  200, 135, 87,  162, 192, 212, 65,  66,  38,  215, 157, 53,  88,  175,
  107, 200, 185, 85,  245, 254, 191, 177, 37,  225, 117, 181, 90,  97,  108,
  189, 174, 248, 152, 225, 23,  224, 49,  152, 118, 45,  169, 114, 126, 161,
  216, 46,  104, 195, 239, 209, 246, 57,  240, 45,  98,  245, 1,   235, 124,
  28,  99,  10,  95,  135, 11,  122, 82,  220, 102, 72,  25,  183, 169, 1,
  206, 107, 253, 228, 215, 152, 230, 226, 136, 77,  15,  16,  29,  180, 255,
  248, 176, 181, 182, 221, 70,  79,  232, 210, 77,  189, 24,  49,  101, 74,
  217, 220, 217, 15,  2,   138, 13,  243, 30,  174, 161, 50,  102, 72,  99,
  95,  214, 208, 209, 59,  255, 162, 122, 121, 213, 200, 141, 202, 126, 94,
  165, 41,  156, 162, 19,  161, 58,  120, 252, 49,  15,  154, 37,  14,  229,
  168, 95,  76,  218, 62,  220, 229, 233, 121, 181, 91,  67,  170, 43,  122,
  186, 107, 168, 3,   72,  197, 170, 154, 35,  113, 155, 238, 177, 76,  172,
  210, 141, 36,  107, 71,  174, 51,  55,  130, 133, 4,   80,  122, 210, 221,
  148, 42,  69,  1,   110, 155, 174, 117, 238, 42,  87,  92,  163, 226, 176,
  203, 101, 224, 80,  19,  61,  34,  70,  58,  236, 187, 94,  129, 105, 10,
  233, 91,  43,  169, 40,  124, 246, 190, 85,  101, 33,  23,  175, 94,  168,
  242, 77,  31,  143, 237, 145, 198, 106, 52,  76,  96,  199, 73,  47,  21,
  181, 207, 33,  49,  179, 213, 212, 115, 10,  80,  183, 211, 229, 75,  25,
  193, 30,  222, 1,   233, 143, 197, 145, 8,   27,  192, 181, 55,  134, 75,
  52,  194, 1,   140, 104, 179, 152, 210, 17,  213, 45,  91,  75,  49,  39,
  241, 249, 173, 118, 122, 102, 16,  99,  213, 146, 151, 109, 90,  172, 138,
  38,  41,  118, 202, 170, 123, 134, 70,  43,  103, 195, 74,  180, 149, 173,
  161, 40,  55,  72,  237, 204, 18,  142, 99,  47,  167, 172, 171, 111, 193,
  204, 91,  28,  149, 31,  99,  244, 160, 5,   158, 237, 139, 15,  30,  96,
  223, 140, 89,  22,  179, 76,  34,  11,  250, 159, 247, 252, 236, 180, 122,
  53,  163, 149, 29,  254, 187, 11,  217, 158, 130, 42,  189, 133, 28,  252,
  211, 124, 113, 39,  113, 168, 107, 194, 52,  75,  255, 184, 159, 199, 52,
  240, 74,  112, 8,   66,  162, 104, 151, 212, 61,  48,  3,   34,  40,  227,
  246, 63,  105, 110, 182, 250, 50,  64,  146, 28,  2,   37,  126, 254, 28,
  14,  22,  73,  159, 108, 52,  115, 49,  110, 26,  137, 222, 20,  225, 79,
  91,  124, 139, 27,  34,  119, 209, 82,  155, 238, 108, 36,  147, 10,  67,
  57,  116, 193, 127, 4,   20,  119, 198, 109, 118, 105, 74,  255, 70,  70,
  70,  93,  196, 205, 201, 76,  161, 87,  90,  208, 4,   46,  252, 102, 94,
  81,  107, 168, 242, 145, 116, 223, 90,  159, 78,  175, 5,   211, 162, 37,
  178, 245, 14,  167, 205, 15,  23,  156, 205, 120, 66,  26,  143, 255, 74,
  155, 61,  81,  87,  251, 169, 203, 119, 254, 85,  30,  216, 177, 101, 119,
  176, 3,   124, 109, 229, 160, 154, 89,  227, 205, 69,  135, 94,  119, 248,
  95,  168, 138, 238, 3,   72,  64,  64,  141, 49,  70,  229, 10,  209, 76,
  37,  123, 235, 108, 136, 243, 81,  203, 255, 11,  249, 114, 210, 76,  140,
  176, 166, 182, 124, 226, 112, 110, 53,  30,  123, 194, 245, 252, 188, 198,
  31,  147, 84,  129, 66,  251, 87,  220, 101, 174, 129, 149, 51,  21,  186,
  221, 142, 209, 191, 145, 127, 42,  8,   199, 86,  79,  232, 224, 246, 25,
  53,  100, 203, 172, 148, 15,  115, 195, 124, 161, 251, 60,  10,  79,  79,
  217, 200, 72,  250, 46,  126, 113, 210, 151, 154, 134, 7,   71,  79,  119,
  152, 154, 152, 235, 60,  193, 26,  210, 207, 146, 18,  250, 116, 236, 179,
  227, 237, 160, 135, 203, 132, 36,  16,  42,  99,  97,  27,  80,  110, 137,
  199, 45,  251, 94,  152, 194, 215, 22,  141, 7,   101, 252, 124, 88,  0,
  105, 202, 21,  148, 121, 232, 49,  23,  32,  246, 163, 52,  186, 235, 221,
  40,  55,  249, 16,  56,  121, 168, 226, 93,  148, 228, 89,  171, 243, 196,
  19,  31,  85,  51,  72,  46,  85,  184, 144, 115, 18,  148, 201, 7,   234,
  87,  175, 255, 237, 212, 194, 212, 47,  80,  117, 148, 184, 23,  123, 53,
  243, 52,  71,  61,  255, 188, 60,  65,  131, 210, 11,  66,  101, 58,  174,
  12,  165, 21,  204, 4,   11,  217, 7,   161, 235, 99,  78,  221, 233, 76,
  202, 58,  26,  82,  31,  115, 192, 141, 238, 181, 0,   0,   0,
]);

/** File for opaque sample WebP. */
export const OPAQUE_WEBP_FILE =
    new File([OPAQUE_WEBP_BYTES], 'valid-opaque.webp', {type: 'image/webp'});

/**
 * Bytes for a transparent BMP.
 *
 * Bytes that are set represent the fields that are needed to identify
 * a transparent BMP, but holistically this is not reflective of a real image's
 * bytes.
 */
export const TRANSPARENT_BMP_BYTES =
    new Uint8Array([66, 77, ...Array.from({length: 27}, () => 0), 50]);

/**
 * File for transparent sample BMP.
 *
 * Not a real BMP file, as it is surprisingly hard to find a transparent BMP,
 * but bytes are set to simulate a transparent BMP.
 */
export const TRANSPARENT_BMP_FILE = new File(
    [TRANSPARENT_BMP_BYTES], 'valid-transparent.bmp', {type: 'image/bmp'});


/**
 *  Bytes for a opaque BMP.
 * https://people.math.sc.edu/Burkardt/data/bmp/all_gray.bmp
 */
export const OPAQUE_BMP_BYTES = new Uint8Array([
  66,  77,  70,  4,   0,   0,   0,   0,   0,   0,   54,  4,   0,   0,   40,
  0,   0,   0,   8,   0,   0,   0,   2,   0,   0,   0,   1,   0,   8,   0,
  0,   0,   0,   0,   16,  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
  0,   0,   1,   0,   0,   0,   1,   0,   0,   0,   0,   0,   0,   1,   1,
  1,   0,   2,   2,   2,   0,   3,   3,   3,   0,   4,   4,   4,   0,   5,
  5,   5,   0,   6,   6,   6,   0,   7,   7,   7,   0,   8,   8,   8,   0,
  9,   9,   9,   0,   10,  10,  10,  0,   11,  11,  11,  0,   12,  12,  12,
  0,   13,  13,  13,  0,   14,  14,  14,  0,   15,  15,  15,  0,   16,  16,
  16,  0,   17,  17,  17,  0,   18,  18,  18,  0,   19,  19,  19,  0,   20,
  20,  20,  0,   21,  21,  21,  0,   22,  22,  22,  0,   23,  23,  23,  0,
  24,  24,  24,  0,   25,  25,  25,  0,   26,  26,  26,  0,   27,  27,  27,
  0,   28,  28,  28,  0,   29,  29,  29,  0,   30,  30,  30,  0,   31,  31,
  31,  0,   32,  32,  32,  0,   33,  33,  33,  0,   34,  34,  34,  0,   35,
  35,  35,  0,   36,  36,  36,  0,   37,  37,  37,  0,   38,  38,  38,  0,
  39,  39,  39,  0,   40,  40,  40,  0,   41,  41,  41,  0,   42,  42,  42,
  0,   43,  43,  43,  0,   44,  44,  44,  0,   45,  45,  45,  0,   46,  46,
  46,  0,   47,  47,  47,  0,   48,  48,  48,  0,   49,  49,  49,  0,   50,
  50,  50,  0,   51,  51,  51,  0,   52,  52,  52,  0,   53,  53,  53,  0,
  54,  54,  54,  0,   55,  55,  55,  0,   56,  56,  56,  0,   57,  57,  57,
  0,   58,  58,  58,  0,   59,  59,  59,  0,   60,  60,  60,  0,   61,  61,
  61,  0,   62,  62,  62,  0,   63,  63,  63,  0,   64,  64,  64,  0,   65,
  65,  65,  0,   66,  66,  66,  0,   67,  67,  67,  0,   68,  68,  68,  0,
  69,  69,  69,  0,   70,  70,  70,  0,   71,  71,  71,  0,   72,  72,  72,
  0,   73,  73,  73,  0,   74,  74,  74,  0,   75,  75,  75,  0,   76,  76,
  76,  0,   77,  77,  77,  0,   78,  78,  78,  0,   79,  79,  79,  0,   80,
  80,  80,  0,   81,  81,  81,  0,   82,  82,  82,  0,   83,  83,  83,  0,
  84,  84,  84,  0,   85,  85,  85,  0,   86,  86,  86,  0,   87,  87,  87,
  0,   88,  88,  88,  0,   89,  89,  89,  0,   90,  90,  90,  0,   91,  91,
  91,  0,   92,  92,  92,  0,   93,  93,  93,  0,   94,  94,  94,  0,   95,
  95,  95,  0,   96,  96,  96,  0,   97,  97,  97,  0,   98,  98,  98,  0,
  99,  99,  99,  0,   100, 100, 100, 0,   101, 101, 101, 0,   102, 102, 102,
  0,   103, 103, 103, 0,   104, 104, 104, 0,   105, 105, 105, 0,   106, 106,
  106, 0,   107, 107, 107, 0,   108, 108, 108, 0,   109, 109, 109, 0,   110,
  110, 110, 0,   111, 111, 111, 0,   112, 112, 112, 0,   113, 113, 113, 0,
  114, 114, 114, 0,   115, 115, 115, 0,   116, 116, 116, 0,   117, 117, 117,
  0,   118, 118, 118, 0,   119, 119, 119, 0,   120, 120, 120, 0,   121, 121,
  121, 0,   122, 122, 122, 0,   123, 123, 123, 0,   124, 124, 124, 0,   125,
  125, 125, 0,   126, 126, 126, 0,   127, 127, 127, 0,   128, 128, 128, 0,
  129, 129, 129, 0,   130, 130, 130, 0,   131, 131, 131, 0,   132, 132, 132,
  0,   133, 133, 133, 0,   134, 134, 134, 0,   135, 135, 135, 0,   136, 136,
  136, 0,   137, 137, 137, 0,   138, 138, 138, 0,   139, 139, 139, 0,   140,
  140, 140, 0,   141, 141, 141, 0,   142, 142, 142, 0,   143, 143, 143, 0,
  144, 144, 144, 0,   145, 145, 145, 0,   146, 146, 146, 0,   147, 147, 147,
  0,   148, 148, 148, 0,   149, 149, 149, 0,   150, 150, 150, 0,   151, 151,
  151, 0,   152, 152, 152, 0,   153, 153, 153, 0,   154, 154, 154, 0,   155,
  155, 155, 0,   156, 156, 156, 0,   157, 157, 157, 0,   158, 158, 158, 0,
  159, 159, 159, 0,   160, 160, 160, 0,   161, 161, 161, 0,   162, 162, 162,
  0,   163, 163, 163, 0,   164, 164, 164, 0,   165, 165, 165, 0,   166, 166,
  166, 0,   167, 167, 167, 0,   168, 168, 168, 0,   169, 169, 169, 0,   170,
  170, 170, 0,   171, 171, 171, 0,   172, 172, 172, 0,   173, 173, 173, 0,
  174, 174, 174, 0,   175, 175, 175, 0,   176, 176, 176, 0,   177, 177, 177,
  0,   178, 178, 178, 0,   179, 179, 179, 0,   180, 180, 180, 0,   181, 181,
  181, 0,   182, 182, 182, 0,   183, 183, 183, 0,   184, 184, 184, 0,   185,
  185, 185, 0,   186, 186, 186, 0,   187, 187, 187, 0,   188, 188, 188, 0,
  189, 189, 189, 0,   190, 190, 190, 0,   191, 191, 191, 0,   192, 192, 192,
  0,   193, 193, 193, 0,   194, 194, 194, 0,   195, 195, 195, 0,   196, 196,
  196, 0,   197, 197, 197, 0,   198, 198, 198, 0,   199, 199, 199, 0,   200,
  200, 200, 0,   201, 201, 201, 0,   202, 202, 202, 0,   203, 203, 203, 0,
  204, 204, 204, 0,   205, 205, 205, 0,   206, 206, 206, 0,   207, 207, 207,
  0,   208, 208, 208, 0,   209, 209, 209, 0,   210, 210, 210, 0,   211, 211,
  211, 0,   212, 212, 212, 0,   213, 213, 213, 0,   214, 214, 214, 0,   215,
  215, 215, 0,   216, 216, 216, 0,   217, 217, 217, 0,   218, 218, 218, 0,
  219, 219, 219, 0,   220, 220, 220, 0,   221, 221, 221, 0,   222, 222, 222,
  0,   223, 223, 223, 0,   224, 224, 224, 0,   225, 225, 225, 0,   226, 226,
  226, 0,   227, 227, 227, 0,   228, 228, 228, 0,   229, 229, 229, 0,   230,
  230, 230, 0,   231, 231, 231, 0,   232, 232, 232, 0,   233, 233, 233, 0,
  234, 234, 234, 0,   235, 235, 235, 0,   236, 236, 236, 0,   237, 237, 237,
  0,   238, 238, 238, 0,   239, 239, 239, 0,   240, 240, 240, 0,   241, 241,
  241, 0,   242, 242, 242, 0,   243, 243, 243, 0,   244, 244, 244, 0,   245,
  245, 245, 0,   246, 246, 246, 0,   247, 247, 247, 0,   248, 248, 248, 0,
  249, 249, 249, 0,   250, 250, 250, 0,   251, 251, 251, 0,   252, 252, 252,
  0,   253, 253, 253, 0,   254, 254, 254, 0,   255, 255, 255, 0,   135, 135,
  135, 135, 135, 135, 135, 135, 135, 135, 135, 135, 135, 135, 135, 135,
]);

/**
 * File for opaque sample BMP.
 */
export const OPAQUE_BMP_FILE =
    new File([OPAQUE_BMP_BYTES], 'valid-opaque.bmp', {type: 'image/bmp'});
