use core::{fmt::Debug, num::ParseIntError};
use jaq_json::{cbor, json, toml, xml};

fn unwrap_collect<T, E: Debug>(iter: impl Iterator<Item = Result<T, E>>) -> Vec<T> {
    iter.collect::<Result<Vec<_>, _>>().unwrap()
}

// Thanks to: <https://stackoverflow.com/a/52992629>
pub fn decode_hex(s: &str) -> Result<Vec<u8>, ParseIntError> {
    (0..s.len())
        .step_by(2)
        .map(|i| u8::from_str_radix(&s[i..i + 2], 16))
        .collect()
}

fn jc(json: &str, cbor_hex: &str) {
    //dbg!(json);
    let json_val = unwrap_collect(json::parse_many(json.as_bytes()));

    let cbor_bin = decode_hex(cbor_hex).unwrap();
    //dbg!(&cbor_bin);
    let cbor_val = unwrap_collect(cbor::parse_many(&*cbor_bin));
    assert_eq!(json_val, cbor_val);

    let mut cbor_bin2 = Vec::new();
    for v in &cbor_val {
        cbor::write(&mut cbor_bin2, v).unwrap()
    }
    //dbg!(&cbor_bin2);

    let cbor_val2 = unwrap_collect(cbor::parse_many(&*cbor_bin2));
    assert_eq!(cbor_val, cbor_val2);
}

#[test]
fn cbor() {
    jc("0", "00");
    jc("1", "01");
    jc("10", "0a");
    jc("23", "17");
    jc("24", "1818");
    jc("25", "1819");
    jc("100", "1864");
    jc("1000", "1903e8");
    jc("1000000", "1a000f4240");
    jc("1000000000000", "1b000000e8d4a51000");
    jc("18446744073709551615", "1bffffffffffffffff");
    jc("18446744073709551616", "c249010000000000000000");
    jc("-18446744073709551616", "3bffffffffffffffff");
    jc("-18446744073709551617", "c349010000000000000000");
    jc("-1", "20");
    jc("-10", "29");
    jc("-100", "3863");
    jc("-1000", "3903e7");
    jc("0.0", "f90000");
    jc("-0.0", "f98000");
    jc("1.0", "f93c00");
    jc("1.1", "fb3ff199999999999a");
    jc("1.5", "f93e00");
    jc("65504.0", "f97bff");
    jc("100000.0", "fa47c35000");
    jc("3.4028234663852886e+38", "fa7f7fffff");
    jc("1.0e+300", "fb7e37e43c8800759c");
    jc("5.960464477539063e-8", "f90001");
    jc("0.00006103515625", "f90400");
    jc("-4.0", "f9c400");
    jc("-4.1", "fbc010666666666666");
    /*
    jc("Infinity", "f97c00");
    jc("NaN", "f97e00");
    jc("-Infinity", "f9fc00");
    jc("Infinity", "fa7f800000");
    jc("NaN", "fa7fc00000");
    jc("-Infinity", "faff800000");
    jc("Infinity", "fb7ff0000000000000");
    jc("NaN", "fb7ff8000000000000");
    jc("-Infinity", "fbfff0000000000000");
    */
    jc("false", "f4");
    jc("true", "f5");
    jc("null", "f6");
    /*
    jc("undefined", "f7");
    jc("simple(16)", "f0");
    jc("simple(255)", "f8ff");
    jc("0(\"2013-03-21T20:04:00Z\")", "c074323031332d30332d32315432303a 30343a30305a");
    jc("1(1363896240)", "c11a514b67b0");
    jc("1(1363896240.5)", "c1fb41d452d9ec200000");
    jc("23(h'01020304')", "d74401020304");
    jc("24(h'6449455446')", "d818456449455446");
    jc("32(\"http://www.example.com\")", "d82076687474703a2f2f7777772e6578 616d706c652e636f6d");
    jc("h''", "40");
    jc("h'01020304'", "4401020304");
    */
    jc("\"\"", "60");
    jc("\"a\"", "6161");
    jc("\"IETF\"", "6449455446");
    jc("\"\\\"\\\\\"", "62225c");
    jc("\"\\u00fc\"", "62c3bc");
    jc("\"\\u6c34\"", "63e6b0b4");
    jc("\"\\ud800\\udd51\"", "64f0908591");
    jc("[]", "80");
    jc("[1, 2, 3]", "83010203");
    jc("[1, [2, 3], [4, 5]]", "8301820203820405");
    jc("[1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25]", "98190102030405060708090a0b0c0d0e0f101112131415161718181819");
    jc("{}", "a0");
    jc("{1: 2, 3: 4}", "a201020304");
    jc("{\"a\": 1, \"b\": [2, 3]}", "a26161016162820203");
    jc("[\"a\", {\"b\": \"c\"}]", "826161a161626163");
    jc(
        "{\"a\": \"A\", \"b\": \"B\", \"c\": \"C\", \"d\": \"D\", \"e\": \"E\"}",
        "a56161614161626142616361436164614461656145",
    );
    /*
    jc("(_ h'0102', h'030405')", "5f42010243030405ff");
    jc("(_ \"strea\", \"ming\")", "7f657374726561646d696e67ff");
    jc("[_ ]", "9fff");
    jc("[_ 1, [2, 3], [_ 4, 5]]", "9f018202039f0405ffff");
    jc("[_ 1, [2, 3], [4, 5]]", "9f01820203820405ff");
    jc("[1, [2, 3], [_ 4, 5]]", "83018202039f0405ff");
    jc("[1, [_ 2, 3], [4, 5]]", "83019f0203ff820405");
    jc("[_ 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25]", "9f0102030405060708090a0b0c0d0e0f101112131415161718181819ff");
    jc("{_ \"a\": 1, \"b\": [_ 2, 3]}", "bf61610161629f0203ffff");
    jc("[\"a\", {_ \"b\": \"c\"}]", "826161bf61626163ff");
    jc("{_ \"Fun\": true, \"Amt\": -2}", "bf6346756ef563416d7421ff");
    */
}

// TODO: test encoding!
#[test]
fn toml() {
    let json = include_bytes!("toml/test.json");
    let toml = include_str!("toml/test.toml");

    let json_val = json::parse_single(json).unwrap();
    let toml_val = toml::parse(toml).unwrap();

    assert_eq!(json_val, toml_val);
}

#[test]
fn xml() {
    let json = include_bytes!("xml/test.json");
    let xml = include_str!("xml/test.xhtml");

    let json_val = unwrap_collect(json::parse_many(json));
    let xml_val = unwrap_collect(xml::parse_many(xml));
    assert_eq!(json_val, xml_val);

    let serialise = |v| format!("{}\n", xml::Xml::try_from(v).unwrap());
    let xml2: Vec<_> = json_val.iter().map(serialise).collect();
    assert_eq!(xml, xml2.concat());
}
