# Copyright 1999-2025 Gentoo Authors
# Distributed under the terms of the GNU General Public License v2

EAPI=8

inherit multilib multilib-build toolchain-funcs

DESCRIPTION="Meta-ebuild for clang runtime libraries"
HOMEPAGE="https://clang.llvm.org/"
S=${WORKDIR}

LICENSE="public-domain"
SLOT="${PV%%.*}"
KEYWORDS="amd64 arm arm64 ~loong ~mips ppc ppc64 ~riscv ~sparc x86 ~arm64-macos ~x64-macos"
IUSE="+compiler-rt libcxx openmp +sanitize"
REQUIRED_USE="sanitize? ( compiler-rt )"

RDEPEND="
	compiler-rt? (
		~llvm-runtimes/compiler-rt-${PV}:${SLOT}[abi_x86_32(+)?,abi_x86_64(+)?]
		sanitize? (
			~llvm-runtimes/compiler-rt-sanitizers-${PV}:${SLOT}[abi_x86_32(+)?,abi_x86_64(+)?]
		)
	)
	libcxx? ( >=llvm-runtimes/libcxx-${PV}[${MULTILIB_USEDEP}] )
	openmp? ( >=llvm-runtimes/openmp-${PV}[${MULTILIB_USEDEP}] )

	llvm-core/clang-common

	~llvm-core/clang-linker-config-${SLOT}
	~llvm-runtimes/clang-rtlib-config-${SLOT}
	~llvm-runtimes/clang-unwindlib-config-${SLOT}
	~llvm-runtimes/clang-stdlib-config-${SLOT}
"

pkg_pretend() {
	if tc-is-clang; then
		ewarn "You seem to be using clang as a system compiler.  As of clang-16,"
		ewarn "upstream has turned a few warnings that commonly occur during"
		ewarn "configure script runs into errors by default.  This causes some"
		ewarn "configure tests to start failing, sometimes resulting in silent"
		ewarn "breakage, missing functionality or runtime misbehavior.  It is"
		ewarn "not yet clear whether the change will remain or be reverted."
		ewarn
		ewarn "For more information, please see:"
		ewarn "https://discourse.llvm.org/t/configure-script-breakage-with-the-new-werror-implicit-function-declaration/65213"
	fi
}

_doclang_cfg() {
	local triple="${1}"

	local tool
	for tool in ${triple}-clang{,++,-cpp}; do
		newins - "${tool}.cfg" <<-EOF
			# This configuration file is used by ${tool} driver.
			@../${tool}.cfg
			@gentoo-plugins.cfg
			@gentoo-runtimes.cfg
		EOF
	done

	# Install symlinks for triples with other vendor strings since some
	# programs insist on mangling the triple.
	local vendor
	for vendor in gentoo pc unknown; do
		local vendor_triple="${triple%%-*}-${vendor}-${triple#*-*-}"
		for tool in clang{,++,-cpp}; do
			if [[ ! -f "${ED}/etc/clang/${SLOT}/${vendor_triple}-${tool}.cfg" ]]; then
				dosym "${triple}-${tool}.cfg" "/etc/clang/${SLOT}/${vendor_triple}-${tool}.cfg"
			fi
		done
	done
}

doclang_cfg() {
	local triple=$(get_abi_CHOST "${abi}")

	_doclang_cfg ${triple}

	# LLVM may have different arch names in some cases. For example in x86
	# profiles the triple uses i686, but llvm will prefer i386 if invoked
	# with "clang" on x86 or "clang -m32" on x86_64. The gentoo triple will
	# be used if invoked through ${CHOST}-clang{,++,-cpp} though.
	#
	# To make sure the correct triples are installed,
	# see Triple::getArchTypeName() in llvm/lib/TargetParser/Triple.cpp
	# and compare with CHOST values in profiles.

	local abi=${triple%%-*}
	case ${abi} in
		armv4l|armv4t|armv5tel|armv6j|armv7a)
			_doclang_cfg ${triple/${abi}/arm}
			;;
		i686)
			_doclang_cfg ${triple/${abi}/i386}
			;;
		sparc)
			_doclang_cfg ${triple/${abi}/sparcel}
			;;
		sparc64)
			_doclang_cfg ${triple/${abi}/sparcv9}
			;;
	esac
}

src_install() {
	insinto "/etc/clang/${SLOT}"
	newins - gentoo-runtimes.cfg <<-EOF
		# This file is initially generated by llvm-runtimes/clang-runtime.
		# It is used to control the default runtimes using by clang.

		@gentoo-rtlib.cfg
		@gentoo-unwindlib.cfg
		@gentoo-stdlib.cfg
		@gentoo-linker.cfg
	EOF
	newins - gentoo-plugins.cfg <<-EOF
		# This file is used to load optional LLVM plugins.
	EOF

	multilib_foreach_abi doclang_cfg
}
