/*
 * Copyright 2024-2025 The OpenSSL Project Authors. All Rights Reserved.
 *
 * Licensed under the Apache License 2.0 (the "License").  You may not use
 * this file except in compliance with the License.  You can obtain a copy
 * in the file LICENSE in the source distribution or at
 * https://www.openssl.org/source/license.html
 */

#define SLH_DSA_SIG_TEST_DET_ITEM(name, alg) {                                 \
    alg,                                                                       \
    name##_priv + sizeof(name##_priv) / 2, sizeof(name##_priv) / 2,            \
    name##_priv, sizeof(name##_priv),                                          \
    name##_msg, sizeof(name##_msg),                                            \
    name##_sig_digest, sizeof(name##_sig_digest),                              \
}

#define SLH_DSA_SIG_TEST_ITEM(name, alg) {                                     \
    alg,                                                                       \
    name##_priv + sizeof(name##_priv) / 2, sizeof(name##_priv) / 2,            \
    name##_priv, sizeof(name##_priv),                                          \
    name##_msg, sizeof(name##_msg),                                            \
    name##_sig_digest, sizeof(name##_sig_digest),                              \
    name##_add_random, sizeof(name##_add_random),                              \
}

#define SLH_DSA_KEYGEN_TEST_ITEM(name, alg) {                                  \
    alg,                                                                       \
    name##_keygen_priv, sizeof(name##_keygen_priv),                            \
}

typedef struct slh_dsa_sig_test_data_st {
    const char *alg;
    const unsigned char *pub;
    size_t pub_len;
    const unsigned char *priv;
    size_t priv_len;
    const unsigned char *msg;
    size_t msg_len;
    /* This is the sha256 digest of the signature, to reduce its size */
    const unsigned char *sig_digest;
    size_t sig_digest_len;
    const unsigned char *add_random;
    size_t add_random_len;
} SLH_DSA_SIG_TEST_DATA;

typedef struct slh_dsa_keygen_test_data_st {
    const char *name;
    const uint8_t *priv;
    size_t priv_len;
} SLH_DSA_KEYGEN_TEST_DATA;

/*
 * Test vectors from
 * usnistgov/ACVP-Server/refs/heads/master/gen-val/json-files/SLH-DSA-sigGen-FIPS205/internalProjection.json
 * Note that the public key is the second half of the private key.
 */
static const uint8_t slh_dsa_sha2_128s_0_priv[] = {
    0x62, 0xb1, 0x97, 0x3a, 0x4d, 0xe0, 0x96, 0x3d, 0x74, 0xc1, 0xcb, 0x30, 0xfc, 0x8f, 0x56, 0x75,
    0xcf, 0xc8, 0x48, 0x80, 0xe4, 0xf0, 0xe1, 0xb4, 0x46, 0xb4, 0xf5, 0xd1, 0x3b, 0x2d, 0x31, 0xcc,
    0xcb, 0x23, 0xeb, 0x45, 0x52, 0x9e, 0x00, 0xd5, 0xf5, 0xe9, 0x51, 0x50, 0x7a, 0x9b, 0x90, 0xe9,
    0x8b, 0x6e, 0x7a, 0x28, 0x4b, 0xa3, 0xf6, 0x3a, 0x69, 0xe6, 0x9a, 0x78, 0x90, 0x83, 0xbe, 0xf6,
};
static const uint8_t slh_dsa_sha2_128s_0_msg[] = {
    0x9D, 0xDF
};
static const uint8_t slh_dsa_sha2_128s_0_sig_digest[] = {
    0xc7, 0xdf, 0xf0, 0xed, 0x25, 0x38, 0x49, 0xef, 0x51, 0x1e, 0x90, 0xbe, 0x0e, 0x2e, 0xb7, 0x71,
    0x65, 0x98, 0x91, 0x23, 0x17, 0x52, 0x9a, 0x61, 0xda, 0xe4, 0x32, 0x9b, 0xf1, 0x49, 0xef, 0x8b
};

static const uint8_t slh_dsa_sha2_192f_0_priv[] = {
    0xc1, 0x0a, 0xdc, 0x69, 0x2b, 0x76, 0xff, 0x6f, 0x34, 0xa6, 0xf0, 0xc8, 0xff, 0xe2, 0xbb, 0x88,
    0xe8, 0x41, 0xeb, 0xd9, 0x2d, 0xa7, 0xff, 0xec, 0xdd, 0x3b, 0xf6, 0xc1, 0x05, 0x66, 0xa0, 0xcf,
    0x69, 0x8e, 0x39, 0x3d, 0xfb, 0x84, 0x58, 0x7d, 0x3b, 0xc8, 0xc7, 0xde, 0x58, 0x69, 0x66, 0x8c,
    0x36, 0xf7, 0xaa, 0x6b, 0x6f, 0x82, 0xa7, 0xf3, 0xc1, 0x09, 0xe8, 0x81, 0x73, 0xa1, 0xe5, 0xf4,
    0x0f, 0xa5, 0x29, 0xa2, 0x9f, 0x48, 0x5c, 0xcb, 0x40, 0xbb, 0x24, 0x7b, 0x9c, 0xba, 0xda, 0x95,
    0x83, 0x92, 0x42, 0x5b, 0xff, 0x8b, 0x20, 0x28, 0x9b, 0x0f, 0x11, 0xaa, 0x3a, 0x51, 0x4e, 0x54,
};
static const uint8_t slh_dsa_sha2_192f_0_add_random[] = {
    0xe6, 0xe7, 0x1a, 0x97, 0x57, 0x07, 0x75, 0xb3, 0x54, 0x98, 0x28, 0x7f, 0x81, 0x8d, 0x8c, 0x2a,
    0x96, 0x00, 0x05, 0x76, 0xc8, 0x5c, 0x0c, 0x71,
};
static const uint8_t slh_dsa_sha2_192f_0_msg[] = { 0x0E, 0x54 };
static const uint8_t slh_dsa_sha2_192f_0_sig_digest[] = {
    0xd0, 0x5e, 0xec, 0x26, 0xb1, 0x2e, 0x21, 0x9b, 0x83, 0xe2, 0x96, 0x84, 0xff, 0x88, 0xa8, 0x2a,
    0x49, 0xe4, 0xd2, 0xc5, 0x2e, 0x43, 0x8b, 0xcf, 0x15, 0xa5, 0xf6, 0x18, 0x10, 0x51, 0x6c, 0x2e,
};
static const uint8_t slh_dsa_sha2_256f_0_priv[] = {
    0x1a, 0x5a, 0x6a, 0x0c, 0x8a, 0xae, 0x35, 0x9c, 0xda, 0xce, 0x45, 0x67, 0x42, 0xc3, 0x71, 0x23,
    0xe8, 0xc5, 0xad, 0x9c, 0xfe, 0xab, 0x75, 0x1b, 0x1d, 0x92, 0x11, 0x6c, 0x81, 0xd6, 0xf5, 0x2c,
    0xb0, 0x10, 0x08, 0x63, 0x37, 0xe5, 0xfb, 0x27, 0xe6, 0x74, 0xc7, 0xf6, 0xee, 0x14, 0x4c, 0xec,
    0x10, 0xee, 0xf8, 0x87, 0x80, 0xf1, 0x0f, 0x0c, 0xf3, 0xf7, 0x43, 0xc8, 0xb7, 0xdb, 0x15, 0xce,
    0x03, 0x90, 0xbf, 0x5e, 0xcc, 0x8f, 0xf5, 0xa0, 0x07, 0x23, 0x6c, 0x56, 0xbf, 0x9d, 0x9d, 0x46,
    0xa3, 0x53, 0x4e, 0x54, 0x58, 0x36, 0xcf, 0x7a, 0xdf, 0x20, 0x0d, 0xd2, 0x86, 0x2f, 0x91, 0x1b,
    0x74, 0x09, 0x5e, 0x73, 0xe6, 0x36, 0x41, 0xfa, 0xb2, 0x6b, 0xd7, 0x17, 0xab, 0x29, 0xbb, 0x1e,
    0xd8, 0x73, 0x5c, 0x53, 0x61, 0x32, 0xa1, 0x9e, 0xac, 0x5d, 0xae, 0xd8, 0xa0, 0x7c, 0xf3, 0xb5,
};
static const uint8_t slh_dsa_sha2_256f_0_msg[] = {
    0xDC, 0x09
};
static const uint8_t slh_dsa_sha2_256f_0_sig_digest[] = {
    0x13, 0xb7, 0x78, 0xcd, 0x29, 0xf4, 0x55, 0x35, 0xcc, 0xe4, 0x56, 0xad, 0x36, 0xff, 0xad, 0x50,
    0x54, 0xc3, 0x14, 0xbe, 0x5f, 0x59, 0xa5, 0x9a, 0xcf, 0xa5, 0x18, 0xb1, 0x6a, 0x6f, 0x57, 0xf0,
};

static const uint8_t slh_dsa_shake_128s_0_priv[] = {
    0x4c, 0xf2, 0xbb, 0xc4, 0x7b, 0x14, 0x51, 0x80, 0xd1, 0xee, 0xa2, 0x98, 0x0c, 0xfb, 0xe2, 0xdc,
    0xc4, 0xf0, 0x70, 0xaf, 0x5b, 0xaf, 0x81, 0xab, 0x2e, 0xae, 0x38, 0x14, 0x4c, 0x35, 0x76, 0xa5,
    0x10, 0x96, 0xaa, 0x01, 0x4c, 0x35, 0x03, 0x02, 0xe1, 0x96, 0x31, 0x7e, 0x0d, 0xbf, 0xb2, 0x91,
    0xd2, 0x4e, 0x52, 0x92, 0x09, 0xde, 0xb4, 0x97, 0x03, 0xcc, 0x3a, 0xd2, 0x9d, 0xfc, 0xe2, 0x4d,
};
static const uint8_t slh_dsa_shake_128s_0_msg[] = { 0xD3, 0x7C };
static const uint8_t slh_dsa_shake_128s_0_sig_digest[] = {
    0xcb, 0xc5, 0xbc, 0x43, 0x51, 0xa1, 0x16, 0xcb, 0x7b, 0x38, 0x28, 0x2c, 0xf9, 0xc9, 0x95, 0x56,
    0x2b, 0x73, 0x6e, 0x85, 0xdc, 0xa6, 0xad, 0xda, 0x42, 0xce, 0x39, 0x90, 0xb8, 0x91, 0xa4, 0x13,
};

static const uint8_t slh_dsa_shake_256f_0_priv[] = {
    0xf7, 0x1d, 0x42, 0xf7, 0x34, 0xb0, 0x73, 0x93, 0x88, 0x74, 0x38, 0xfc, 0x47, 0xc8, 0x5e, 0x60,
    0x23, 0x92, 0x9d, 0xae, 0xeb, 0x86, 0x44, 0xe7, 0xe5, 0xbd, 0xb9, 0x4d, 0xf0, 0x0a, 0x8e, 0x48,
    0xa4, 0xee, 0x88, 0x08, 0xd9, 0x5d, 0x81, 0x1a, 0x10, 0x55, 0xe0, 0xb1, 0xba, 0x89, 0xe0, 0x8b,
    0xbd, 0xf7, 0x2b, 0xe2, 0x91, 0x2e, 0x2b, 0x7f, 0x6b, 0xf9, 0x73, 0xa6, 0x55, 0xcd, 0x5b, 0x1b,
    0xf6, 0x62, 0x20, 0x15, 0x24, 0x3a, 0xbd, 0x05, 0x3b, 0x45, 0x63, 0xb9, 0x16, 0xa8, 0xf3, 0x7c,
    0x67, 0xa6, 0xe3, 0x0a, 0xd1, 0xc8, 0x5d, 0x65, 0xed, 0x7d, 0x3c, 0xcf, 0x53, 0xd0, 0x8f, 0x27,
    0xde, 0xea, 0x9d, 0x3e, 0x58, 0xe4, 0xc1, 0x86, 0xb3, 0x75, 0x70, 0x11, 0x0f, 0x35, 0xd5, 0xd4,
    0xe4, 0xcf, 0x36, 0xbc, 0x9e, 0x5b, 0x26, 0xa4, 0x75, 0xbe, 0x3f, 0x36, 0x26, 0xd5, 0x3e, 0xcb,
};
static const uint8_t slh_dsa_shake_256f_0_msg[] = { 0x52 };
static const uint8_t slh_dsa_shake_256f_0_sig_digest[] = {
    0x83, 0x30, 0x10, 0x89, 0x2d, 0x88, 0xfe, 0xf0, 0xf6, 0x6a, 0x81, 0x46, 0xab, 0x8d, 0x8f, 0x73,
    0xfe, 0xe5, 0xf8, 0xec, 0xea, 0xad, 0x8c, 0xdc, 0x36, 0xfc, 0xb8, 0x9a, 0xf5, 0xaa, 0x9c, 0x41,
};

static const uint8_t slh_dsa_shake_128f_0_priv[] = {
    0x75, 0x10, 0x49, 0x1b, 0x47, 0x0a, 0xe3, 0x65, 0x8e, 0xd5, 0x0a, 0xd7, 0xa4, 0xe6, 0xc5, 0x8c,
    0x86, 0x33, 0x67, 0x42, 0xc3, 0x24, 0x4b, 0x95, 0xaa, 0x33, 0x8b, 0x70, 0x89, 0xdf, 0x84, 0xb8,
    0x3d, 0x17, 0x30, 0x69, 0x3a, 0xdd, 0xc8, 0x9b, 0xf7, 0xb8, 0x1e, 0x32, 0x13, 0x6a, 0x28, 0xc6,
    0xec, 0x8d, 0x96, 0xc4, 0x6d, 0x3b, 0xbe, 0x28, 0x12, 0xa5, 0x5b, 0x98, 0xe3, 0x89, 0x26, 0xae,
};
static const uint8_t slh_dsa_shake_128f_0_add_random[] = {
    0xfe, 0x4b, 0x12, 0xe6, 0x8b, 0x0e, 0x3b, 0x91, 0x01, 0xb2, 0x0c, 0x17, 0xc5, 0x71, 0x1c, 0x59,
};
static const uint8_t slh_dsa_shake_128f_0_msg[] = {
    0x23, 0x95, 0xf6, 0x4a, 0x2a, 0x92, 0x69, 0x86, 0x32, 0x77, 0x73, 0x1e, 0x43, 0x57, 0x70, 0x01,
    0x8f, 0xc3, 0x18, 0xd8, 0xf6, 0x4b, 0x5b, 0x7d, 0x7d, 0x91, 0xd5, 0x2d, 0xaa, 0x7e, 0x0d, 0x1e,
    0x51, 0xcf, 0xc8, 0xf9, 0xe1, 0x14, 0x7b, 0x8e, 0xe9, 0x40, 0x67, 0xa3, 0x87, 0x61, 0xf7, 0x64,
    0x02, 0xb5, 0x55, 0xb1, 0x04, 0x17, 0x5b, 0xeb, 0x8f, 0xd8, 0x5e, 0xa5, 0x20, 0x52, 0xc6, 0x99,
    0x1a, 0xfb, 0x7f, 0x3b, 0xbb, 0x72, 0xb6, 0xe5, 0x75, 0x1d, 0xef, 0x52, 0x37, 0xeb, 0x55, 0xdc,
    0x37, 0xf5, 0x7f, 0x71, 0x17, 0x28, 0x19, 0xd1, 0x0a, 0x84, 0x6c, 0x97, 0xf9, 0xf0, 0xab, 0xd9,
    0x92, 0x9e, 0x8e, 0xdd, 0x3e, 0x2f, 0x5a, 0x82, 0x26, 0x6d, 0xda, 0x1e, 0xe8, 0x66, 0x6f, 0xfc,
    0x6f, 0x58, 0x96, 0xcc, 0x2c, 0x59, 0x37, 0xa9, 0xd8, 0xbe, 0x6a, 0x16, 0xf5, 0xdd, 0x52, 0x4f,
    0x3b, 0xd8, 0xb9, 0xa6, 0xe7, 0xc5, 0x34, 0xcd, 0xc1, 0xac, 0xcb, 0x4f, 0xd1, 0xf1, 0x90, 0x75,
    0x9a, 0x61, 0x74, 0x4b, 0x3f, 0xe7, 0xb1, 0x4c, 0x44, 0x76, 0xa2, 0x73, 0xfd, 0x1c, 0x2e, 0xf1,
    0xbd, 0xab, 0x0f, 0x78, 0xdc, 0x09, 0x5e, 0xe4, 0xa3,
};
static const uint8_t slh_dsa_shake_128f_0_sig_digest[] = {
    0xd9, 0xee, 0x15, 0x31, 0xeb, 0xab, 0x30, 0xd5, 0xd9, 0xb1, 0x0f, 0x43, 0x74, 0xbd, 0xf9, 0xf6,
    0xc0, 0xb5, 0x87, 0xd0, 0xa2, 0x19, 0x0c, 0x9e, 0x2c, 0x98, 0xc7, 0xa3, 0x71, 0x17, 0x4a, 0x41,
};

static SLH_DSA_SIG_TEST_DATA slh_dsa_sig_testdata[] = {
    SLH_DSA_SIG_TEST_DET_ITEM(slh_dsa_sha2_128s_0, "SLH-DSA-SHA2-128s"),
    SLH_DSA_SIG_TEST_ITEM(slh_dsa_sha2_192f_0, "SLH-DSA-SHA2-192f"),
    SLH_DSA_SIG_TEST_DET_ITEM(slh_dsa_sha2_256f_0, "SLH-DSA-SHA2-256f"),
    SLH_DSA_SIG_TEST_DET_ITEM(slh_dsa_shake_128s_0, "SLH-DSA-SHAKE-128s"),
    SLH_DSA_SIG_TEST_ITEM(slh_dsa_shake_128f_0, "SLH-DSA-SHAKE-128f"),
    SLH_DSA_SIG_TEST_DET_ITEM(slh_dsa_shake_256f_0, "SLH-DSA-SHAKE-256f"),
};

static const uint8_t slh_dsa_sha2_128s_0_keygen_priv[] = {
    0xaa, 0x9c, 0xc7, 0xdc, 0xa4, 0x91, 0xfc, 0x86, 0xbc, 0xb1, 0x5a, 0x70, 0x9a, 0x15, 0xe9, 0xb3,
    0x90, 0x5c, 0x80, 0x0b, 0x6e, 0x2f, 0xb9, 0xb5, 0x4b, 0x6b, 0x05, 0x0e, 0xe5, 0xe4, 0xde, 0x9a,
    0xfa, 0x54, 0x64, 0xd1, 0xc6, 0x61, 0xfe, 0xd3, 0x8b, 0x2a, 0x51, 0xca, 0x3e, 0xae, 0x71, 0xba,
    0xca, 0xe3, 0xd1, 0x86, 0x52, 0x15, 0xe3, 0xd3, 0x85, 0x0e, 0x8c, 0x1b, 0x82, 0x92, 0xbf, 0x42,
};
static const uint8_t slh_dsa_sha2_128f_0_keygen_priv[] = {
    0xe2, 0xbd, 0xaa, 0x37, 0xc8, 0xcf, 0xfe, 0x5e, 0x8d, 0x56, 0x76, 0xc2, 0x32, 0x67, 0x89, 0x0c,
    0x31, 0x44, 0x17, 0x58, 0xf5, 0x73, 0x28, 0x58, 0x81, 0xcd, 0xc8, 0x2a, 0xb9, 0x11, 0xdd, 0x84,
    0x72, 0xdc, 0x8d, 0x26, 0xdf, 0x6e, 0xf7, 0x08, 0xf4, 0xc4, 0x1a, 0xf9, 0xfd, 0x04, 0xb6, 0x5a,
    0x89, 0x27, 0x92, 0x72, 0x29, 0x89, 0x1d, 0x47, 0xa6, 0x0d, 0x67, 0xec, 0xef, 0x3d, 0x2c, 0x17,
};
static const uint8_t slh_dsa_sha2_192s_0_keygen_priv[] = {
    0x44, 0x2e, 0x44, 0x6e, 0x73, 0x33, 0x0a, 0xfb, 0x98, 0x70, 0x46, 0x56, 0x32, 0x8f, 0x4d, 0xd7,
    0x33, 0x4f, 0x8a, 0x9a, 0xb4, 0xd9, 0x2c, 0xbe, 0x79, 0x0f, 0x91, 0xc2, 0xe9, 0x2a, 0x81, 0xaf,
    0xee, 0x0a, 0x7e, 0xc4, 0x5a, 0x3d, 0x60, 0x93, 0x46, 0x27, 0x6f, 0x5a, 0x32, 0x87, 0x55, 0xa1,
    0x7d, 0xd4, 0x16, 0x08, 0xf5, 0x9e, 0x49, 0x26, 0x68, 0xd8, 0x1d, 0x03, 0x44, 0x13, 0x94, 0xfe,
    0x7a, 0x8e, 0x7e, 0x58, 0x70, 0x5d, 0x76, 0x32, 0xba, 0x8b, 0xc6, 0x6d, 0x04, 0xc9, 0x9e, 0xe5,
    0xc7, 0x21, 0xe4, 0xff, 0x4b, 0xbe, 0x78, 0x15, 0xcc, 0xf9, 0x23, 0xe8, 0x1e, 0x10, 0xc4, 0x75,
};
static const uint8_t slh_dsa_sha2_192f_0_keygen_priv[] = {
    0x94, 0x2d, 0xd5, 0x88, 0xb0, 0xe0, 0x63, 0x60, 0x60, 0xb1, 0x02, 0x3b, 0xaf, 0x1d, 0xf3, 0x60,
    0x02, 0xb9, 0xd3, 0xc7, 0x25, 0x69, 0x04, 0xf9, 0x48, 0x59, 0x6f, 0x0f, 0x2f, 0x17, 0x50, 0x55,
    0x85, 0xf0, 0x22, 0x42, 0x34, 0x33, 0xb3, 0xa3, 0xef, 0x50, 0x97, 0x7b, 0x98, 0xb0, 0x1e, 0x8e,
    0x3e, 0x1d, 0x7d, 0xcf, 0x88, 0x0c, 0x2e, 0xa3, 0xc8, 0x87, 0x21, 0x22, 0xa9, 0x96, 0xd5, 0xd6,
    0x23, 0x3b, 0xdf, 0xbd, 0x57, 0xfa, 0x64, 0x1f, 0xad, 0x3c, 0x81, 0xbb, 0xe6, 0x77, 0x8b, 0x1f,
    0x78, 0x81, 0x95, 0x13, 0x1b, 0x3f, 0xca, 0x91, 0x85, 0x4f, 0xf3, 0xb0, 0x75, 0xbf, 0x00, 0x09,
};
static const uint8_t slh_dsa_sha2_256s_0_keygen_priv[] = {
    0xae, 0x6c, 0xa8, 0x66, 0x4e, 0xc0, 0xb9, 0x17, 0x9d, 0x4e, 0x33, 0xc4, 0xde, 0xfe, 0x01, 0xfc,
    0xd5, 0x89, 0xf6, 0x0b, 0xa4, 0x50, 0x2f, 0xe7, 0x41, 0x6f, 0x2a, 0xcd, 0x96, 0xe1, 0x39, 0xf1,
    0x3f, 0xda, 0x20, 0x69, 0x14, 0x7f, 0x44, 0xea, 0xbd, 0x5b, 0xbf, 0x29, 0xc7, 0x4a, 0x20, 0xcb,
    0x0f, 0x0c, 0xc2, 0xa1, 0x2b, 0xab, 0x58, 0x34, 0xb7, 0x73, 0x53, 0x0a, 0xf5, 0x04, 0x90, 0x0a,
    0x13, 0x4d, 0x5e, 0xd3, 0xc6, 0x0b, 0x46, 0x34, 0x4a, 0x84, 0xa4, 0x5d, 0x46, 0x83, 0xb1, 0xac,
    0x55, 0xfb, 0x22, 0x88, 0x6b, 0xa9, 0x47, 0x8e, 0xa9, 0xca, 0x93, 0xf2, 0x7b, 0x9a, 0xa2, 0xc2,
    0x7c, 0x5c, 0x99, 0x08, 0xf0, 0x86, 0xe5, 0x79, 0x56, 0xf8, 0x5d, 0xe8, 0x4b, 0x43, 0x8e, 0xf1,
    0xf0, 0x82, 0xcd, 0x17, 0x6d, 0xff, 0x3c, 0x5b, 0x8b, 0xe7, 0x10, 0xbc, 0x86, 0x99, 0xa1, 0x43,
};
static const uint8_t slh_dsa_sha2_256f_0_keygen_priv[] = {
    0x37, 0xd1, 0x80, 0x6d, 0xd0, 0x90, 0x35, 0x30, 0x64, 0xa9, 0x82, 0x27, 0x6a, 0xeb, 0xd2, 0x0e,
    0x73, 0x18, 0xf3, 0x6b, 0x17, 0xec, 0x52, 0x09, 0xd4, 0x00, 0x66, 0x51, 0x76, 0x0c, 0xa0, 0x43,
    0x89, 0x66, 0x30, 0xd5, 0x1c, 0x45, 0xa8, 0xf7, 0xc1, 0xda, 0x31, 0x19, 0x2f, 0x41, 0x20, 0x4d,
    0xeb, 0x71, 0xbb, 0xc4, 0xfb, 0x47, 0x70, 0x0a, 0x91, 0xec, 0x47, 0xbb, 0x4a, 0xcf, 0x3a, 0x38,
    0xda, 0xbb, 0xd0, 0x01, 0x12, 0x52, 0x0e, 0x60, 0x06, 0x1d, 0xa2, 0x3f, 0x4c, 0x83, 0xa5, 0xc4,
    0x75, 0xa0, 0xad, 0x3a, 0xb7, 0x5f, 0x59, 0xe5, 0xc7, 0xb8, 0xce, 0x24, 0x30, 0xde, 0xb4, 0x80,
    0xc0, 0x78, 0x49, 0xdd, 0x9f, 0x8d, 0x99, 0x3d, 0xce, 0x56, 0x85, 0xd0, 0x84, 0x01, 0x34, 0xbd,
    0x11, 0x59, 0x20, 0x55, 0xf7, 0x4c, 0xde, 0x3a, 0xa9, 0xe7, 0xd2, 0x5b, 0x33, 0xc3, 0x10, 0x67,
};

static const uint8_t slh_dsa_shake_128s_0_keygen_priv[] = {
    0xec, 0xd0, 0xa0, 0x56, 0x96, 0xc2, 0xec, 0x6b, 0xfe, 0x85, 0xb8, 0xb1, 0xb2, 0x02, 0x24, 0x86,
    0xbf, 0xd8, 0x73, 0x2b, 0x1a, 0x42, 0xce, 0xa2, 0x36, 0xe9, 0x82, 0x2a, 0xa9, 0xe7, 0x24, 0xe6,
    0x37, 0xd6, 0x64, 0x79, 0x21, 0x98, 0x17, 0x6c, 0x12, 0xe8, 0x12, 0x1c, 0xcf, 0x69, 0xe6, 0x84,
    0xfc, 0xf6, 0xff, 0xe2, 0xa0, 0x42, 0x43, 0xdc, 0x3e, 0x8e, 0xd3, 0xee, 0x6e, 0x44, 0xcd, 0xef,
};
static const uint8_t slh_dsa_shake_128f_0_keygen_priv[] = {
    0xbb, 0xc7, 0x43, 0x06, 0xf7, 0x5d, 0xc2, 0xda, 0xf7, 0x37, 0x2b, 0x3c, 0x98, 0x41, 0xa4, 0xd6,
    0x85, 0x2c, 0x17, 0xb4, 0x59, 0xf1, 0x69, 0x2b, 0x8e, 0x9a, 0x1a, 0x0d, 0xac, 0xe5, 0xba, 0x26,
    0x38, 0x0c, 0x99, 0x30, 0x4a, 0x0d, 0xdd, 0x32, 0xf3, 0x44, 0xb9, 0x51, 0x44, 0xe1, 0xfd, 0xef,
    0x60, 0xbb, 0xc2, 0x34, 0x0e, 0x08, 0x77, 0x0f, 0xb4, 0x1a, 0x80, 0xa7, 0x6c, 0xb0, 0x8e, 0x34,
};
static const uint8_t slh_dsa_shake_192s_0_keygen_priv[] = {
    0x2d, 0x57, 0x48, 0xb1, 0xac, 0xda, 0x82, 0x99, 0xe2, 0xe9, 0x05, 0x54, 0x5d, 0x2e, 0x66, 0xdf,
    0xce, 0xc0, 0x24, 0x61, 0xbc, 0x36, 0x73, 0xdd, 0x9d, 0x83, 0x04, 0x16, 0xb1, 0xb8, 0x2a, 0x66,
    0x5d, 0xd4, 0x35, 0xa3, 0xf3, 0xa8, 0xe2, 0x12, 0x4d, 0xf5, 0x9a, 0x0e, 0x2b, 0xc2, 0xec, 0x73,
    0x63, 0x3f, 0xf6, 0x5c, 0x72, 0x4a, 0xe1, 0x70, 0x45, 0xc0, 0x94, 0x61, 0x75, 0x49, 0x08, 0x1c,
    0x94, 0x0f, 0x2e, 0xf4, 0xd4, 0xa1, 0x0e, 0x75, 0x59, 0xc7, 0x50, 0x3f, 0xe1, 0xe9, 0x6f, 0xb6,
    0x7b, 0x0e, 0x32, 0xac, 0xdc, 0xf7, 0x12, 0xfb, 0xef, 0x61, 0x0f, 0x8d, 0x21, 0xda, 0xac, 0x86,
};
static const uint8_t slh_dsa_shake_192f_0_keygen_priv[] = {
    0x3b, 0x78, 0x89, 0xd2, 0x54, 0xf5, 0xad, 0x87, 0x88, 0x2d, 0xac, 0xbf, 0x3c, 0x58, 0x49, 0x56,
    0x67, 0x40, 0x16, 0x01, 0xec, 0x47, 0x39, 0x79, 0x7e, 0x93, 0x5c, 0xbc, 0x27, 0xf3, 0xd7, 0xcd,
    0x62, 0x7b, 0x61, 0x1c, 0xbe, 0xf7, 0x54, 0x6c, 0x7f, 0x27, 0x75, 0x1e, 0x18, 0x15, 0x5c, 0xc4,
    0x26, 0x6b, 0x6f, 0xb5, 0x6b, 0xd0, 0xd9, 0x22, 0xe0, 0x48, 0x27, 0x00, 0xa4, 0x28, 0x0c, 0xc1,
    0xf0, 0x70, 0x52, 0x80, 0x4a, 0x8f, 0x46, 0xfd, 0xed, 0x39, 0x54, 0xba, 0x0a, 0x9d, 0xa4, 0x5d,
    0x0d, 0x18, 0x86, 0x74, 0x37, 0x77, 0x7f, 0x60, 0x21, 0x1b, 0xb9, 0xcd, 0x03, 0xa6, 0x0d, 0x4b,
};
static const uint8_t slh_dsa_shake_256s_0_keygen_priv[] = {
    0xa6, 0x53, 0x91, 0x8f, 0xa5, 0x57, 0xb5, 0x25, 0x0d, 0x11, 0xe9, 0x6a, 0x74, 0xf5, 0xd8, 0xc6,
    0x2b, 0x5f, 0xf6, 0x71, 0x52, 0xac, 0xf2, 0x27, 0xaa, 0xf4, 0xbf, 0x52, 0x11, 0xaa, 0x60, 0x0e,
    0x96, 0x1b, 0x4e, 0x67, 0x9c, 0xeb, 0x9d, 0x4e, 0x24, 0xdf, 0x86, 0xed, 0x22, 0xf4, 0x05, 0xd8,
    0xed, 0xe5, 0x62, 0x36, 0x72, 0xf4, 0x73, 0x7f, 0xc2, 0xfb, 0x65, 0xf8, 0xbf, 0x3b, 0x55, 0x6c,
    0x31, 0x5a, 0x5e, 0x79, 0x5b, 0x70, 0x16, 0x76, 0xe6, 0xb8, 0x30, 0x9c, 0x89, 0xcc, 0x6a, 0xd0,
    0xc3, 0xd6, 0x9f, 0xf4, 0xf7, 0xf4, 0x32, 0xdc, 0x61, 0x32, 0x9f, 0xbb, 0x1b, 0xe6, 0x06, 0x9f,
    0x08, 0x87, 0xb1, 0xa8, 0xf1, 0xb3, 0xa8, 0xb9, 0x77, 0x0a, 0x08, 0x71, 0x63, 0x0e, 0x46, 0xdd,
    0x04, 0x79, 0xc5, 0xbd, 0x12, 0x3e, 0x8e, 0xeb, 0x6b, 0xe7, 0x0f, 0x29, 0x99, 0xc4, 0x66, 0xa1,
};
static const uint8_t slh_dsa_shake_256f_0_keygen_priv[] = {
    0xd5, 0x0f, 0x77, 0x71, 0x3c, 0xf5, 0x91, 0x93, 0xfe, 0x5f, 0x10, 0xb8, 0xaa, 0xf1, 0x87, 0x6b,
    0x0d, 0x3e, 0x7e, 0x41, 0x81, 0xed, 0x3f, 0xae, 0x34, 0x68, 0x76, 0x26, 0x98, 0x01, 0x75, 0x84,
    0x36, 0xc2, 0x2b, 0x81, 0x94, 0x04, 0x03, 0x15, 0x74, 0x46, 0xd4, 0x4f, 0x04, 0xc0, 0x9f, 0xd2,
    0x8d, 0xf7, 0x2c, 0x6d, 0x40, 0x21, 0xb2, 0x71, 0x54, 0x40, 0x55, 0x78, 0x30, 0x81, 0xb7, 0xc0,
    0x46, 0xd2, 0xac, 0x50, 0x75, 0x1a, 0x0c, 0x71, 0xec, 0x85, 0x0b, 0xb7, 0xd1, 0x6a, 0x67, 0x9a,
    0x66, 0x90, 0x1e, 0xee, 0x05, 0x38, 0x9d, 0x46, 0x15, 0x94, 0x0e, 0xfd, 0x89, 0xc0, 0x18, 0x2f,
    0xde, 0xbc, 0xfb, 0xf3, 0x52, 0x7c, 0xc7, 0x68, 0xc5, 0xec, 0x6c, 0xb5, 0xb6, 0xcc, 0x10, 0xa6,
    0x74, 0x39, 0x07, 0x45, 0xd8, 0x61, 0x2d, 0x09, 0x20, 0x18, 0x7d, 0xf6, 0x58, 0xd3, 0xb7, 0xef,
};

static const SLH_DSA_KEYGEN_TEST_DATA slh_dsa_keygen_testdata[] = {
    SLH_DSA_KEYGEN_TEST_ITEM(slh_dsa_sha2_128s_0, "SLH-DSA-SHA2-128s"),
    SLH_DSA_KEYGEN_TEST_ITEM(slh_dsa_sha2_128f_0, "SLH-DSA-SHA2-128f"),
    SLH_DSA_KEYGEN_TEST_ITEM(slh_dsa_sha2_192s_0, "SLH-DSA-SHA2-192s"),
    SLH_DSA_KEYGEN_TEST_ITEM(slh_dsa_sha2_192f_0, "SLH-DSA-SHA2-192f"),
    SLH_DSA_KEYGEN_TEST_ITEM(slh_dsa_sha2_256s_0, "SLH-DSA-SHA2-256s"),
    SLH_DSA_KEYGEN_TEST_ITEM(slh_dsa_sha2_256f_0, "SLH-DSA-SHA2-256f"),
    SLH_DSA_KEYGEN_TEST_ITEM(slh_dsa_shake_128s_0, "SLH-DSA-SHAKE-128s"),
    SLH_DSA_KEYGEN_TEST_ITEM(slh_dsa_shake_128f_0, "SLH-DSA-SHAKE-128f"),
    SLH_DSA_KEYGEN_TEST_ITEM(slh_dsa_shake_192s_0, "SLH-DSA-SHAKE-192s"),
    SLH_DSA_KEYGEN_TEST_ITEM(slh_dsa_shake_192f_0, "SLH-DSA-SHAKE-192f"),
    SLH_DSA_KEYGEN_TEST_ITEM(slh_dsa_shake_256s_0, "SLH-DSA-SHAKE-256s"),
    SLH_DSA_KEYGEN_TEST_ITEM(slh_dsa_shake_256f_0, "SLH-DSA-SHAKE-256f"),
};
